/*
************************************************************************
*
*   peak.h -
*
*   Copyright (c) 1995
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   All Rights Reserved
*
*   Date of last modification : 95/09/15
*   Pathname of SCCS file     : /export/home3/cb/garant-1.0/src/SCCS/s.peak.h
*   SCCS identification       : 1.3
*
************************************************************************
*/
/**************************************************************************/
/*                             peak.h                                     */
/*                                                                        */
/* measured peaks                                                         */
/**************************************************************************/

#ifndef _PEAK_H_
#define _PEAK_H_

/* externaly define classes */

class AssWriter;
class PeakWriter;
class Spectrum;
class ErPeak;
class Coherence;

/* base classes imported from: */

#include "element.h"
#include "wert.h"
#include "global.h"
#include "imp.h"

/* local data structures */
class Limit;
class Sublist;

/* exported classes */
class Peak;			/* describing measured peaks                 */
class Assignment;		/* assignment of one measured peak to many   */
				/* expected peaks                            */
class LiPeak;			/* list of peaks ordered after spectrum and  */
				/* external (EASY) peak number               */
class NdLiPeak;			/* list of peaks optimized to access peaks   */
				/* lying in defined n dim. cubes             */


/* constant definitions */
#define INDISTINGUISHABLE 0
#define AMBIGOUS          1
#define DISTINGUISHABLE   2

class Assignment : public Element {
   int maxAss;			/* size of ass array                         */
   ErPeak **ass;		/* list of possible assignments              */
   int nrAss;			/* number of possible assignments            */
public:
   Assignment() {nrAss = 0; maxAss = 0; ass = 0; t = ASS; }
   ~Assignment();
   void addAss(ErPeak *);	/* add assignment possibility to list        */
   void writeAss(AssWriter &,Spectrum *);
				/* Append assignment to (EASY) ass. list     */
   void removeAss(int);		/* remove assignments of one type (SUB/DEST) */
   void resetAss();		/* Clear all assignment possibilities        */
   ErPeak *start(int &);	/* iteration over all possible assignments   */
   ErPeak *next(int &);
};

class PeakInfo: public Element {/* Class containing all information of       */
				/* measured peaks except their position      */
   float p;			/* Scoring for the assignment of the peak    */
public:
   friend Peak;
   friend Matching;
   Assignment ass;		/* Assignment to expected peaks              */
   Coherence *userAss[DIMENSION];
				/* assignment provided by the user           */
   float prob;			/* Probability that the peak is no artefact  */
   int nr;			/* External (EASY) peak number               */
   Wert vol;			/* Volume of the peak                        */
   char comment[MAX_COMMENT];	/* Comment 				     */
   int color;			/* Peak color                                */
   char intflag;	        /* Used integration method                   */
   Spectrum *spec;		/* Spectrum the peak is belonging to         */
   int nrAss;			/* Number of expected peaks to which the peak*/
				/* is assigned to                            */
   ErPeak **asslink;		/* assignment of peak to expected peak       */
   int maxAsslinks;
   int nrAsslinks;
   int fixed;			/* TRUE ......... do not change     	     */
				/* FALSE ........ include in opt.  	     */
   void establishAsslink(ErPeak *);
				/* add ErPeak to asslink                     */
   void retractAsslink(int);	/* retract asslink with given index < nrAssl.*/
   int nrIntraDegen(ErPeak *);	/* number of degeneracies with only one      */
				/* coherence differing                       */
   int getAmbigiuety(ErPeak *);	/* determine wheter given peak can be        */
				/* distinguished from the allready assigned  */
				/* peaks                                     */
   PeakInfo(int, Wert, Spectrum *, float, int, char *,char);
				/* Initialize with nr,vol,spec,prob          */
				/* comment and integration method            */
   PeakInfo(int n, Spectrum *s) {nr = n; spec = s; }
				/* Initialization used for searching         */
   PeakInfo(Spectrum *s);	/* Initialization for new peaks              */
};

class Peak : public Element {	/* Class describing measured peaks           */
				/* including their unfolded position         */
public:
   PeakInfo *p;			/* information about the peak                */
   float w[DIMENSION];		/* Unfolded ppm position of the peak         */
   short int fold[DIMENSION];	/* number of times peak was folded           */
   Peak(PeakInfo *, float *);
				/* initialize with shift and PeakInfo        */
   Peak(PeakInfo *inf);
				/* initialize with PeakInfo                  */
   inline void resetAss() {p->nrAss=0; p->nrAsslinks=0; }
   virtual void print(int);	/* Write peak informatuion to log file       */
   void setP(float);		/* Set scoring of the peak                   */
   void addP(float);		/* 'Add' scoring s1 to the current score s:  */
				/*     s = max(s,s1)                         */
   float getP();		/* Return the scoring                        */
   float Peak::calcPSelect(Wert *,Wert *,int *);
				/* calc. selection probability according to  */
				/* assigned freq., expected freq.,           */
   void write(PeakWriter &, int,int &, int &, int &);
				/* Append peak to (EASY) peak list           */
				/* INPUT: type of fragment for the assignment*/
				/* OUTPUT: nr. of unassigned, degenerate and */
				/*        user assigned peaks                */
   void writeAssPos(AssWriter &, int,int &, int &);
				/* Append peak to (EASY) assignment list     */
				/* INPUT: type of fragment for the assignment*/
				/* OUTPUT: nr. of unassigned and degenerate  */
				/*        peaks                              */
};


class LiPeak : public Index {	/* List of peaks sorted after external (EASY)*/
				/* peaknumber and spectrum type              */
   void print(int);		/* List all measured peaks to the log file   */
public:
   LiPeak() : Index(1024) {}
   virtual int compare(const Element *p1, const Element *p2);
				/* compare according to external (EASY) peak */
				/* number and spectrum type                  */
   int load(Spectrum *,char *,int); 
				/* Load peaks for given spectrum             */
				/* INPUT: filename, spectrum type            */
 				/* flag 0 ... peaks to be assigned           */
				/* flag 1 ... reference peaks                */
   void unfold(Spectrum *s, int dim, int n, int m);
				/* unfold peaks of given spectrum in         */
				/* dimension dim from n to m times           */
   void resetUserAss(Spectrum *);
				/* Reset the all user assignments of s spec. */
   void writeMeasPeaks(Spectrum *, char *, char *, int, int);
				/* Write assigned peak list                  */
				/* INPUT: spectrum type, file name peak list */
				/*        file name proton list, type of ass.*/
				/*        to write (i.e. DESTFRAG or SUBFRAG)*/
				/*        flag whether to write only peaks   */
				/* 	  with p.p > 0.0                     */
   void writeAssPos(Spectrum *, char *, char *, int);
				/* Write out all assignment possibilities    */
				/* INPUT: spectrum type, file name ass list  */
				/*        file name proton list, type of ass.*/
				/*        to write (i.e. DESTFRAG or SUBFRAG)*/
   void resetAssPos(int);	/* Reset all assignment possibilities 'ass'  */
				/* of a given type (SUB. or DEST. FRAG)      */
   void mapAssPos(LiErPeak &);	/* generate ass. possibilities for the dest. */
				/* fragment from the ass. possibilities to   */
				/* the subfragment                           */
				/* INPUT: List of peaks to search for comp.  */
   void cmpAssPos();		/* compare ass. possibilities to the dest.   */
				/* fragment to ass. possibilities to         */
				/* the subfragment                           */
   Peak *find(int nr,Spectrum *s);
				/* Find peak with given number and spec type */
   inline Peak *start() {return((Peak *)Index::start() ); }
				/* Iterate over all peaks                    */
   inline Peak *next() {return((Peak *)Index::next() ); }
   inline Peak *start(Iterator &i) {return((Peak *)Index::start(i) ); }
   inline Peak *next(Iterator &i) {return((Peak *)Index::next(i) ); }
   void listPeaks();
};

class LiPeakName : public Index {
public:
   LiPeakName(int n) : Index(n) {}
   virtual int compare(const Element *p1, const Element *p2);
				/* compare according to name of assignment   */
                                /* and external (EASY) peak number & spectype*/
};

class LiPeakDim : public Index {
   int dim;
   float limit;
public:
   LiPeakDim() : Index(512) {}
   void setDim(int d) {dim=d;}
   Peak *startRange(float, float);
   Peak *nextRange();
   virtual int compare(const Element *p1, const Element *p2);
				/* compare according to shift in specified   */
				/* dimension                                 */
};

class NdLiPeak {		/* n dimensional partitioned list            */
   Peak **li;			/* sorted list of peaks                      */
   Limit *liLim;		/* splitting points (ppm) for partititioning */
   int max;			/* size of the list                          */
   int dim;			/* dimesnionality of the list                */
   int nr;			/* number of peaks in the list               */
   Sublist *stack;		/* list of unused splitting points, used in  */
				/* next()                                    */
   int nrStack;			/* nr of unused splitting points             */
   float *low;			/* lower limit of search region              */
   float *up;			/* upper limit of search region              */
   void recSort(const Sublist,int);
				/* n dim. sorting (partitioning) of Sublist  */
				/* INPUT: dim. used first for partitioning   */
   void quicksort(const Sublist,int);
				/* quicksort for sublist                     */
				/* INPUT: dim. used for sorting              */
   Peak *search(Sublist);	/* iterative search for peaks in search      */
				/* region [low,up]                           */
   Peak *searchUnusedClose(float *,Sublist);
				/* recursive search for an unused peak close */
				/* to given shift. Used by selectUnusedClose */
   Peak *searchRangeClose(float *,Sublist);
				/* recursive search for an unused peak close */
				/* to given shift, within search region.     */
				/* Used by selectUnusedClose                 */
public:
   float maxshift[DIMENSION];   /* maximal and minimal shift occuring in list*/
   float minshift[DIMENSION];
   NdLiPeak() {max =0;}
   void init(int,int);		/* initialize with max and dim               */
   void add(Peak *);		/* add to list (not sorted)                  */
   void sort();                 /* n dimensional partitioning (=sorting)     */
   Peak *start(float *, float *);
				/* iterate over all peaks lying in defined   */
				/* box                                       */
   Peak *next();
   Peak *selectUnusedClose(Wert *);
				/* select an unassigned peak close to given  */
				/* shift                                     */
   Peak *selectUnusedClose(Wert *,float *, float *);
				/* select an unassigned peak close to given  */
				/* shift within specified range              */
};

#endif
