
/*
************************************************************************
*
*   interpreter.C -
*
*   Copyright (c) 1995
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   All Rights Reserved
*
*   Date of last modification : 95/09/15
*   Pathname of SCCS file     : /export/home3/cb/garant-1.0/src/SCCS/s.interpreter.C
*   SCCS identification       : 1.5
*
************************************************************************
*/
/**************************************************************************/
/*                         interpreter.cc                                 */
/*                                                                        */
/* class to call routines:                                                */
/*    defines user variables                                              */
/*    gets values of the user variables                                   */
/*    deciphers command                                                   */
/**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "opt.h"
#include "global.h"
#include "bib.h"
#include "interpreter.h"
#include "specTypes.h"
#include "envir.h"
#include "log.h"
#include "score.h"
#include "str.h"
#include "cmdline.h"
#include "peak.h"
#include "specs.h"
#include "prep.h"
#include "match.h"

/* Environment on which to work */

Envir *envirCmd;

/* Whether libraries have been read */
static int fragLibDef = 0;
static int atNamLibDef= 0;
static int ssLibDef   = 0;
static int secShiftLib= 0;

/* forward declarations */
class PARspectrum;    
class PARsub;
class PARdest;


/* Definition of the commands */

class CmdWord {
public:
   CmdWord(char *word) {strcpy(ident,word); }
   char ident[MAXLINE];		/* command word                           */
   int check(char *);		/* check if match, return matched syntax  */
   void printcall();		/* print command word out                 */
};

class Parameter {
public:
   Parameter() {};
   char arg[MAXLINE];		/* file extension                         */
   virtual void printcall() {}	/* print command word out                 */
   virtual int start(char *) {return 0;}
				/* iterate over all valid substitutions   */
				/* returns TRUE if valid substitutions    */
   virtual int next() {return 0;}
   virtual void *get() {return 0;}
};

class PARInteger : public Parameter {
   int n;
public:
   virtual void printcall();	/* print substituted parameter           */
   virtual int start(char *);
   virtual int next() {return 0; }
   virtual void *get() {return &n; }
};

class PARFloat : public Parameter {
   float n;
public:
   virtual void printcall();	/* print substituted parameter           */
   virtual int start(char *);
   virtual int next() {return 0; }
   virtual void *get() {return &n; }
};

class PARsptype : public Parameter {
   char *name;
   int cur;
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next();
   virtual void *get() {return &cur; }
};

class PARspec : public Parameter {
   char *name;
   Spectrum *ret;
   int cur;
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next();
   virtual void *get() {return ret; }
};

class PARfilter : public Parameter {
   char *name;
   EquivDim *ret;
   int cur;
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next();
   virtual void *get() {return ret; }
};

class PARcrit : public Parameter {
   char *templ;
   Crit *ret;
   int typeList;
   Iterator cur;
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next();
   virtual void *get() {return ret; }
};

class PARoptpos : public Parameter {
   char *templ;
   OptPos *ret;
   Iterator cur;
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next();
   virtual void *get() {return ret; }
};

class PARread : public Parameter {
   char *name;
   char cur[MAXFILENAME];
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next() {return 0;}
   virtual void *get() {return cur; }
};

class PARname : public Parameter {
   char *name;
   char cur[MAXFILENAME];
public:
   friend PARspectrum;
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next() {return 0;}
   virtual void *get() {return cur; }
};

class PARspectrum : public PARname {
public:
   virtual int start(char *);
};

class PARfrag : public Parameter {
   FragNrRequest r;
public:
   friend PARsub;
   friend PARdest;
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next();
   virtual void *get() {return &r.nr; }
};

class PARsub : public PARfrag {
public:
   virtual int start(char *);
};

class PARdest : public PARfrag {
public:
   virtual int start(char *);
};

class PARwrite : public Parameter {
   char *name;
   char cur[MAXFILENAME];
public:
   virtual void printcall();	/* print command word out                 */
   virtual int start(char *);
   virtual int next() {return 0;}
   virtual void *get() {return cur; }
};

class Command {			/* base class for all commands            */
   CmdWord *word[MAXPARAMS];    /* command words defining the command     */
   int nrWords;			/* number of command words                */
   char *syntax;
   Parameter *newPar(char *, char *);
				/* return pointer to par. of given type   */
				/* INPUT: type, arg (i.e file extension)  */
public:
   Command(char *s);
   Parameter *par[MAXPARAMS];	/* parameters of the command              */
   int nrPar;			/* number of parameters                   */
   int valid;			/* are the current substituted pars valid?*/
   int check(Cmd &);		/* check if command matches syntax        */
   int cntMatches(Cmd &);       /* count the number of matches between    */
				/* the definition and the call            */
   void start(Cmd &);		/* iterate over all combination of valid  */
   void next(Cmd &);		/* parameters                             */
   virtual int call(Envir &) {return 0;}
				/* perform action on environment          */
   void printdef();		/* print the definition of the command    */
   virtual void printcall();	/* print the actual call                  */
   virtual void initlib();	/* make sure libraries are initialized    */
};

class CMDspec: public  Command
{
public:
   CMDspec() : Command("spectrum <spectype>") {}
   virtual int call(Envir &envir) {
      LiSpecType liSpecType;
      return liSpecType.use(*(int *)par[0]->get(),envir);
   }
};

class CMDspecName: public  Command
{
public:
   CMDspecName() : Command("spectrum <spectype> <specfile>") {}
   virtual int call(Envir &envir) {
      LiSpecType liSpecType;
      return liSpecType.use(*(int *)par[0]->get(),(char *)par[1]->get(),envir);
   }
};

class CMDdest: public Command
{
public:
   CMDdest() : Command("load sequence <read(.seq)>") {}
   virtual int call(Envir &envir) {
      return envir.dia.ss.fragAdd((char *)par[0]->get(),DESTFRAG);
   }
};
   
class CMDsub: public Command
{
public:
   CMDsub() : Command("load comparison sequence <read(.seq)>") {}
   virtual int call(Envir &envir) {
      return envir.dia.ss.fragAdd((char *)par[0]->get(),SUBFRAG);
   }
};

class CMDreadLibs: public Command
{
public: 
   CMDreadLibs(): Command("initlib") {}
   virtual int call(Envir &) {return 1;}
};   
 
class CMDfragLib: public Command
{
public:
   CMDfragLib() : Command("load fragment library <read(.lib)>") {}
   virtual int call(Envir &) {
      BibAccess bib;
      fragLibDef = bib.readlib((char *)par[0]->get());
      return fragLibDef;
   }
   virtual void initlib() {}
};
 
class CMDatNamLib: public Command
{
public:
   CMDatNamLib() : Command("load atom name library <read(.lib)>") {}
   virtual int call(Envir &) {
      BibAccess bib;
      atNamLibDef = bib.readEquivAt((char *)par[0]->get());
      return atNamLibDef;
   }
   virtual void initlib() {
      BibAccess bib;
      char fn[MAXLINE];
      if(!fragLibDef) {
         sprintf(fn,"%s%s",LIB_PATH,"pseudecepp.lib");
         fragLibDef = bib.readlib(fn);
      }
   }
};

class CMDsecShiftLib: public Command
{
public:
   CMDsecShiftLib() : Command("load secondary structure shifts <read(.lib)>") {}
   virtual int call(Envir &) {
      BibAccess bib;
      secShiftLib = bib.readSecShifts((char *)par[0]->get());
      return secShiftLib;
   }
   virtual void initlib() {
      BibAccess bib;
      char fn[MAXLINE];
      if(!fragLibDef) {
         sprintf(fn,"%s%s",LIB_PATH,"pseudecepp.lib");
         fragLibDef = bib.readlib(fn);
      }
   }
};

class CMDssLib: public Command
{
public:
   CMDssLib() : Command("load spin system library <read(.lib)>") {}
   virtual int call(Envir &) {
      BibAccess bib;
      ssLibDef = bib.readEquivFrags((char *)par[0]->get());
      return ssLibDef;
   }
   virtual void initlib() {
      BibAccess bib;
      char fn[MAXLINE];
      if(!fragLibDef) {
         sprintf(fn,"%s%s",LIB_PATH,"pseudecepp.lib");
         fragLibDef = bib.readlib(fn);
      }
   }
};
 
class CMDliAt: public Command
{
public:
   CMDliAt() : Command("list atoms <name>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.listAt((char *)par[0]->get());
      return 1;
   }
};
 
class CMDliPe: public Command
{
public:
   CMDliPe() : Command("list peaks") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.listPeaks();
      return 1;
   }
};
 
class CMDliExp: public Command
{
public:
   CMDliExp(): Command("list expected peaks <name>") {}
   virtual int call(Envir &envir) {
      envir.dia.listPeaks((char *)par[0]->get());
      return 1;
   }
};

class CMDsetPeakImp: public Command
{
public:
   CMDsetPeakImp(): Command("define peak importance <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.setPeakImp((char*)par[0]->get(),
                                 *((float *)par[1]->get()));
      return 1;
   }
};


class CMDmCoup: public Command
{
public:
   CMDmCoup(): Command("make couplings") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.jMake();
      return 1;
   }
};

class CMDmNoe: public Command
{
public:
   CMDmNoe(): Command("make noe") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.statNOEMake();
      return 1;
   }
};

class CMDmNoeRange: public Command
{
public:
   CMDmNoeRange(): Command("make noe <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.statNOEMake(*((int *)par[0]->get()),*((int *)par[1]->get()));
      return 1;
   }
};

class CMDmPeaks: public Command
{
public:
   CMDmPeaks(): Command("make peaks") {}
   virtual int call(Envir &envir) {
      envir.dia.peakMake(envir);
      return 1;
   }
};

class CMDmCo: public Command
{
public:
   CMDmCo(): Command("make coherences") {}
   virtual int call(Envir &envir) {
      envir.dia.coMake();
      return 1;
   } 
};

class CMDldFreqDest: public Command
{
public:
   CMDldFreqDest(): Command("load frequencies <read(.prot)>") {}
   virtual int call(Envir &envir) {
      envir.dia.loadFreqCo((char*)par[0]->get(),DESTFRAG);
      return 1;
   }
};

class CMDsetTolFreq: public Command
{
public:
   CMDsetTolFreq(): Command("define frequency tolerance <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.setFreqTol((char*)par[0]->get(),*((float *)par[1]->get()));
      return 1;
   }
};

class CMDdefSecShifts: public Command
{
public:
   CMDdefSecShifts(): Command("define secondary shifts <read(.txt)>") {}
   virtual int call(Envir &envir) {
      envir.dia.defSecShifts((char*)par[0]->get());
      return 1;
   }
};

class CMDpseudoAtCor: public Command
{
public:
   CMDpseudoAtCor(): Command("define pseudoatom correction <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.Atoms.setPseudCor((char*)par[0]->get(),
                                 *((float *)par[1]->get()));
      return 1;
   }
};

class CMDldPeaks: public Command
{
public:
   CMDldPeaks(): Command("load peaks <spectrum> <read(.peaks)>") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.load((Spectrum *)par[0]->get(),(char *)par[1]->get(),0);
      return 1;
   }
};

class CMDldRefPeaks: public Command
{
public:
   CMDldRefPeaks(): Command("load reference peaks <spectrum> <read(.peaks)>") {}
   virtual int call(Envir &envir) {
      envir.dia.refpeaks.load((Spectrum *)par[0]->get(),(char *)par[1]->get(),1);
      return 1;
   }
};

class CMDmatchPeaks: public Command
{
public:
   CMDmatchPeaks(): Command("match <spectrum> <float> <float>") {}
   virtual int call(Envir &envir) {
      envir.matching.match((Spectrum *)par[0]->get(),
			    envir.dia.peaks,envir.dia.refpeaks,
                            *((float *)par[1]->get()),
                            *((float *)par[2]->get()));
      return 1;
   }
};

class CMDunfoldPeaks: public Command
{
public:
   CMDunfoldPeaks(): Command(
      "unfold peaks <spectrum> <integer> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.unfold((Spectrum *)par[0]->get(),*((int *)par[1]->get()),
         *((int *)par[2]->get()),*((int *)par[3]->get()));
      return 1;
   }
};

class CMDldDestAss: public Command
{
public:
   CMDldDestAss(): Command(
      "load assignments <spectrum> <read(.peaks)> <read(.prot)> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.loadPeakAss(envir.dia.peaks,(Spectrum *)par[0]->get(),
         (char *)par[1]->get(), (char *)par[2]->get(),DESTFRAG,
         *((float *)par[3]->get()));
      return 1;
   }
};

class CMDldSubAss: public Command
{
public:
   CMDldSubAss(): Command(
      "load comparison assignments <spectrum> <read(.peaks)> <read(.prot)> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.loadPeakAss(envir.dia.peaks,(Spectrum *)par[0]->get(),
         (char *)par[1]->get(),(char *)par[2]->get(),SUBFRAG,
         *((float *)par[3]->get()));
      return 1;
   }
};

class CMDwDestCo: public Command
{
public:
   CMDwDestCo(): Command("write coherences <write(.prot)>") {}
   virtual int call(Envir &envir) {
      envir.dia.writeCo((char *)par[0]->get(),DESTFRAG);
      return 1;
   }
};

class CMDwDestPeaks: public Command
{
public:
   CMDwDestPeaks(): Command(
      "write expected peaks <spectrum> <write(.peaks)> <read(.prot)>") {}
   virtual int call(Envir &envir) {
      envir.dia.writeExpPeaks((Spectrum *)par[0]->get(),(char *)par[1]->get(),
         (char *)par[2]->get(), DESTFRAG,0);
      return 1;
   }
};

class CMDwExpPeaks: public Command
{
public:
   CMDwExpPeaks(): Command(
      "write expected peaks <spectrum> <write(.peaks)> <read(.prot)> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.writeExpPeaks((Spectrum *)par[0]->get(),(char *)par[1]->get(),
         (char *)par[2]->get(), DESTFRAG,*((int *)par[3]->get()));
      return 1;
   }
};

class CMDappExpPeaks: public Command
{
public:
   CMDappExpPeaks(): Command(
      "append expected peaks <spectrum> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.appendExpPeaks((Spectrum *)par[0]->get(),DESTFRAG,
                              *((int *)par[1]->get()),*((int *)par[2]->get()));
      return 1;
   }
};

class CMDmarkMeasPeaks: public Command
{
public:
   CMDmarkMeasPeaks(): Command(
      "mark possible peaks <spectrum> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.markPosPeaks((Spectrum *)par[0]->get(),*((int *)par[1]->get()),*
                             ((int *)par[2]->get()));
      return 1;
   }
};

class CMDassRes: public Command
{
public:
   CMDassRes(): Command("reset assignments") {}
   virtual int call(Envir &envir) {
      envir.dia.resetAss();
      return 1;
   }
};

class CMDcalcFreq: public Command
{
public:
   CMDcalcFreq(): Command("calculate frequencies") {}
   virtual int call(Envir &envir) {
      envir.dia.calcMeanFreq();
      /* envir.dia.cpMeanToAssFreq(); */
      return 1;
   }
};

class CMDresetFreq: public Command
{
public:
   CMDresetFreq(): Command("reset assigned frequencies") {}
   virtual int call(Envir &envir) {
      envir.dia.resetAssF();
      return 1;
   }
};

class CMDwLog: public Command
{
public:
   CMDwLog(): Command("write log <write(.log)>") {}
   virtual int call(Envir &) {
      Log log;
      log.open((char *)par[0]->get());
      return 1;
   }
   virtual void printcall() {}
};

class CMDaLog: public Command
{
public:
   CMDaLog(): Command("append log <write(.log)>") {}
   virtual int call(Envir &) {
      Log log;
      log.append((char *)par[0]->get());
      return 1;
   }
   virtual void printcall() {}
};

class CMDcloseLog: public Command
{
public:
   CMDcloseLog(): Command("close log") {}
   virtual int call(Envir &) {
      Log log;
      log.close();
      return 1;
   }
   virtual void printcall() {}
};

class CMDgenMap: public Command
{
public:
   CMDgenMap(): Command("determine equivalent peaks") {}
   virtual int call(Envir &envir) {
      envir.dia.genSsEquiv();
      return 1;
   }
};

class CMDrepFrag: public Command
{
public:
   CMDrepFrag(): Command("report fragments") {}
   virtual int call(Envir &envir) {
      envir.dia.reportFrags(envir.critFrag);
      return 1;
   }
};

class CMDassStat: public Command
{
public:
   CMDassStat(): Command("assignment statistics <spectrum>") {}
   virtual int call(Envir &envir) {
      envir.dia.assStatistic((Spectrum *)par[0]->get());
      return 1;
   }
};

class CMDrepGlobal: public Command
{
public:
   CMDrepGlobal(): Command("report global") {}
   virtual int call(Envir &envir) {
      envir.dia.scoreGlobal();
      return 1;
   }
};

class CMDrepCalc: public Command
{
public:
   CMDrepCalc(): Command("report calculate") {}
   virtual int call(Envir &envir) {
      envir.dia.reportCalc(envir.critCo,envir.critFrag,envir.critGlobal);
      return 1;
   }
};

class CMDaddFrCritII: public Command
{
public:
   CMDaddFrCritII(): 
      Command("criteria fragment intrainter <name> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragII(envir.dia.ss.Frags,(char *)par[0]->get(),
                         *((int *)par[1]->get()),*((int *)par[2]->get()));
      return 1;
   }
};

class CMDaddFrCritOvrlp: public Command
{
public:
   CMDaddFrCritOvrlp(): 
      Command("criteria fragment overlap <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragOverlap(envir.dia.ss.Frags,(char *)par[0]->get());
      return 1;
   }
};

class CMDaddFrCritMap: public Command
{
public:
   CMDaddFrCritMap(): 
      Command("criteria fragment mapping <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragMap(envir.dia.ss.Frags,(char *)par[0]->get());
      return 1;
   }
};

class CMDaddFrCritCor: public Command
{
public:
   CMDaddFrCritCor(): 
      Command("criteria fragment correct <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragCorrect(envir.dia.ss.Frags,(char *)par[0]->get(),
					"");
      return 1;
   }
};

class CMDaddFrCritCorTempl: public Command
{
public:
   CMDaddFrCritCorTempl(): 
      Command("criteria fragment correct <name> <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragCorrect(envir.dia.ss.Frags,(char *)par[0]->get(),
                                    (char *)par[1]->get());
      return 1;
   }
};

class CMDaddFrCritCorRes: public Command
{
public:
   CMDaddFrCritCorRes(): 
      Command("criteria fragment corres <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragCorRes(envir.dia.ss.Frags,(char *)par[0]->get(),
					"");
      return 1;
   }
};

class CMDaddFrCritCorResTempl: public Command
{
public:
   CMDaddFrCritCorResTempl(): 
      Command("criteria fragment corres <name> <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragCorRes(envir.dia.ss.Frags,(char *)par[0]->get(),
                                    (char *)par[1]->get());
      return 1;
   }
};

class CMDaddFrCritCorSs: public Command
{
public:
   CMDaddFrCritCorSs(): 
      Command("criteria fragment SScorrect <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragCorrSS(envir.dia.ss.Frags,(char *)par[0]->get());
      return 1;
   }
};

class CMDaddFrCritTempl: public Command
{
public:
   CMDaddFrCritTempl(): 
      Command("criteria fragment template <name> <name>") {}
   virtual int call(Envir &envir) {
      envir.critFrag.addFragTempl(envir.dia.ss.Frags,(char *)par[0]->get(),
                                (char *)par[1]->get());
      return 1;
   }
};

class CMDaddCoCritTempl: public Command
{
public:
   CMDaddCoCritTempl(): 
      Command("criteria coherence template <name> <name>") {}
   virtual int call(Envir &envir) {
      envir.critCo.addCoTempl(envir.dia.liCo,(char *)par[0]->get(),
                                (char *)par[1]->get());
      return 1;
   }
};

class CMDaddCoCritDegen: public Command
{
public:
   CMDaddCoCritDegen(): 
      Command("criteria coherence degenerated <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.critCo.addCoDegen(envir.dia.liCo,(char *)par[0]->get(),
                                *((float *)par[1]->get()));
      return 1;
   }
};

class CMDaddCoCritDegIntra: public Command
{
public:
   CMDaddCoCritDegIntra(): 
      Command("criteria coherence intradegen <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.critCo.addCoDegIntra(envir.dia.liCo,(char *)par[0]->get(),
                                *((float *)par[1]->get()));
      return 1;
   }
};

class CMDaddCoCritDegInter: public Command
{
public:
   CMDaddCoCritDegInter(): 
      Command("criteria coherence interdegen <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.critCo.addCoDegInter(envir.dia.liCo,(char *)par[0]->get(),
                                *((float *)par[1]->get()));
      return 1;
   }
};

class CMDaddCoCritMap: public Command
{
public:
   CMDaddCoCritMap(): 
      Command("criteria coherence mapping <name>") {}
   virtual int call(Envir &envir) {
      envir.critCo.addCoMap(envir.dia.liCo,(char *)par[0]->get());
      return 1;
   }
};

class CMDaddCoCritOvrlp: public Command
{
public:
   CMDaddCoCritOvrlp(): 
      Command("criteria coherence overlap <name>") {}
   virtual int call(Envir &envir) {
      envir.critCo.addCoOverlap(envir.dia.liCo,(char *)par[0]->get());
      return 1;
   }
};

class CMDsetCritThresh: public Command
{
public:
   CMDsetCritThresh(): 
      Command("criteria threshold <crit> <float> <float>") {}
   virtual int call(Envir &) {
      ((Crit *)par[0]->get())->setThresh(*((float *)par[1]->get()),
                                          *((float *)par[2]->get()) );
      return 1;
   }
};

class CMDcritComOn: public Command
{
public:
   CMDcritComOn(): 
      Command("criteria comment on <crit>") {}
   virtual int call(Envir &) {
      ((Crit *)par[0]->get())->setComment(1);
      return 1;
   }
};

class CMDcritComOf: public Command
{
public:
   CMDcritComOf(): 
      Command("criteria comment off <crit>") {}
   virtual int call(Envir &) {
      ((Crit *)par[0]->get())->setComment(0);
      return 1;
   }
};

class CMDrepCos: public Command
{
public:
   CMDrepCos(): Command("report coherences <name>") {}
   virtual int call(Envir &envir) {
      envir.dia.reportCos(envir.critCo,(char *)par[0]->get());
      return 1;
   }
};

class CMDlocalOpt: public Command
{
public:
   CMDlocalOpt(): Command("local optimization <integer>") {}
   virtual int call(Envir &envir) {
      Optimizer opt(envir.dia);
      opt.opt(*((int *)par[0]->get()));
      envir.dia.scoreGlobal();
      return 1;
   }
};

class CMDevolOpt: public Command
{
public:
   CMDevolOpt(): Command(
      "evolutionary optimization <integer> <float>") {}
   virtual int call(Envir &envir) {
      static OptGen opt(envir.dia);
      opt.evolOpt(*((int *)par[0]->get()), envir,*((float *)par[1]->get()));
      return 1;
   }
};

class CMDwMeasPeaks: public Command
{
public:
   CMDwMeasPeaks(): Command(
      "write peaks <spectrum> <write(.peaks)> <read(.prot)>") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.writeMeasPeaks((Spectrum *)par[0]->get(),
         (char *)par[1]->get(),
         (char *)par[2]->get(), DESTFRAG, 0);
      return 1;
   }
};

class CMDwRefPeaks: public Command
{
public:
   CMDwRefPeaks(): Command(
      "write reference peaks <spectrum> <write(.peaks)> <read(.prot)>") {}
   virtual int call(Envir &envir) {
      envir.dia.refpeaks.writeMeasPeaks((Spectrum *)par[0]->get(),
         (char *)par[1]->get(),
         (char *)par[2]->get(), DESTFRAG, 0);
      return 1;
   }
};

class CMDresUserAss: public Command
{
public:
   CMDresUserAss(): Command(
      "reset user assignments <spectrum>") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.resetUserAss((Spectrum *)par[0]->get());
      return 1;
   }
};

class CMDwAssMeasPeaks: public Command
{
public:
   CMDwAssMeasPeaks(): Command(
      "write assigned peaks <spectrum> <write(.peaks)> <read(.prot)>") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.writeMeasPeaks((Spectrum *)par[0]->get(),
         (char *)par[1]->get(),
         (char *)par[2]->get(), DESTFRAG, 1);
      return 1;
   }
};

class CMDwAssPos: public Command
{
public:
   CMDwAssPos(): Command(
   "write possible assignments <spectrum> <write(.assign)> <read(.prot)>")
       {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.writeAssPos((Spectrum *)par[0]->get(),
         (char *)par[1]->get(),
         (char *)par[2]->get(), DESTFRAG);
      return 1;
   }
};

class CMDresetAss: public Command
{
public:
   CMDresetAss(): Command(
      "reset possible assignments") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.resetAssPos(DESTFRAG);
      return 1;
   }
};

class CMDcmpAss: public Command
{
public:
   CMDcmpAss(): Command(
      "compare peak assignments") {}
   virtual int call(Envir &envir) {
      envir.dia.peaks.cmpAssPos();
      return 1;
   }
};

class CMDgenAss: public Command
{
public:
   CMDgenAss(): Command(
      "copyto possible assignments") {}
   virtual int call(Envir &envir) {
      envir.dia.addPeakAss();
      return 1;
   }
};

class CMDoptPosInit: public Command
{
public:
   CMDoptPosInit(): Command(
      "initialize optimization possibilities <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.initOptPos(*((int *)par[0]->get()));
      return 1;
   }
};

class CMDoptPosOvw: public Command
{
public:
   CMDoptPosOvw(): Command(
      "overwrite optimization possibility <float> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.optPos.update(1.0/ (*((float *)par[0]->get())),
				*((int *)par[1]->get()),
				*((int *)par[2]->get()),
				envir.dia);
      return 1;
   }
};

class CMDoptPosSel: public Command
{
public:
   CMDoptPosSel(): Command(
      "select optimization possibilities <integer> <float> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.avgScore =
         envir.dia.nrActive =
           envir.dia.optPos.select(
				*((float *)par[1]->get()),
				*((int *)par[0]->get()),
				*((int *)par[2]->get()),
				*((int *)par[3]->get()),
                                envir.dia.activePos);
      return 1;
   }
};

class CMDldCoordSub: public Command
{
public:
   CMDldCoordSub(): Command("load comparison coordinates <read(.cor)>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.loadCoord((char*)par[0]->get(),SUBFRAG);
      return 1;
   }
};

class CMDldCoordDest: public Command
{
public:
   CMDldCoordDest(): Command("load coordinates <read(.cor)>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.loadCoord((char*)par[0]->get(),DESTFRAG);
      return 1;
   }
};

class CMDdistNOEfloat: public Command
{
public:
   CMDdistNOEfloat(): Command(
      "make distance noe floating <integer> <float> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.distNOEMake(*((int *)par[0]->get()),
         *((float*)par[1]->get()),
         *((float*)par[2]->get()),1);
      return 1;
   }
};

class CMDdistNOEfix: public Command
{
public:
   CMDdistNOEfix(): Command(
      "make distance noe fixed <integer> <float> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.distNOEMake(*((int *)par[0]->get()),
         *((float*)par[1]->get()),
         *((float*)par[2]->get()),0);
      return 1;
   }
};

class CMDliDifFreq: public Command
{
public:
   CMDliDifFreq(): Command("compare frequencies <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.calcMeanFreq();
      envir.dia.liCo.listDif((char *)par[0]->get(),*((float*)par[1]->get()));
      return 1;
   }
};

class CMDseedRand: public Command
{
public:
   CMDseedRand(): Command("seed random numbers <integer>") {}
   virtual int call(Envir &) {
      long seed;
      seed = *((int *)par[0]->get());
      srand48(seed);
      srand((unsigned int)seed);
      return 1;
   }
};

class CMDchckInc: public Command
{
public:
   CMDchckInc(): Command(
      "check inconsistencies") {}
   virtual int call(Envir &envir) {
      envir.dia.checkIncon();
      return 1;
   }
};

class CMDchckMiss: public Command
{
public:
   CMDchckMiss(): Command(
      "list missing peaks <spectrum> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.reportMissPeaks((Spectrum *)par[0]->get(),DESTFRAG,
                              *((int *)par[1]->get()));
      return 1;
   }
};

class CMDfixAss: public Command
{
public:
   CMDfixAss(): Command("fix assignments <name>") {}
   virtual int call(Envir &envir) {
      envir.dia.fixAss((char *)par[0]->get(),1);
      return 1;
   }
};

class CMDreleaseAss: public Command
{
public:
   CMDreleaseAss(): Command("release assignments <name>") {}
   virtual int call(Envir &envir) {
      envir.dia.fixAss((char *)par[0]->get(),0);
      return 1;
   }
};

class CMDposAss: public Command
{
public:
   CMDposAss(): Command(
      "make possible assignments") {}
   virtual int call(Envir &envir) {
      envir.dia.addPossAss();
      return 1;
   }
};

class CMDsetAtomScore: public Command
{
public:
   CMDsetAtomScore(): Command("define atom probability <name> <float>") {}
   virtual int call(Envir &envir) {
      envir.dia.ss.Atoms.setScore((char*)par[0]->get(),
                                 *((float *)par[1]->get()));
      return 1;
   }
};

class CMDscorePop: public Command
{
public:
   CMDscorePop(): Command(
      "score population  <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.dia.optPos.scorePop(*((int *)par[0]->get()),
			      *((int *)par[1]->get()),
			      envir.dia);
      return 1;
   }
};

class CMDfilterDef1: public Command
{
public:
   CMDfilterDef1(): 
      Command("filter define <name> <spectype> <integer>") {}
   virtual int call(Envir &envir) {
      envir.filter.addEq((char *)par[0]->get(),
                       *((int *)par[1]->get()),
                       *((int *)par[2]->get()));
      return 1;
   }
};

class CMDfilterDef2: public Command
{
public:
   CMDfilterDef2(): 
      Command("filter define <name> <spectype> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.filter.addEq((char *)par[0]->get(),
                       *((int *)par[1]->get()),
                       *((int *)par[2]->get()),
                       *((int *)par[3]->get()));
      return 1;
   }
};

class CMDfilterDef3: public Command
{
public:
   CMDfilterDef3(): 
      Command("filter define <name> <spectype> <integer> <integer> <integer>") {}
   virtual int call(Envir &envir) {
      envir.filter.addEq((char *)par[0]->get(),
                       *((int *)par[1]->get()),
                       *((int *)par[2]->get()),
                       *((int *)par[3]->get()),
                       *((int *)par[4]->get()));
      return 1;
   }
};

class CMDfilterAdd1: public Command
{
public:
   CMDfilterAdd1(): 
      Command("filter add <filter> <spectype> <integer>") {}
   virtual int call(Envir &) {
      ((EquivDim *)par[0]->get())->addDim(
                          *((int *)par[1]->get()),
                          *((int *)par[2]->get())
                        );
      return 1;
   }
};

class CMDfilterAdd2: public Command
{
public:
   CMDfilterAdd2(): 
      Command("filter add <filter> <spectype> <integer> <integer>") {}
   virtual int call(Envir &) {
      ((EquivDim *)par[0]->get())->addDim(
                          *((int *)par[1]->get()),
                          *((int *)par[2]->get()),
                          *((int *)par[3]->get())
                        );
      return 1;
   }
};

class CMDfilterAdd3: public Command
{
public:
   CMDfilterAdd3(): 
      Command("filter add <filter> <spectype> <integer> <integer> <integer>") {}
   virtual int call(Envir &) {
      ((EquivDim *)par[0]->get())->addDim(
                          *((int *)par[1]->get()),
                          *((int *)par[2]->get()),
                          *((int *)par[3]->get()),
                          *((int *)par[4]->get())
                        );
      return 1;
   }
};

class CMDfilter: public Command
{
public:
   CMDfilter(): 
      Command("filter <filter> <spectrum> <spectrum> <write(.peaks)> <read(.prot)>") {}
   virtual int call(Envir &) {
      ((EquivDim *)par[0]->get())->filter(
         (Spectrum *)par[1]->get(),
         (Spectrum *)par[2]->get(),
         (char *)par[3]->get(),
         (char *)par[4]->get());
      return 1;
   }
};

class CMDfilter2: public Command
{
public:
   CMDfilter2(): 
      Command("filter <filter> <spectrum> <spectrum>") {}
   virtual int call(Envir &) {
      ((EquivDim *)par[0]->get())->filter(
          (Spectrum *)par[1]->get(),(Spectrum *)par[2]->get(),"","");
      return 1;
   }
};

Call::Call(Envir *e) {
   envirCmd = e;

   commands[0] = new CMDspec;
   commands[1] = new CMDdest;
   commands[2] = new CMDsub;
   commands[3] = new CMDgenMap;
   commands[4] = new CMDliAt;
   commands[5] = new CMDliPe;
   commands[6] = new CMDliExp;
   commands[7] = new CMDaddFrCritII;
   commands[8] = new CMDaddFrCritMap;
   commands[9] = new CMDmCoup;
   commands[10] = new CMDmNoe;
   commands[11] = new CMDmPeaks;
   commands[12] = new CMDmCo;
   commands[13] = new CMDposAss;
   commands[14] = new CMDldFreqDest;
   commands[15] = new CMDldPeaks;
   commands[16] = new CMDldDestAss;
   commands[17] = new CMDldSubAss;
   commands[18] = new CMDfixAss;
   commands[19] = new CMDwDestCo;
   commands[20] = new CMDwDestPeaks;
   commands[21] = new CMDchckMiss;
   commands[22] = new CMDspecName;
   commands[23] = new CMDchckInc;
   commands[24] = new CMDassRes;
   commands[25] = new CMDcalcFreq;
   commands[26] = new CMDcloseLog;
   commands[27] = new CMDwLog;
   commands[28] = new CMDwAssMeasPeaks;
   commands[29] = new CMDappExpPeaks;
   commands[30] = new CMDpseudoAtCor;
   commands[31] = new CMDfragLib;
   commands[32] = new CMDatNamLib;
   commands[33] = new CMDssLib;
   commands[34] = new CMDrepFrag;
   commands[35] = new CMDaddFrCritTempl;
   commands[36] = new CMDsetCritThresh;
   commands[37] = new CMDcritComOn;
   commands[38] = new CMDcritComOf;
   commands[39] = new CMDrepCos;
   commands[40] = new CMDaddCoCritTempl;
   commands[41] = new CMDaddCoCritDegen;
   commands[42] = new CMDaddCoCritDegIntra;
   commands[43] = new CMDaddCoCritDegInter;
   commands[44] = new CMDrepCalc;
   commands[45] = new CMDlocalOpt;
   commands[46] = new CMDwMeasPeaks;
   commands[47] = new CMDresetAss;
   commands[48] = new CMDmarkMeasPeaks;
   commands[49] = new CMDgenAss;
   commands[50] = new CMDoptPosInit;
   commands[51] = new CMDassStat;
   commands[52] = new CMDoptPosOvw;
   commands[53] = new CMDaddFrCritCorSs;
   commands[54] = new CMDsetAtomScore;
   commands[55] = new CMDevolOpt;
   commands[56] = new CMDresetFreq;
   commands[57] = new CMDaddFrCritCor;
   commands[58] = new CMDoptPosSel;
   commands[59] = new CMDresUserAss;
   commands[60] = new CMDdistNOEfloat;
   commands[61] = new CMDscorePop;
   commands[62] = new CMDrepGlobal;
   commands[63] = new CMDsetTolFreq;
   commands[64] = new CMDcmpAss;
   commands[65] = new CMDaddCoCritMap;
   commands[66] = new CMDaddCoCritOvrlp;
   commands[67] = new CMDaddFrCritOvrlp;
   commands[68] = new CMDliDifFreq;
   commands[69] = new CMDseedRand;
   commands[70] = new CMDaLog;
   commands[71] = new CMDsetPeakImp;
   commands[72] = new CMDwAssPos;
   commands[73] = new CMDunfoldPeaks;
   commands[74] = new CMDreleaseAss;
   commands[75] = new CMDwExpPeaks;
   commands[76] = new CMDldCoordSub;
   commands[77] = new CMDldCoordDest;
   commands[78] = new CMDdistNOEfix;
   commands[79] = new CMDmNoeRange;
   commands[80] = new CMDaddFrCritCorTempl;
   commands[81] = new CMDaddFrCritCorRes;
   commands[82] = new CMDaddFrCritCorResTempl;
   commands[83] = new CMDfilterDef1;
   commands[84] = new CMDfilterDef2;
   commands[85] = new CMDfilterDef3;
   commands[86] = new CMDfilterAdd1;
   commands[87] = new CMDfilterAdd2;
   commands[88] = new CMDfilterAdd3;
   commands[89] = new CMDfilter;
   commands[90] = new CMDfilter2;
   commands[91] = new CMDsecShiftLib;
   commands[92] = new CMDdefSecShifts;
   commands[93] = new CMDreadLibs;
   commands[94] = new CMDldRefPeaks;
   commands[95] = new CMDmatchPeaks;
   commands[96] = new CMDwRefPeaks;

   nrCmds = 97;
}

int Call::applyCmd(Envir &envir,Cmd &cmd)
{
   int i;
   int count = 0;
   int nrCmd;
   Log log;
   int res = 0;

   if(cmd.mode == 2) {
      log.w("... ERROR unknown command\n");
      helpSyntax(cmd);
      cmd.mode = 0;
      res=1;
   } else {
      for(i=0;i<nrCmds;i++) if(commands[i]->check(cmd)) {
         count++;
         nrCmd=i;
      }
      if(count == 0) {
         cmd.mode = 2;
         res=1;
      } else if(count > 1) {
         cmd.mode = 0;
         log.w("... ERROR multiple possible commands:\n");
         for(i=0;i<nrCmds;i++)
            if(commands[i]->check(cmd)) commands[i]->printdef();
         log.w("\n");
         res=1;
      } else if(count ==1) {
         cmd.mode = 0;
         commands[nrCmd]->start(cmd);
         while(commands[nrCmd]->valid) {
            commands[nrCmd]->printcall();
            commands[nrCmd]->initlib();
            if(commands[nrCmd]->call(envir)) {
            } else {
               log.w(" ---> aborted\n");
            }
            commands[nrCmd]->next(cmd);
         }
      }
   }
   return res;
}

void Call::helpSyntax(Cmd &cmd)
{
   Log log;
   int i;
   int written = 0;
   int maxMatch;

   for(maxMatch=cmd.nparam+1; maxMatch>0; maxMatch--) {
      for(i=0;i<nrCmds;i++) {
         if(commands[i]->cntMatches(cmd) == maxMatch) {
            if(!written) {
               log.w("... possible commands are:\n");
               written = 1;
            }
            commands[i]->printdef();
         }
      }
   }
}

int Call::setVars(UserVar &vars)
{
   int i,j;
   int nr;			/* Number of the variable                 */
   LiSpecType liSpecType;

   nr = 0;
   for(i=0;i<liSpecType.nrExp;i++) {
      sprintf(vars.var[nr++],"%s_noise",liSpecType.l[i].name);
      sprintf(vars.var[nr++],"%s_imp",liSpecType.l[i].name);
      for(j=0;j<liSpecType.l[i].dim;j++)
         sprintf(vars.var[nr++],"%s_w%d",liSpecType.l[i].name,j+1);
      for(j=0;j<liSpecType.l[i].dim;j++)
         sprintf(vars.var[nr++],"%s_acc_w%d",liSpecType.l[i].name,j+1);
      for(j=0;j<liSpecType.l[i].dim;j++)
         sprintf(vars.var[nr++],"%s_matchf_w%d",liSpecType.l[i].name,j+1);
   }
   sprintf(vars.var[nr++],"pick_acc");
   sprintf(vars.var[nr++],"select");
   sprintf(vars.var[nr++],"score");
   sprintf(vars.var[nr++],"avgscore");
   sprintf(vars.var[nr++],"mutations");
   sprintf(vars.var[nr++],"random");
//   printf("Trying to initialize %d variables\n",nr);
   return(vars.init(nr));
}

static float pickAcc = 3.0;

void Call::setValues(UserVar &vars, Envir &envir)
{
   int i,j;
   int nr = 0;			/* Number of the variable                 */
   LiSpecType liSpecType;

   for(i=0;i<liSpecType.nrExp;i++) {
      if(liSpecType.l[i].used) {
         sprintf(vars.vardef[nr++],"%g",liSpecType.l[i].noise);
         sprintf(vars.vardef[nr++],"%g",liSpecType.l[i].importance);
         for(j=0;j<liSpecType.l[i].dim;j++)
            sprintf(vars.vardef[nr++],"%d",liSpecType.l[i].permut[j]+1);
         for(j=0;j<liSpecType.l[i].dim;j++)
            sprintf(vars.vardef[nr++],"%g",liSpecType.l[i].accPick[j]);
         for(j=0;j<liSpecType.l[i].dim;j++)
            sprintf(vars.vardef[nr++],"%g",liSpecType.l[i].matchFac[j]);
      } else {
         sprintf(vars.vardef[nr++],"NULL");
         sprintf(vars.vardef[nr++],"NULL");
         for(j=0;j<liSpecType.l[i].dim;j++)
            sprintf(vars.vardef[nr++],"NULL");
         for(j=0;j<liSpecType.l[i].dim;j++)
            sprintf(vars.vardef[nr++],"NULL");
         for(j=0;j<liSpecType.l[i].dim;j++)
            sprintf(vars.vardef[nr++],"NULL");
      }
   }
   sprintf(vars.vardef[nr++],"%g",pickAcc);
   sprintf(vars.vardef[nr++],"%s",envir.optPosSel);
   sprintf(vars.vardef[nr++],"%g",envir.dia.score);
   sprintf(vars.vardef[nr++],"%g",envir.dia.avgScore);
   sprintf(vars.vardef[nr++],"%g",envir.dia.ratioMut);
   sprintf(vars.vardef[nr++],"%g",drand48());
//   printf("Values set for %d variables\n",nr);
}


int Call::getValues(UserVar &vars, Envir &)
{
   int i,j;
   int nr = 0;			/* Number of hte variable                 */
   int error = 0;		/* A variable can not be converted        */
   LiSpecType liSpecType;

   for(i=0;i<liSpecType.nrExp;i++) {
      if(liSpecType.l[i].used) {
         if(sscanf(vars.vardef[nr++],"%g",&liSpecType.l[i].noise) != 1) {
            error = 1;
            sprintf(vars.vardef[nr-1],"%g",liSpecType.l[i].noise);
         }
         if(sscanf(vars.vardef[nr++],"%g",&liSpecType.l[i].importance) != 1){
            error = 1;
            sprintf(vars.vardef[nr-1],"%g",liSpecType.l[i].importance);
         }
         for(j=0;j<liSpecType.l[i].dim;j++) {
            if(sscanf(vars.vardef[nr++],"%d",&liSpecType.l[i].permut[j]) != 1 ||
               liSpecType.l[i].permut[j]<1 || 
               liSpecType.l[i].permut[j]>liSpecType.l[i].dim){
               error = 1;
               sprintf(vars.vardef[nr-1],"%d",liSpecType.l[i].permut[j]+1);
            } else {
               liSpecType.l[i].permut[j]--;
            }
         }
         for(j=0;j<liSpecType.l[i].dim;j++)
            if(sscanf(vars.vardef[nr++],"%g",&liSpecType.l[i].accPick[j])!=1){
               error = 1;
               sprintf(vars.vardef[nr-1],"%g",liSpecType.l[i].accPick[j]);
            }
         for(j=0;j<liSpecType.l[i].dim;j++)
            if(sscanf(vars.vardef[nr++],"%g",&liSpecType.l[i].matchFac[j])!=1){
               error = 1;
               sprintf(vars.vardef[nr-1],"%g",liSpecType.l[i].matchFac[j]);
            }
      } else {
         nr += 2 + 3 * liSpecType.l[i].dim;
      }
   }
   if(sscanf(vars.vardef[nr++],"%g",&pickAcc)!=1){
       error = 1;
       sprintf(vars.vardef[nr-1],"%g",pickAcc);
   } else {
      for(i=0;i<liSpecType.nrExp;i++)
         if(liSpecType.l[i].used)
            for(j=0;j<liSpecType.l[i].dim;j++)
              liSpecType.l[i].accSpec[j] = liSpecType.l[i].accPick[j] * pickAcc;
   }
   nr += 5;

//   printf("Values read for %d variables\n",nr);

   if(error) vars.error = 1;
   return(!error);
}

void Call::callmacro(char *line)
{
   MacroCall call;

   strcpy(call.cmd,line);
}

Command::Command(char *s)
{
   char line[MAXLINE];
   char token[MAXLINE];
   char typePar[MAXLINE];

   syntax = s;
   strcpy(line,syntax);
   nrPar = 0; nrWords = 0;
   strip(line," ");
   while(line[0] != '\0') {
      getToken(line,token," ");
      if(token[0] == '<') {
         getToken(token,typePar," <>()");
         strip(token," <>()");
         par[nrPar] = newPar(typePar,token);
         nrPar++;
      } else {
         word[nrWords] = new CmdWord(token);
         nrWords++;
      }
   }
}

Parameter *Command::newPar(char* type,char * arg)
{
   Parameter *res=0;

   if(strcmp(type,"integer") == 0)         res = new PARInteger();
   else if(strcmp(type,"float") == 0)      res = new PARFloat();
   else if(strcmp(type,"spectype") == 0)   res = new PARsptype();
   else if(strcmp(type,"spectrum") == 0)   res = new PARspec();
   else if(strcmp(type,"fragment") == 0)   res = new PARfrag();
   else if(strcmp(type,"destfrag") == 0)   res = new PARdest();
   else if(strcmp(type,"subfrag") == 0)    res = new PARsub();
   else if(strcmp(type,"read") == 0)       res = new PARread();
   else if(strcmp(type,"write") == 0)      res = new PARwrite();
   else if(strcmp(type,"name") == 0)       res = new PARname();
   else if(strcmp(type,"specfile") == 0)   res = new PARspectrum();
   else if(strcmp(type,"crit") == 0)       res = new PARcrit();
   else if(strcmp(type,"optpos") == 0)     res = new PARoptpos();
   else if(strcmp(type,"filter") == 0)     res = new PARfilter();

   strcpy(res->arg,arg);

   return res;
}

int Command::check(Cmd &cmd)
{
   int i=-1;
   int iWord = 0;
   int iPar = 0;
   int res;

   if(cmd.nparam + 1 != nrPar + nrWords) res = 0;
   else res = 1;

   while(res && i<cmd.nparam && (iPar < nrPar || iWord < nrWords)) {
      if(iWord < nrWords) {
         if(i== -1) res = word[iWord]->check(cmd.cmd);
         else res &= word[iWord]->check(cmd.param[i]);
         iWord++; i++;
      } else if(iPar < nrPar) {
         res &= par[iPar]->start(cmd.param[i]);
         iPar++; i++;
      } else res = 0;
   }
   return res;
}

int Command::cntMatches(Cmd &cmd)
{
   int i=-1;
   int iWord = 0;
   int iPar = 0;
   int nrMatch = 0;
   int res;

   iWord = 0; iPar = 0;
   while(iWord < nrWords) {
      res = 0;
      for(i=-1; i<cmd.nparam && !res; i++) {
         if(iWord < nrWords) {
            if(i== -1) res = word[iWord]->check(cmd.cmd);
            else res = word[iWord]->check(cmd.param[i]);
         }
      }
      if(res) nrMatch++;
      iWord++;
   }
   return nrMatch;
}

void Command::start(Cmd &cmd)
{
   int iPar;

   valid = 1;
   for(iPar=0; iPar<nrPar; iPar++) {
      valid &= par[iPar]->start(cmd.param[iPar + nrWords - 1]);
   }
}

void Command::next(Cmd &cmd)
{
   int iPar = 0;
   valid = 0;

   while(!valid && iPar<nrPar) {
      valid = par[iPar]->next();
      if(!valid) {
         par[iPar]->start(cmd.param[iPar + nrWords - 1]);
         iPar++;
      }
   }
}

void Command::printdef()
{
   Log log;
   char str[MAXLINE];

   sprintf(str,"%s\n",syntax);
   log.w(str);
}

void Command::printcall()
{
   int iWord = 0;
   int iPar = 0;
   Log log;

   log.w("\n\nCOMMAND:");

   while(iPar < nrPar || iWord < nrWords) {
      if(iWord < nrWords) {
         log.w(" ");
         word[iWord]->printcall();
         iWord++;
      } else {
         log.w(" ");
         par[iPar]->printcall();
         iPar++;
      }
   }
}

void Command::initlib()
{
   BibAccess bib;
   char fn[MAXLINE];

   if(!fragLibDef) {
      sprintf(fn,"%s%s",LIB_PATH,"pseudecepp.lib");
      fragLibDef = bib.readlib(fn);
   }
   if(!atNamLibDef) {
      sprintf(fn,"%s%s",LIB_PATH,"atomname.lib");
      atNamLibDef = bib.readEquivAt(fn);
   }
   if(!secShiftLib) {
      sprintf(fn,"%s%s",LIB_PATH,"shift.lib");
      secShiftLib = bib.readSecShifts(fn);
   }
   if(!ssLibDef) {
      sprintf(fn,"%s%s",LIB_PATH,"spinsystem.lib");
      ssLibDef = bib.readEquivFrags(fn);
   }
}

int CmdWord::check(char *w)
{
   int len;
   int res = 0;

   len = strlen(w);
   if(len <= strlen(ident)) res = (strncmp(w,ident,len) == 0);

   return res;
}

void CmdWord::printcall()
{
   Log log;

   log.w(ident);
}

void PARInteger::printcall()
{
   Log log;
   char str[MAXLINE];

   sprintf(str," %d",n);
   log.w(str);
}

int PARInteger::start(char *str)
{
   return (sscanf(str,"%d",&n) == 1);
}

void PARFloat::printcall()
{
   Log log;
   char str[MAXLINE];

   sprintf(str," %g",n);
   log.w(str);
}

int PARFloat::start(char *str)
{
   return (sscanf(str,"%f",&n) == 1);
}

void PARsptype::printcall()
{
   LiSpecType li;
   Log log;
   char n[MAXLINE];

   li.name(n,cur);
   log.w(n);
}

int PARsptype::start(char *n)
{
   LiSpecType li;

   name = n;
   return li.start(name,cur);
}

int PARsptype::next()
{
   LiSpecType li;

   return li.next(name,cur);
}

void PARfrag::printcall()
{
   FragNameRequest nameR;
   Log log;

   nameR.nr = r.nr;
   nameR.start();
   if(strchr(nameR.name,' ') == 0) {
      log.w(nameR.name);
   } else {
      log.w("'");
      log.w(nameR.name);
      log.w("'");
   }
}

int PARfrag::start(char *n)
{
   strcpy(r.name,n);
   r.type = DESTSUB;
   r.start();
   return !r.eof;
}

int PARdest::start(char *n)
{
   strcpy(r.name,n);
   r.type = DESTFRAG;
   r.start();
   return !r.eof;
}

int PARsub::start(char *n)
{
   strcpy(r.name,n);
   r.type = SUBFRAG;
   r.start();
   return !r.eof;
}

int PARfrag::next()
{
   r.read();

   return !r.eof;
}

void PARread::printcall()
{
   Log log;
   char str[MAXLINE];

   if(strchr(name,'.')==0)
      sprintf(str,"%s%s",name,arg);
   else
      sprintf(str,"%s",name);
   log.w(str);
}

int PARread::start(char *n)
{
   int i,res;

   name = n;
   res=0;
   for(i=strlen(name)-1;i>=0 && name[i]!='/' && name[i]!=':' && !res;i--)
      if(name[i]=='.') res=1;
   if(!res)
      sprintf(cur,"%s%s",name,arg);
   else
      sprintf(cur,"%s",name);
   if(name[0] != 0) return 1;
   else return 0;
}

void PARname::printcall()
{
   Log log;
   char str[MAXLINE];

   sprintf(str,"%s",name);
   log.w(str);
}

int PARname::start(char *n)
{
   name = n;
   sprintf(cur,"%s",name);
   if(name[0] != 0) return 1;
   else return 0;
}

int PARspectrum::start(char *n)
{
   Spec_Param_Data *temp;

   name = n;
   sprintf(cur,"%s",name);
   temp = Open_Spec_RO(name);
   if(temp) Close_Spec(temp);

   if(temp != 0) return 1;
   else return 0;
   return 1;
}

void PARwrite::printcall()
{
   Log log;
   char str[MAXLINE];

   if(strchr(name,'.')==0)
      sprintf(str,"%s%s",name,arg);
   else
      sprintf(str,"%s",name);
   log.w(str);
}

int PARwrite::start(char *n)
{
   int i,res;

   name = n;
   res=0;
   for(i=strlen(name)-1;i>=0 && name[i]!='/' && name[i]!=':' && !res;i--)
      if(name[i]=='.') res=1;
   if(!res)
      sprintf(cur,"%s%s",name,arg);
   else
      sprintf(cur,"%s",name);
   if(name[0] != 0) return 1;
   else return 0;
}

void PARspec::printcall()
{
   Log log;
   char n[MAXLINE];

   ret->name(n);
   log.w(n);
}

int PARspec::start(char *n)
{
   name = n;
   ret=usedSpec.start(name,cur);
   if(ret != 0) return 1;
   else return 0;
}

int PARspec::next()
{
   ret=usedSpec.next(name,cur);
   if(ret != 0) return 1;
   else return 0;
}

void PARfilter::printcall()
{
   Log log;
   char n[MAXLINE];

   ret->getname(n);
   log.w(n);
}

int PARfilter::start(char *n)
{
   name = n;
   ret=envirCmd->filter.start(name,cur);
   if(ret != 0) return 1;
   else return 0;
}

int PARfilter::next()
{
   ret=envirCmd->filter.next(name,cur);
   if(ret != 0) return 1;
   else return 0;
}

void PARcrit::printcall()
{
   Log log;
   char n[MAXLINE];

   ret->iname(n);
   log.w(n);
}

int PARcrit::start(char *n)
{
   templ = n;
   typeList = -1;
   return next();
}

int PARcrit::next()
{
   ret = 0;
   if(typeList == -1) {
      typeList = 0;
      ret=(Crit *)envirCmd->critGlobal.startPars(cur,templ);
   }
   while(ret == 0 && typeList < 3)
   {
      if(typeList==0) ret=(Crit *)envirCmd->critGlobal.nextPars(cur,templ);
      if(typeList==1) ret=(Crit *)envirCmd->critFrag.nextPars(cur,templ);
      if(typeList==2) ret=(Crit *)envirCmd->critCo.nextPars(cur,templ);
      if(ret == 0) {
         typeList++;
         if(typeList==1) ret=(Crit *)envirCmd->critFrag.startPars(cur,templ);
         if(typeList==2) ret=(Crit *)envirCmd->critCo.startPars(cur,templ);
      }
   }

   if(ret != 0) return 1;
   else return 0;
}

void PARoptpos::printcall()
{
   Log log;
   char n[MAXLINE];

   ret->iname(n);
   log.w(n);
}

int PARoptpos::start(char *n)
{
   templ = n;
   ret=(OptPos *)envirCmd->dia.optPos.startPars(cur,templ);

   if(ret != 0) return 1;
   else return 0;
}

int PARoptpos::next()
{
   ret=(OptPos *)envirCmd->dia.optPos.nextPars(cur,templ);

   if(ret != 0) return 1;
   else return 0;
}

