/*
************************************************************************
*
*   element.h -
*
*   Copyright (c) 1995
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   All Rights Reserved
*
*   Date of last modification : 95/09/15
*   Pathname of SCCS file     : /export/home3/cb/garant-1.0/src/SCCS/s.element.h
*   SCCS identification       : 1.2
*
************************************************************************
*/
/**************************************************************************/
/*                            element.h                                   */
/*                                                                        */
/* general class to define general elements and ordered lists of elements */
/**************************************************************************/

#ifndef _ELEMENT_H_
#define _ELEMENT_H_

class Element;
class Iterator;
class Index;

/* classes that have special access to the Index and Iterator class    */
class LiRel;			/* list of relations betwen atoms     */
class LiErPeak;			/* list of expected peaks             */
class LiCo;			/* list of coherences                 */
class OrdLiAtom;		/* atoms sorted by their names        */
class LiOpt;			/* list of optimizers                 */
class LiPeakDim;		/* list of measured peaks, by position*/
class OptPosSelection;
class LiEquiv;

enum Type {NONE, ATOM, RELATION, COUPLING, NOE, BOND, EXPPEAK, PEAK, COHERENCE, SQCO, SPEC, ASS};

class Element {			/* base class used to define elements */
					/* of a list                          */
public:
   friend Index;
   Element();
   virtual ~Element() {}
   virtual void print(int);
   virtual int match(char *) {return 1;}/* does element match the template ?  */
   virtual void iname(char *) {}	/* return identification name of elem.*/
   Type t;				/* what sort of element ??            */
};

class Iterator {			/* class to use for sequential access */
					/* to all elements of a list          */
public:
   int cur;				/* number of the current element      */
   friend Index;
   friend LiRel;
   friend LiErPeak;
};

class Index {			/* ordered list of elements           */
   Element **l;				/* list of the elements               */
   int n;				/* number of elements                 */
   int max;				/* maximal number of elements         */
   int cur;				/* current element for non recursive  */
					/* iterations                         */
   void quicksort(int, int);		/* recursive part of quicksort        */
   void revquicksort(int, int);		/* recursive part of reverse quicksort*/
public:
   friend LiRel;
   friend LiOpt;
   friend LiErPeak;
   friend LiCo;
   friend OrdLiAtom;
   friend LiPeakDim;
   friend OptPosSelection;
   friend LiEquiv;
   friend LiEquivDim;
   Index(int);
   virtual ~Index();
   virtual int compare(const Element *,const Element *);
					/* virtual function used for ordering */
					/* result < 0 ... first < second par. */
					/* result = 0 ... first = second par. */
					/* result > 0 ... first > second par. */
   Element *start();			/* global iteration, first el.        */
   Element *start(Iterator &);		/* local iteration, first el.         */
					/* next gives 2,3,... el.             */
   void init(Iterator &);		/* local iteration, no element        */
					/* next gives 1,2,3,... el.           */
   inline Element *next()		/* global iteration, ret. next element*/
      {
         if(cur<n) {cur++; return(l[cur-1]); }
         else return(0);
      }
   inline Element *next(Iterator &i)	/* local itetration, ret. next element*/
      {
         if(i.cur<n) {i.cur++; return(l[i.cur-1]); }
         else return(0);
      }
   inline Element *nextCyc(Iterator &i) /* next element, if at end of list    */
					/* then continue from the beginning   */
      {
         if(i.cur>=n) i.cur -= n;
         i.cur++;
         return(l[i.cur-1]);
      }
   inline Element *prevCyc(Iterator &i) /* previous element, if at beginning  */
					/* of the list continue at the end    */
      {
         i.cur--;
         if(i.cur < 0) i.cur = n-1;
         if(i.cur == 0) return l[n-1];
         else return l[i.cur-1];
      }
   Element *getCyc(Iterator &,int);
					/* element, with given offset         */
   Element *find(const Element &);	/* find element in list that has the  */
					/* same ordering as the provided      */
					/* element                            */
   Element *startPars(Iterator &, char *);
					/* parse for a elements matching      */
					/* a given template                   */
   Element *nextPars(Iterator &, char *);
   inline Element *get(Iterator i)
					/* return specified element           */
      {
         if(i.cur-1 >= 0 && i.cur-1 < n) return l[i.cur-1];
         else return 0;
      }
   int insert(Element *);		/* insert element into list           */
					/* 1 .. ok; 0 .. no space;            */
					/* -1 .. doublicate                   */
   int add(Element *);			/* add element to list, no ordering   */
					/* 1 .. ok; 0 .. no space;            */
   void remove(); 			/* frees index without deleting       */
					/* elements                           */
   void del(); 				/* frees index and deletes elements   */
   void print(int);			/* print the list of elements         */
   virtual void list(int,char *);	/* list element matching template     */
   inline int getn() {return(n); }	/* return the number of elements      */
   void sort(int dir = 1);		/* sort entries, calls quicksort      */
					/* INPUT: direction for sorting       */
   Element *select(float);		/* slect optimizer                    */
                                	/* INPUT: ratio of nr of times best   */
					/* opt. should be selected compared to*/
					/* the number of times the worst opt. */
					/* should be selected                 */
};

class Cycle {				/* cyclic list: fifo                  */
   Element **l;				/* contains the elements              */
   int v,b;				/* borders [v,b)                      */
   int max;				/* size of the list                   */
   int cur;				/* Laufvariable                       */
public:
   Cycle(int);				/* init cycle of given size           */
   ~Cycle();
   void reset();			/* reset cycle i.e. v=b=0             */
   void add(Element *);			/* add element to the end             */
   void addfront(Element *);		/* add element to the front           */
   void addrandom(Element *);		/* add at random position             */
   void addrandom(Element *,int);	/* add at random position in the n    */
					/* last elements, n==1 --> last       */
   void addrandomfront(Element *,int);	/* add at random position in the n    */
					/* first elements                     */
   Element *get();			/* get first element                  */
   Element *getrandom();		/* get 'random' element               */
   int getn() { return b-v; }
   inline Element *start()		/* global iteration, first el.        */
       {
          cur = v;
          return(next());
       }
   inline Element *next()		/* global iteration, ret. next element*/
      {
         if(cur<b)
           {cur++; if(cur-1<max) return(l[cur-1]); else return(l[cur-1-max]);}
         else return(0);
      }
};
   

#endif
