/*
************************************************************************
*
*   coord_conversions.c -
*
*   Copyright (c) 1995
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   All Rights Reserved
*
*   Date of last modification : 95/09/15
*   Pathname of SCCS file     : /export/home3/cb/garant-1.0/src/SCCS/s.coord_conversions.c
*   SCCS identification       : 1.2
*
************************************************************************
*/
/******************************************************************************
 * routines to convert spectral coordinates                                   *
 *                                                                            *
 * Sp_P ........ Find coord in ppm given spectral coord                       *
 * P_Sp ........ Convert coord from ppm into points                           *
 * Sp_Hz ....... Convert spectral coordinate to Hz in given dimens.           *
 * Fold ........ folds peak. converts frequency into folding value and        *
 *               a folded frequency                                           *
 * UnFold ...... calculates the unfolded frequency from the folding           *
 *               value and the folded frequency                               *
 *                                                                            *
 * All routines take as input parameter a pointer to the Spec_Param_Data      *
 * data structure                                                             *
 *                                                                            *
 * programed version 0.1 ............................... 24.8.92 C.Bartels    *
 *                                                                            *
 ******************************************************************************/

#include <stdio.h>
#include <math.h>
#include "constants.h"
#include "data_structures.h"

void
Sp_P(spec,point,ppm)

Spec_Param_Data *spec;		/* pointer to spectral data                  */
float *point;			/* coordinate in spectral points [0,size)    */
float *ppm;			/* coordinat in ppm [max_ppm - sw, max_ppm)  */

{
   int i;
   float height;
   float max,sw;

   for(i=0;i<spec->dim;i++) {
      height = (float)spec->size[i];
      max = spec->max_ppm[i];
      sw = spec->sw[i];
      ppm[i] = max - (float)point[i] * sw / height;
   }
}


void
P_Sp(spec,ppm,point)

Spec_Param_Data *spec;		/* pointer to spectral data                  */
float *point;			/* coordinate in spectral points [0,size)    */
float *ppm;			/* coordinate in ppm (max_ppm - sw, max_ppm] */

{
   int i;
   float height;
   float max,sw;

   for(i=0;i<spec->dim;i++) {
      height = (float)spec->size[i];
      max = spec->max_ppm[i];
      sw = spec->sw[i];
      point[i] = (max - ppm[i]) * height / sw;
   }
}


/************************************************************************
* Converts frequency into folding value and a folded frequency          *
************************************************************************/

void
Fold(spec,freq,fold)

Spec_Param_Data *spec;	      /* pointer to spectral data              */
float freq[DIMENSION];        /* frequency                             */
short int fold[DIMENSION];    /* number of times frequency was folded  */

{
   int i;
   int nr;
   float low,up,w;

   for(i=0;i<spec->dim;i++) {
      up = spec->max_ppm[i]; low = spec->max_ppm[i]-spec->sw[i];
      if(spec->folding[i] == TPPI) {
         fold[i] = (short int)floor((freq[i]-low) / spec->sw[i]);
         freq[i] -= fold[i] * spec->sw[i];
         if((fold[i]/2)*2 != fold[i]) freq[i] = up + low - freq[i];
      }
      if(spec->folding[i] == RSH) {
         fold[i] = (short int)floor((freq[i]-low) / spec->sw[i]);
         freq[i] -= fold[i] * spec->sw[i];
      }
   }
}


/************************************************************************
* Converts folded frequency and folding value into unfolded frequency   *
************************************************************************/

void
UnFold(spec,freq,fold)

Spec_Param_Data *spec;	      /* pointer to spectral data              */
float freq[DIMENSION];        /* frequency in ppm                      */
short int fold[DIMENSION];    /* number of times frequency was folded  */

{
   int i;
   int nr;
   float low,up,w;

   for(i=0;i<spec->dim;i++) {
      up = spec->max_ppm[i]; low = spec->max_ppm[i]-spec->sw[i];
      if(spec->folding[i] == TPPI) {
         if((fold[i]/2)*2 != fold[i]) freq[i] = up + low - freq[i];
         freq[i] += fold[i] * spec->sw[i];
      }
      if(spec->folding[i] == RSH) {
         freq[i] += fold[i] * spec->sw[i];
      }
   }
}


