/*
************************************************************************
*
*   bib.h -
*
*   Copyright (c) 1995
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   All Rights Reserved
*
*   Date of last modification : 95/09/15
*   Pathname of SCCS file     : /export/home3/cb/garant-1.0/src/SCCS/s.bib.h
*   SCCS identification       : 1.2
*
************************************************************************
*/
/**************************************************************************/
/*                              bib.h                                     */
/*                                                                        */
/* classes to access text files                                           */
/**************************************************************************/


#ifndef _BIB_H_
#define _BIB_H_

#include "global.h"
#include "wert.h"
#include <stdio.h>

/* path for libraries */
extern char LIB_PATH[];

class AssRequest;
class PeakRequest;

class atomNamesT;

class BibAccess {
public:
   BibAccess() {}
   int readlib(char *);			/* read ecepp library defining    */
					/* fragment types                 */
   int readSecShifts(char *);		/* read lib. with 2nd struc.      */
					/* chemical shifts                */
   int readEquivAt(char *);		/* read library defining equiv.   */
					/* atom names                     */
   int readEquivFrags(char *);		/* read library gouping fragments */
					/* of the same spin sysstem type  */
   int fragLoad(char *,int);		/* add sequence of frags. to the  */
					/* list of subfragments           */
					/* INPUT: filename		  */
					/*        sub. or dest. fragment ?*/
					/* returns TRUE for success       */
   int coLoad(char *, int);		/* load coherence file            */
					/* INPUT: filename		  */
					/*        sub. or dest. fragment ?*/
					/* returns TRUE for success       */
   int equivAt(int,int,int,int);	/* are two atoms equivalent       */
					/* INPUT: atom nr. and frag. nr of*/
					/*        first and second atom   */
					/*        (a1,f1,a2,f2)           */
					/* if both atoms belong to subfrag*/
					/* or dest. fragment they must be */
					/* of the same ss type            */
					/* if one belongs to a subfr. and */
					/* the other to a dest. frag. a   */
					/* mapping must be defined        */
					/* equivalent names are tested    */
					/* only for the first atom        */
   int equivFrag(char *,char *);	/* are two fragments equivalent   */
					/* INPUT: name of                 */
					/*        first and second frag.  */
					/*        (f1,f2)                 */
   int equivAtNam(char *, char *);	/* are two atom names equivalent ?*/
};

/* General parent class for reading ***************************************/
class Request {
protected:
   int index;				/* number of the read element      */
public:
   Request() {index = 0;eof=1;}
   ~Request() {}
   int eof;       			/* 1  .. EOF, 0 .. ok, -1 .. error */
};

/* Read all available atoms ********************************************/
class AtomRequest : public Request {
   int iAt;				/* number of atoms within fragment    */
   int iSub;				/* number of subfragment within seq.  */
   int oldMolNr;			/* to decide atom belongs to          */
					/* N-Terminus                         */
public:
   AtomRequest();
   void start();			/* Read all atoms                     */
   void read();
   char name[MAXNAME];			/* name of the amino acid             */
   char type[MAXNAME];			/* type of the amino acid             */
   int nrAa;				/* sequence number for EASY           */
   int nrFrag;				/* internal fragment number           */
   int nr;				/* atomnr.                            */
   int molNr;				/* number of the mollecule            */
   int typeFrag;			/* dest. or sub-fragment ?            */
   float shift,delshift;		/* chemical shift                     */
   float prob;                    	/* probability of degeneracy          */
};

/* read all available fragments */
class FragRequest : public Request {
   int iSub;				/* number of subfragment within seq.  */
public:
   FragRequest();
   void start();			/* read all fragments                 */
   void read();
   char *name;				/* name of the fragment               */
   int nrInt;				/* internal number of the fragment    */
   int nrExt;				/* external (EASY) number             */
   int type;				/* sub. or dest. fragment ?           */
   int equivNr;				/* same number ==same spin system type*/
};

/* Read all available bonds ********************************************/
class BondRequest : public Request {
   int iSub;				/* number of subfragment within seq.  */
   int iAt;				/* number of atoms within fragment    */
   int iBd;				/* number of bond for atom            */
public:
   BondRequest();
   void start();   			/* Read all bonds                     */
   void read();
   int nr1,nr2;				/* number of the connected atoms      */
   int molNr;				/* number of the mollecule            */
};

/* Read all pseudo astoms ********************************************/
class PseudRequest : public Request {
   int iSub;				/* number of subfragment within seq.  */
   int iPseud;				/* number of pseudo atoms within frag.*/
   int iAt;				/* number of atom of pseudo atom      */
public:
   PseudRequest();
   void start();   			/* Read all bonds                     */
   void read();
   int pseudnr,atnr;			/* number of atom and pseudoatom      */
};

class SecShiftRequest:public Request {  /* get secondary struct. shifts       */
public:
   int get(int,int,char *);
   Wert shift;
};


class FreqRequest : public Request {
   int iCo;				/* current coherence                  */
public:
   FreqRequest();
   void start();   			/* Read all coherences                */
   void read();
   char name[MAXNAME];			/* name of the coherence              */
   int nrExt;				/* external number of co              */
   Wert shift;				/* chemical shift (in ppm)            */
};

class PeakRequest : public Request {	/* read chemical shifts and volume    */
					/* from EASY peaklist *.peaks         */
   int peaknr;				/* number of peak read in from file   */
   FILE *fp;				/* peaklist filepointer               */
   int readline();			/* read line of peak file             */
   char line[MAXLINE];			/* current line                       */
   int nrCo[DIMENSION];			/* coherence numbers from file        */
public:
   friend AssRequest;
   PeakRequest();
   void start(char *,int,float,int *);	/* iterate over all peaks in file     */
   virtual void read();			/* dim & perm & noise have to be set  */
   int dim;				/* dimensionality of peak list        */
   int perm[DIMENSION];			/* order of dimesnions                */
   float nl;				/* noise level used to calc. prob     */
   int nr;				/* external (EASY) peak number        */
   float w[DIMENSION];			/* chemical shift of peak (ppm)       */
   Wert vol;				/* volume and volume error            */
   float prob;				/* prob. that peak is no artefact     */
   char comment[MAX_COMMENT];		/* comment for the peak               */
   int color;				/* peak color                         */
   char intflag;			/* integration flag                   */
};

class AssRequest : public PeakRequest {	/* read names of assigned co's        */
   void genCoNames();			/* generate co's names from nrCo[]    */
   atomNamesT *names[DIMENSION];	/* reference to atom names structure  */
public:
   int nrFrag[DIMENSION];		/* assigned external frag. number     */
   virtual void read();			/* iterate over all peaks in file     */
   int equivName(int,char *);		/* is name in given dim. equivalent ? */
   char *getName(int);			/* get co name of ass. in given dim.  */
};

class CoordRequest : public Request {	/* read atom coordinates              */
					/* from DIANA peaklist *.cor          */
   FILE *fp;				/* filepointer                        */
   int readline();			/* read line of coordinate file       */
   char line[MAXLINE];			/* current line                       */
public:
   CoordRequest();
   void start(char *);			/* iterate over all atoms in file     */
   void read();				/* dim & perm & noise have to be set  */
   float coord[3];			/* coordinate of atom                 */
   int nrExt;				/* external number of peak            */
   char name[MAXNAME];			/* atom name                          */
};


class FragNrRequest : public Request {	/* return all internal fragment       */
					/* numbers matching a given name      */
   int curFrag;				/* current position within frag list  */
public:
   char name[MAXNAME];			/* name of fragment, may contain wild */
					/* carts                              */
   int type;				/* sub or destination fragment ?      */
   char resname[MAXNAME];		/* residue name                       */
   int nrExt;				/* external number of residue         */
   int nrDef;				/* is the external number defined     */
   int nameDef;				/* is the frag. name defined          */
   int nr;				/* internal number of found fragment  */
   void start();			/* iteration over all possible frags  */
   void read();
};

class FragNameRequest : public Request {/* return fragment name for given     */
					/* internal frag. number              */
public:
   char name[MAXNAME];			/* name of fragment                   */
   char resname[MAXNAME];		/* residue name                       */
   int nrExt;				/* external number of residue         */
   int nr;				/* internal number of fragment        */
   void start();			/* get name                           */
   void read() {eof =1; }
};

class OptPosRequest : public Request {	/* read an optimization possibility   */
   FILE *fp;
   char line[MAXLINE];			/* current line                       */
public:
   OptPosRequest();
   void start(char *);			/* iterate over all peaks in file     */
   void read();				/* read next entry                    */
   int nrPeak;				/* external (EASY) peak number        */
   float prob;				/* local ass. probability	      */
};

/* Writing different files *************************************************/

class Writer {
public:
   int eof;				/* 1 .. eof, 0 .. ok, -1 .. error     */
   char fn[MAXFILENAME];		/* filename                           */
   FILE *fp;				/* file                               */
   Writer() {eof = 1; }
   void writeCom(char *);		/* write line of comment to the file  */
   void close() {fclose(fp); eof=1; }   /* close file                         */
};

class CoWriter : public Writer {
   int nrCo;
public:
   char *name;				/* Name of the coherence              */
   int nrAa;				/* Number of corresbonding aa in EASY */
   Wert shift;				/* chem. shift and error              */
   void start(char *);			/* start writing coherence file with  */
					/* given name                         */
   int write();				/* Writes and returns number of the co*/
};

class PeakWriter : public Writer {
   int curPeak;				/* number of peak written             */
public:
   int  dim;				/* dimensionality of peak list        */
   int perm[DIMENSION];			/* permutation for writing peaks out  */
   Wert w[DIMENSION];			/* shift of peak                      */
   Wert v;				/* peak volume                        */
   char coAtNam[DIMENSION][MAXNAME];	/* atom names of the coherences       */
   int coResNum[DIMENSION];		/* residue numbers of the coherences  */
   int nrPeak; 				/* peak number                        */
   char comment[MAX_COMMENT];		/* comment for the peak               */
   int color;				/* peak color                         */
   char intflag;			/* integration flag                   */
   void start(char *);			/* open file, write header            */
   void write();			/* write peak list entry              */
};

class OptPosWriter : public Writer {	/* write an optimization possibility  */
public:
   void start(char *);			/* open file for writing              */
   void write();			/* write next entry                   */
   int nrPeak;				/* external (EASY) peak number        */
   float prob;				/* local probability of assignment    */
};

class AssWriter : public Writer {
   int curPeak;				/* peak count                         */
public:
   int nrPeak;				/* external peak number               */
   int dim;				/* dimensionality of peak list        */
   int perm[DIMENSION];			/* permutation for writing peaks out  */
   char coAtNam[DIMENSION][MAXNAME];	/* atom names of the coherences       */
   int coResNum[DIMENSION];		/* residue numbers of the coherences  */
   char namCoLi[MAXLINE];		/* name of coherence file             */
   float tolerance[DIMENSION];		/* tolerance used for assignments     */
   void start(char *);			/* open file write header             */
   void startPeak();			/* start entry for new peak           */
   void writeAssPos();			/* write an assignment possibility    */
					/* for the current peak               */
};


#endif
