/*
************************************************************************
*
*   RandomCost.c - minimization by random cost
*
*   Copyright (c) 1994-95
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 95/12/12
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.RandomCost.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

/* Reference: Locating global minima in optimization problems by a
   random-cost approach, Nature, Vol 361, Feb 93, 708-710 */

#include <random_cost.h>

#include <stdlib.h>
#include <math.h>

#define MAX_STEP 1000

static float
random1(void)
{
  return (float) rand() / (((unsigned) 1 << 15) - 1);
}

void
RandomCostGenStart(float *xStart, int dim, float *xMin, float *xMax)
{
  int i;

  for (i = 0; i < dim; i++)
    xStart[i] = xMin[i] + random1() * (xMax[i] - xMin[i]);
}

BOOL
RandomCostMin(MinFunc func, int dim, float *xCurr, void *clientData,
    float *xMin, float *xMax, float *xInc, int prec)
{
  float *incFac;
  int updateNo, step;
  int *negInd, *posInd;
  float *negInc, *posInc;
  float d, valCurr, xOld, inc;
  int negNo, posNo;
  float negSum, posSum, negAvg, posAvg;
  float cost, prob;
  int ind, i, k;

  updateNo = 2 * dim * prec;

  negInd = malloc(updateNo * sizeof(*negInd));
  posInd = malloc(updateNo * sizeof(*posInd));
  negInc = malloc(updateNo * sizeof(*negInc));
  posInc = malloc(updateNo * sizeof(*posInc));

  incFac = malloc(2 * prec * sizeof(*incFac));
  d = 1.0;
  for (i = 0; i < prec; i++) {
    incFac[2 * i] = d;
    incFac[2 * i + 1] = - d;
    d /= 2.0;
  }

  /* iteration */
  for (step = 0; step < MAX_STEP; step++) {
    valCurr = func(xCurr, dim, clientData);

    negNo = 0;
    posNo = 0;
    negSum = 0.0;
    posSum = 0.0;

    for (i = 0; i < dim; i++) {
      xOld = xCurr[i];
      for (k = 0; k < 2 * prec; k++) {
	inc = incFac[k] * xInc[i];
	xCurr[i] = xOld + inc;

	if (xCurr[i] < xMin[i] || xCurr[i] > xMax[i])
	  continue;

	cost = func(xCurr, dim, clientData) - valCurr;
	if (cost == 0.0)
	  continue;

	if (cost <= 0.0) {
	  negInd[negNo] = i;
	  negInc[negNo] = inc;
	  negNo++;
	  negSum += cost;
	} else {
	  posInd[posNo] = i;
	  posInc[posNo] = inc;
	  posNo++;
	  posSum += cost;
	}
      }
      xCurr[i] = xOld;
    }

    if (negNo == 0)
      return TRUE;

    negAvg = negSum / negNo;

    if (posNo == 0) {  /* maximum */
      i = rand() % negNo;
      ind = negInd[i];
      inc = negInc[i];
    } else {
      posAvg = posSum / posNo;

      prob = posAvg / (posAvg - negAvg);

      if (random1() < prob) {
	i = rand() % negNo;
	ind = negInd[i];
        inc = negInc[i];
      } else {
	i = rand() % posNo;
	ind = posInd[i];
        inc = posInc[i];
      }
    }

    xCurr[ind] += inc;
  }

  return FALSE;
}
