/*
************************************************************************
*
*   ModCalc.c - modular arithmetics for BIGINT
*
*   Copyright (c) 1994
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 94/06/02
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.ModCalc.c
*   SCCS identification       : 1.1
*
************************************************************************
*/
#include <mod_calc.h>

void
McPow(                                  /* z = x ^ y (m) */
    BIGINT z,
    BIGINT x,
    BIGINT y,
    BIGINT m
)
{
  BIGINT  x1, y1;

  x1 = BiNew();
  y1 = BiNew();

  BiAss(x1, x);
  BiAss(y1, y);
  BiIntAss(z, 1);

  while (!BiZero(y1))
    if (BiOdd(y1)) {
      BiMul(z, x1);
      BiMod(z, m);
      BiDec(y1);
    } else {
      BiMul(x1, x1);
      BiMod(x1, m);
      BiDiv2(y1);
    }

  BiFree(x1);
  BiFree(y1);
}

static int
SignSub(int s1, BIGINT x1, int s2, BIGINT x2)
{
  BIGINT  h;
  int     s;

  h = BiNew();

  if (s1)
    if (s2)
      if (BiLess(x1, x2)) {
        BiAss(h, x1);
        BiAss(x1, x2);
        BiSub(x1, h);
        s = 0;
      } else {
        BiSub(x1, x2);
        s = 1;
    } else {
      BiAdd(x1, x2);
      s = 1;
  } else if (s2) {
    BiAdd(x1, x2);
    s = 0;
  } else if (BiLess(x2, x1)) {
    BiSub(x1, x2);
    s = 0;
  } else {
    BiAss(h, x1);
    BiAss(x1, x2);
    BiSub(x1, h);
    s = 1;
  }

  BiFree(h);
  return s;
}

void
McInv(                                  /* a * x = 1 (m) */
    BIGINT a,
    BIGINT x,
    BIGINT m
)
{
  BIGINT  a1, m1, y, u, v, t, h, tu, s, tx;
  int     sy, sv;                       /* signs */

  a1 = BiNew();
  m1 = BiNew();
  y = BiNew();
  u = BiNew();
  v = BiNew();
  t = BiNew();
  h = BiNew();
  tu = BiNew();
  s = BiNew();
  tx = BiNew();

  BiAss(a1, a);
  BiAss(m1, m);
  BiIntAss(x, 1);
  BiIntAss(y, 0);
  sy = 1;
  BiIntAss(u, 0);
  BiIntAss(v, 1);
  sv = 1;

  while (!BiEqual(a1, m1))
    if (BiLess(m1, a1)) {
      /* t = (a1 - m1 - 1) / m1 + 1 */
      BiAss(t, a1);
      BiSub(t, m1);
      BiDec(t);
      BiDiv(t, m1);
      BiInc(t);
      /* a1 = a1 - t * m1 */
      BiAss(h, t);
      BiMul(h, m1);
      BiSub(a1, h);
      BiAss(tu, t);
      BiMul(tu, u);
      if (BiLess(x, tu)) {
        /* s = (t * u - x - 1) / m + 1 */
        BiAss(s, tu);
        BiSub(s, x);
        BiDec(s);
        BiDiv(s, m);
        BiInc(s);
        /* x = x + s * m */
        BiAss(h, s);
        BiMul(h, m);
        BiAdd(x, h);
        /* y = y - s * a */
        BiAss(h, s);
        BiMul(h, a);
        sy = SignSub(sy, y, 1, h);
      }
      /* x = x - t * u */
      BiSub(x, tu);
      /* y = y - t * v */
      BiAss(h, t);
      BiMul(h, v);
      sy = SignSub(sy, y, sv, h);
    } else {
      /* t = (m1 - a1 - 1) / a1 + 1 */
      BiAss(t, m1);
      BiSub(t, a1);
      BiDec(t);
      BiDiv(t, a1);
      BiInc(t);
      /* m1 = m1 - t * a1 */
      BiAss(h, t);
      BiMul(h, a1);
      BiSub(m1, h);
      BiAss(tx, t);
      BiMul(tx, x);
      if (BiLess(u, tx)) {
        /* s = (t * x - u - 1) / m + 1 */
        BiAss(s, tx);
        BiSub(s, u);
        BiDec(s);
        BiDiv(s, m);
        BiInc(s);
        /* u = u + s * m */
        BiAss(h, s);
        BiMul(h, m);
        BiAdd(u, h);
        /* v = v - s * a */
        BiAss(h, s);
        BiMul(h, a);
        sv = SignSub(sv, v, 1, h);
      }
      /* u = u - t * x */
      BiSub(u, tx);
      /* v = v - t * y */
      BiAss(h, t);
      BiMul(h, y);
      sv = SignSub(sv, v, sy, h);
    }

  BiFree(a1);
  BiFree(m1);
  BiFree(y);
  BiFree(u);
  BiFree(v);
  BiFree(t);
  BiFree(h);
  BiFree(tu);
  BiFree(s);
  BiFree(tx);
}
