/*
************************************************************************
*
*   GaussJordan.c - linear equation solution by Gauss-Jordan elimination
*
*   Copyright (c) 1994
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 95/07/20
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.GaussJordan.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <gauss_jordan.h>

#include <stdlib.h>
#include <math.h>

#define SWAP(a, b) {float temp = a; a = b; b = temp;}

/* Linear equation solution by Gauss-Jordan elimination.
   Algorithm adapted from Numerical Recipes. */

BOOL
GaussJordan(float **a, float *b, int n)
{
  int *indxc, *indxr, *ipiv;
  int i, icol, irow, j, k, l, ll;
  float big, dum, pivinv;

  indxc = malloc(n * sizeof(*indxc));
  indxr = malloc(n * sizeof(*indxr));
  ipiv = malloc(n * sizeof(*ipiv));

  for (j = 0; j < n; j++)
    ipiv[j] = 0;
  
  for (i = 0; i < n; i++) {
    big = 0.0;

    for (j = 0; j < n; j++) {
      if (ipiv[j] == 1) 
	continue;

      for (k = 0; k < n; k++) {
	if (ipiv[k] > 1) {
	  free(indxc);
	  free(indxr);
	  free(ipiv);
	  return FALSE;
	}

	if (ipiv[k] == 1)
	  continue;

	if (a[j][k] >= big) {
	  big = a[j][k];
	  irow = j;
	  icol = k;
	} else if (a[j][k] <= - big) {
	  big = - a[j][k];
	  irow = j;
	  icol = k;
	}
      }
    }

    ipiv[icol]++;

    if (irow != icol) {
      for (l = 0; l < n; l++)
	SWAP(a[irow][l], a[icol][l]);
      SWAP(b[irow], b[icol]);
    }

    indxr[i] = irow;
    indxc[i] = icol;

    if (a[icol][icol] == 0.0) {
      free(indxc);
      free(indxr);
      free(ipiv);
      return FALSE;
    }

    pivinv = 1.0 / a[icol][icol];

    a[icol][icol] = 1.0;
    for (l = 0; l < n; l++)
      a[icol][l] *= pivinv;
    b[icol] *= pivinv;

    for (ll = 0; ll < n; ll++) {
      if (ll == icol)
	continue;
      
      dum = a[ll][icol];
      a[ll][icol] = 0.0;
      for (l = 0; l < n; l++)
	a[ll][l] -= a[icol][l] * dum;
      b[ll] -= b[icol] * dum;
    }
  }

  for (l = n - 1; l >= 0; l--) {
    if (indxr[l] == indxc[l])
      continue;

    for (k = 0; k < n; k++)
      SWAP(a[k][indxr[l]], a[k][indxc[l]]);
  }

  free(indxc);
  free(indxr);
  free(ipiv);

  return TRUE;
}
