/*
************************************************************************
*
*   ContourLine.c - contour line calculation
*
*   Copyright (c) 1995
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/04/12
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.ContourLine.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include "contour_line.h"

#include <stdio.h>
#include <stdlib.h>
#include <values.h>

#include <bool.h>
#include <mat_vec.h>

typedef enum {
  P_LEFT,
  P_RIGHT,
  P_BOTTOM,
  P_TOP
} Pos;

static void
traceContour(float *valA, int xs, int ys,
    int *cellA, int startI, BOOL interiour, Pos prev,
    float xMin, float yMin, float xFact, float yFact, float level,
    ContourLine **contAP, int *contNoP)
{
  int contSize, pNo, currI;
  float (*p)[2];
  int i1, i2, i;
  float c1, c2;
  BOOL finished;
  int pat;
  ContourLine *contA;
  int contNo;

  contSize = 10;
  p = malloc(contSize * sizeof(*p));
  pNo = 0;

  currI = startI;
  i1 = startI % (xs - 1);
  i2 = startI / (xs - 1);
  finished = FALSE;

  for (;;) {
    switch (prev) {
      case P_LEFT:
	c1 = i1;
	c2 = i2 + (level - valA[i2 * xs + i1]) /
	    (valA[(i2 + 1) * xs + i1] - valA[i2 * xs + i1]);
	break;
      case P_RIGHT:
	c1 = i1 + 1;
	c2 = i2 + (level - valA[i2 * xs + i1 + 1]) /
	    (valA[(i2 + 1) * xs + i1 + 1] - valA[i2 * xs + i1 + 1]);
	break;
      case P_BOTTOM:
	c1 = i1 + (level - valA[i2 * xs + i1]) /
	    (valA[i2 * xs + i1 + 1] - valA[i2 * xs + i1]);
	c2 = i2;
	break;
      case P_TOP:
	c1 = i1 + (level - valA[(i2 + 1) * xs + i1]) /
	    (valA[(i2 + 1) * xs + i1 + 1] - valA[(i2 + 1) * xs + i1]);
	c2 = i2 + 1;
	break;
    }

    if (pNo >= contSize) {
      contSize *= 2;
      p = realloc(p, contSize * sizeof(*p));
    }

    p[pNo][0] = c1 * xFact + xMin;
    p[pNo][1] = c2 * yFact + yMin;
    pNo++;

    if (finished)
      break;

    pat = cellA[currI];

    switch (pat) {
      case 1:
      case 3:
      case 11:
	i1--;
	finished = (i1 == -1);
	prev = P_RIGHT;
	break;
      case 8:
      case 12:
      case 13:
	i1++;
	finished = (i1 == xs - 1);
	prev = P_LEFT;
	break;
      case 2:
      case 10:
      case 14:
	i2--;
	finished = (i2 == -1);
	prev = P_TOP;
	break;
      case 4:
      case 5:
      case 7:
	i2++;
	finished = (i2 == ys - 1);
	prev = P_BOTTOM;
	break;
      case 6:
	if (prev == P_LEFT) {
	  i2++;
	  finished = (i2 == ys - 1);
	  prev = P_BOTTOM;
	} else {
	  i2--;
	  finished = (i2 == -1);
	  prev = P_TOP;
	}
	break;
      case 9:
	if (prev == P_TOP) {
	  i1--;
	  finished = (i1 == -1);
	  prev = P_RIGHT;
	} else {
	  i1++;
	  finished = (i1 == xs - 1);
	  prev = P_LEFT;
	}
	break;
    }

    if (pat != 6 && pat != 9)
      cellA[currI] = -1;

    currI = i2 * (xs - 1) + i1;

    if (interiour && currI == startI)
      finished = TRUE;
  }

  contA = *contAP;
  contNo = *contNoP;

  contA = realloc(contA, (contNo + 1) * sizeof(*contA));

  contA[contNo].xA = realloc(p, pNo * sizeof(*p));
  contA[contNo].pointNo = pNo;
  contNo++;

  *contAP = contA;
  *contNoP = contNo;
}

void
ContourLineCalc(float *valA, int xs, int ys,
    float xMin, float yMin, float xRange, float yRange, float level,
    ContourLine **contAP, int *contNoP)
{
  int cxs, cys;
  int *cellA;
  float xFact, yFact;
  ContourLine *contA = NULL;
  int contNo;
  int startI, pNo;
  float (*p)[2];
  int i1, i2, i;

  cxs = xs - 1;
  cys = ys - 1;

  cellA = malloc(cxs * cys * sizeof(*cellA));
  for (i = 0; i < cxs * cys; i++)
    cellA[i] = 0;

  for (i2 = 0; i2 < ys; i2++)
    for (i1 = 0; i1 < xs; i1++)
      if (valA[i2 * xs + i1] > level) {
	if (i1 < cxs && i2 < cys)
	  cellA[i2 * cxs + i1] |= 1;
	if (i1 > 0 && i2 < cys)
	  cellA[i2 * cxs + i1 - 1] |= 2;
	if (i2 > 0 && i1 < cxs)
	  cellA[(i2 - 1) * cxs + i1] |= 4;
	if (i1 > 0 && i2 > 0)
	  cellA[(i2 - 1) * cxs + i1 - 1] |= 8;
      }

  xFact = xRange / (xs - 1);
  yFact = yRange / (ys - 1);

  contA = malloc(sizeof(*contA));
  contNo = 0;

  /* first do the ones that start at bottom border */
  startI = 0;
  for (i1 = 0; i1 < cxs; i1++) {
    switch (cellA[startI]) {
      case 1:
      case 5:
      case 9:
      case 13:
	traceContour(valA, xs, ys, cellA, startI, FALSE, P_BOTTOM,
	    xMin, yMin, xFact, yFact, level,
	    &contA, &contNo);
	break;
      default:
	/* do nothing */
	break;
    }
    startI++;
  }

  /* top border */
  startI = (cys - 1) * cxs;
  for (i1 = 0; i1 < cxs; i1++) {
    switch (cellA[startI]) {
      case 8:
      case 9:
      case 10:
      case 11:
	traceContour(valA, xs, ys, cellA, startI, FALSE, P_TOP,
	    xMin, yMin, xFact, yFact, level,
	    &contA, &contNo);
	break;
      default:
	/* do nothing */
	break;
    }
    startI++;
  }

  /* left border */
  startI = 0;
  for (i2 = 0; i2 < cys; i2++) {
    switch (cellA[startI]) {
      case 4:
      case 6:
      case 12:
      case 14:
	traceContour(valA, xs, ys, cellA, startI, FALSE, P_LEFT,
	    xMin, yMin, xFact, yFact, level,
	    &contA, &contNo);
	break;
      default:
	/* do nothing */
	break;
    }
    startI += cxs;
  }

  /* right border */
  startI = cxs - 1;
  for (i2 = 0; i2 < cys; i2++) {
    switch (cellA[startI]) {
      case 2:
      case 3:
      case 6:
      case 7:
	traceContour(valA, xs, ys, cellA, startI, FALSE, P_RIGHT,
	    xMin, yMin, xFact, yFact, level,
	    &contA, &contNo);
	break;
      default:
	/* do nothing */
	break;
    }
    startI += cxs;
  }

  /* interiour */
  for (i2 = 0; i2 < cys; i2++)
    for (i1 = 0; i1 < cxs; i1++) {
      startI = i2 * cxs + i1;
      switch (cellA[startI]) {
	case 1:
	case 5:
	case 13:
	  traceContour(valA, xs, ys, cellA, startI, TRUE, P_BOTTOM,
	      xMin, yMin, xFact, yFact, level,
	      &contA, &contNo);
	  break;
	case 8:
	case 10:
	case 11:
	  traceContour(valA, xs, ys, cellA, startI, TRUE, P_TOP,
	      xMin, yMin, xFact, yFact, level,
	      &contA, &contNo);
	  break;
	case 4:
	case 12:
	case 14:
	  traceContour(valA, xs, ys, cellA, startI, TRUE, P_LEFT,
	      xMin, yMin, xFact, yFact, level,
	      &contA, &contNo);
	  break;
	case 2:
	case 3:
	case 7:
	  traceContour(valA, xs, ys, cellA, startI, TRUE, P_RIGHT,
	      xMin, yMin, xFact, yFact, level,
	      &contA, &contNo);
	  break;
	default:
	  /* do nothing */
	  break;
      }
    }

  free(cellA);

  if (contNo == 0) {
    free(contA);
    contA = NULL;
  }

  *contAP = contA;
  *contNoP = contNo;
}
