/*
************************************************************************
*
*   PrimHand.c - manage data for primaries
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/13
*   Pathname of SCCS file     : /sgiext/autopsy/src/prim/SCCS/s.PrimHand.c
*   SCCS identification       : 1.1
*
************************************************************************
*/

#include <prim_hand.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <linlist.h>
#include <prop_tab.h>
#include <attr_struc.h>
#include <attr_mng.h>
#include <data_hand.h>
#include "prim_struc.h"

typedef struct {
  PrimType type;
  PropRefP refP;
  PrimPropCB propCB;
  void *clientData;
} PropCBData;

static LINLIST PrimList = NULL;
static LINLIST PropCBList = NULL;

static void
primSpecInvalidCB(DhSpecP specP, void *clientData)
{
  PrimObjP primP = clientData;

  if (primP->specP == specP)
    PrimDestroy(primP);
}

void
PrimAddInvalidCB(PrimObjP primP)
{
  if (primP->specP != NULL)
    DhAddSpecInvalidCB(primSpecInvalidCB, primP);
}

void
PrimDestroyAll(void)
{
  ListClose(PrimList);
  PrimList = NULL;
}

static void
freePrim(void *p, void *clientData)
{
  PrimObjP primP = p;
  int i;

  if (primP->specP != NULL)
    DhRemoveSpecInvalidCB(primSpecInvalidCB, primP);

  AttrReturn(primP->attrP);
  PropFreeTab(primP->propTab);

  switch (primP->type) {
    case PT_TEXT:
      if (primP->u.text.str != NULL)
	free(primP->u.text.str);
      break;
    case PT_DRAWOBJ:
      if (primP->u.drawobj.xA != NULL) {
	free(primP->u.drawobj.xA);
	free(primP->u.drawobj.dxA);
      }
      break;
  }
}

PrimObjP
PrimNew(PrimType type, DhSpecP specP)
{
  struct PrimObjS primS;
  PrimObjP primP;
  PropRefP refP;

  if (PrimList == NULL) {
    PrimList = ListOpen(sizeof(struct PrimObjS));
    ListAddDestroyCB(PrimList, freePrim, NULL, NULL);
    primS.num = 0;
  } else if (ListSize(PrimList) == 0) {
    primS.num = 0;
  } else {
    primP = ListLast(PrimList);
    primS.num = primP->num + 1;
  }

  primS.type = type;
  primS.specP = specP;

  if (type == PT_TEXT) {
    primS.u.text.str = NULL;
  } else if (type == PT_DRAWOBJ) {
    primS.u.drawobj.type = PDT_LINE;
    primS.u.drawobj.style = PDS_LINE;
    primS.u.drawobj.xA = NULL;
    primS.u.drawobj.dxA = NULL;
    primS.u.drawobj.pointNo = 0;
  }

  primS.attrP = AttrGetInit();
  primS.propTab = PropNewTab(TRUE);

  primP = ListInsertLast(PrimList, &primS);
  PrimAddInvalidCB(primP);

  return primP;
}

void
PrimDestroy(PrimObjP primP)
{
  int num;

  ListRemove(PrimList, primP);

  primP = ListFirst(PrimList);
  num = 0;
  while (primP != NULL) {
    primP->num = num;
    primP = ListNext(PrimList, primP);
    num++;
  }
}

void
PrimApply(PrimType type, PropRefP refP,
    PrimApplyFunc applyF, void *clientData)
{
  PrimObjP primP, nextPrimP;

  primP = ListFirst(PrimList);
  while (primP != NULL) {
    nextPrimP = ListNext(PrimList, primP);
    if ((type == PT_ALL || type == primP->type) &&
	(primP->propTab[refP->index] & refP->mask))
      applyF(primP, clientData);
    primP = nextPrimP;
  }
}

void
PrimSpecApply(PrimType type, PropRefP refP, DhSpecP specP,
    PrimApplyFunc applyF, void *clientData)
{
  PrimObjP primP, nextPrimP;

  primP = ListFirst(PrimList);
  while (primP != NULL) {
    nextPrimP = ListNext(PrimList, primP);
    if ((type == PT_ALL || type == primP->type) && specP == primP->specP &&
	(primP->propTab[refP->index] & refP->mask))
      applyF(primP, clientData);
    primP = nextPrimP;
  }
}

PrimObjP
PrimFindNumber(int num)
{
  return ListPos(PrimList, num - 1);
}

void
PrimAddPropCB(PrimType type, PropRefP refP,
    PrimPropCB propCB, void *clientData)
{
  PropCBData cbData;

  if (PropCBList == NULL)
    PropCBList = ListOpen(sizeof(PropCBData));
  
  cbData.type = type;
  cbData.refP = refP;
  cbData.propCB = propCB;
  cbData.clientData = clientData;

  (void) ListInsertLast(PropCBList, &cbData);
}

void
PrimSetProp(PropRefP refP, PrimObjP primP, BOOL onOff)
{
  PropCBData *cbDataP;

  if (onOff == PrimGetProp(refP, primP))
    return;

  primP->propTab = PropChange(primP->propTab, refP, onOff);

  cbDataP = ListFirst(PropCBList);
  while (cbDataP != NULL) {
    if (cbDataP->refP == refP &&
        (cbDataP->type == PT_ALL || cbDataP->type == primP->type))
      cbDataP->propCB(primP, refP, onOff, cbDataP->clientData);
    cbDataP = ListNext(PropCBList, cbDataP);
  }
}

BOOL
PrimGetProp(PropRefP refP, PrimObjP primP)
{
  return (primP->propTab[refP->index] & refP->mask) != 0;
}

unsigned *
PrimGetPropTab(PrimObjP primP)
{
  return primP->propTab;
}

void
PrimSetAttr(PrimObjP primP, AttrP attrP)
{
  primP->attrP = attrP;
}

void
PrimSetPos(PrimObjP primP, Vec3 x)
{
  Vec3Copy(primP->u.text.x, x);
}

void
PrimSetVec(PrimObjP primP, Vec3 v)
{
  Vec3Copy(primP->u.text.dx, v);
}

void
PrimSetText(PrimObjP primP, char *str)
/* str is copied, must be freed by caller if it was malloced */
{
  if (primP->u.text.str != NULL)
    free(primP->u.text.str);
  
  primP->u.text.str = malloc(strlen(str) + 1);
  (void) strcpy(primP->u.text.str, str);
}

void
PrimSetDrawobjType(PrimObjP primP, PrimDrawobjType type)
{
  primP->u.drawobj.type = type;
}

void
PrimSetDrawobjStyle(PrimObjP primP, PrimDrawobjStyle style)
{
  primP->u.drawobj.style = style;
}

void
PrimSetPoint(PrimObjP primP, int ind, Vec3 x, Vec3 dx)
{
  if (primP->u.drawobj.xA == NULL) {
    primP->u.drawobj.xA = malloc((ind + 1) * sizeof(*primP->u.drawobj.xA));
    primP->u.drawobj.dxA = malloc((ind + 1) * sizeof(*primP->u.drawobj.dxA));
    primP->u.drawobj.pointNo = ind + 1;
  } else if (ind >= primP->u.drawobj.pointNo) {
    primP->u.drawobj.xA = realloc(primP->u.drawobj.xA,
	(ind + 1) * sizeof(*primP->u.drawobj.xA));
    primP->u.drawobj.dxA = realloc(primP->u.drawobj.dxA,
	(ind + 1) * sizeof(*primP->u.drawobj.dxA));
    primP->u.drawobj.pointNo = ind + 1;
  }

  Vec3Copy(primP->u.drawobj.xA[ind], x);
  Vec3Copy(primP->u.drawobj.dxA[ind], dx);
}

int
PrimGetNumber(PrimObjP primP)
{
  return primP->num + 1;  /* return numbers starting at 1 */
}

PrimType
PrimGetType(PrimObjP primP)
{
  return primP->type;
}

DhSpecP
PrimGetSpec(PrimObjP primP)
{
  return primP->specP;
}

AttrP
PrimGetAttr(PrimObjP primP)
{
  return primP->attrP;
}

void
PrimGetPos(PrimObjP primP, Vec3 x)
{
  Vec3Copy(x, primP->u.text.x);
}

void
PrimGetVec(PrimObjP primP, Vec3 v)
{
  Vec3Copy(v, primP->u.text.dx);
}

char *
PrimGetText(PrimObjP primP)
/* returns a pointer to internal memory */
{
  return primP->u.text.str;
}

PrimDrawobjType
PrimGetDrawobjType(PrimObjP primP)
{
  return primP->u.drawobj.type;
}

PrimDrawobjStyle
PrimGetDrawobjStyle(PrimObjP primP)
{
  return primP->u.drawobj.style;
}

void
PrimGetPoint(PrimObjP primP, int ind, Vec3 x, Vec3 dx)
{
  Vec3Copy(x, primP->u.drawobj.xA[ind]);
  Vec3Copy(dx, primP->u.drawobj.dxA[ind]);
}

int
PrimGetPointNo(PrimObjP primP)
{
  return primP->u.drawobj.pointNo;
}

LINLIST
PrimListGet(void)
{
  return PrimList;
}
