/*
************************************************************************
*
*   ExprEval.c - expression evaluation
*
*   Copyright (c) 1994
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/15
*   Pathname of SCCS file     : /sgiext/autopsy/src/expr/SCCS/s.ExprEval.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <expr.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <string.h>
#include <math.h>

#include <mat_vec.h>
#include <dstr.h>
#include <strmatch.h>
#include <data_hand.h>
#include <prim_hand.h>
#include "expr_scan.h"

#define INIT_CODE_SIZE 10

typedef enum {
  OC_NOP,

  OC_PLUS_INT,
  OC_MINUS_INT,
  OC_MUL_INT,
  OC_DIV_INT,
  OC_REM,
  OC_NEG_INT,

  OC_PLUS_FLOAT,
  OC_MINUS_FLOAT,
  OC_MUL_FLOAT,
  OC_DIV_FLOAT,
  OC_NEG_FLOAT,
  OC_SQRT,
  OC_LOG,

  OC_DUP,
  OC_CAST,
  OC_CAST_1,

  OC_EQU_INT,
  OC_NEQ_INT,
  OC_LEQ_INT,
  OC_LSS_INT,
  OC_GEQ_INT,
  OC_GTR_INT,
  OC_INSET_INT,
  OC_OUTSET_INT,

  OC_EQU_FLOAT,
  OC_NEQ_FLOAT,
  OC_LEQ_FLOAT,
  OC_LSS_FLOAT,
  OC_GEQ_FLOAT,
  OC_GTR_FLOAT,
  OC_INSET_FLOAT,
  OC_OUTSET_FLOAT,

  OC_EQU_STR,
  OC_NEQ_STR,

  OC_OR,
  OC_AND,
  OC_NOT,

  OC_GET_INT,
  OC_GET_FLOAT,
  OC_GET_STR,
  OC_GET_PROP,

  OC_EXISTS,
  OC_ALL,

  OC_ENT_SPEC,
  OC_ENT_PEAK,
  OC_ENT_PRIM,

  OC_GET_NUMBER,
  OC_GET_NAME,
  OC_GET_AMP,
  OC_GET_SYMM,
  OC_GET_UNIF,
  OC_GET_QUAL,
  OC_GET_SHIFT0,
  OC_GET_SHIFT1,
  OC_GET_SHIFT2,
  OC_GET_SHIFT3,
  OC_GET_SHIFT4,
  OC_GET_WIDTH0,
  OC_GET_WIDTH1,
  OC_GET_WIDTH2,
  OC_GET_WIDTH3,
  OC_GET_WIDTH4,
  OC_GET_ATTR,

  OC_END
} OpCode;

struct ExprS {
  int sp;
  int maxStackSize;
  int codeSize;
  OpCode *code;
  int pcStart, pcEnd;
  DataEntityType entType;
  DhSpecP specP;
  DhPeakP peakP;
  PrimObjP primP;
  ExprResType resType;
  OpCode predOp;
  BOOL predResult;
};

typedef struct {
  ExprResType resType;
  int rangeNo;
} Ident;

typedef union {
  BOOL b;
  int i;
  float f;
  char *s;
} StackElem;

#define OP_CODE_NO(size) (((size) + sizeof(OpCode) - 1) / sizeof(OpCode))

static StackElem *Stack = NULL;
static int StackSize = 0;

static ExprP CurrExprP;
static ExprSym Sym;
static char *ErrorMsg;

static void
getSym(void)
{
  Sym = ExprScanGetSym();
}

static void
setError(char *errMsg)
{
  if (ErrorMsg == NULL)
    ErrorMsg = errMsg;
}

static void
checkCodeSpace(void)
{
  if (CurrExprP->pcEnd > CurrExprP->codeSize) {
    CurrExprP->codeSize *= 2;
    CurrExprP->code = realloc(CurrExprP->code,
        CurrExprP->codeSize * sizeof(OpCode));
  }
}

static void
addOpCode(OpCode oc)
{
  CurrExprP->pcEnd++;
  checkCodeSpace();
  CurrExprP->code[CurrExprP->pcEnd - 1] = oc;
}

static void
addBytes(void *p, int size)
{
  int oldLen = CurrExprP->pcEnd;

  CurrExprP->pcEnd += OP_CODE_NO(size);
  checkCodeSpace();
  (void) memcpy(CurrExprP->code + oldLen, p, size);
}

static void
typeCast(Ident *id1P, Ident *id2P)
{
  if (id1P->resType == id2P->resType)
    return;
  
  if (id1P->resType == ER_INT && id2P->resType == ER_FLOAT) {
    addOpCode(OC_CAST_1);
    id1P->resType = ER_FLOAT;
  } else if (id1P->resType == ER_FLOAT && id2P->resType == ER_INT) {
    addOpCode(OC_CAST);
    id2P->resType = ER_FLOAT;
  } else {
    setError("different types for binary operation");
  }
}

static void
incStack(void)
{
  CurrExprP->sp++;
  if (CurrExprP->sp > CurrExprP->maxStackSize)
    CurrExprP->maxStackSize = CurrExprP->sp;
}

static void
decStack(void)
{
  CurrExprP->sp--;
}

static void parseExpr(Ident *);

static void
parseQuantExpr(Ident *idP)
{
  DataEntityType oldType;
  OpCode ocQuant, oc;

  if (CurrExprP->entType == DE_NONE)
    setError("quantor not allowed");

  if (Sym == ES_EXISTS)
    ocQuant = OC_EXISTS;
  else
    ocQuant = OC_ALL;

  getSym();
  if (Sym == ES_LPAR)
    getSym();
  else
    setError("( expected");
  
  oldType = CurrExprP->entType;

  if (Sym == ES_SPEC) {
    oc = OC_ENT_SPEC;
    CurrExprP->entType = DE_SPEC;
  } else if (Sym == ES_PEAK) {
    oc = OC_ENT_PEAK;
    CurrExprP->entType = DE_PEAK;
  } else if (Sym == ES_PRIM) {
    oc = OC_ENT_PRIM;
    CurrExprP->entType = DE_PRIM;
  } else {
    oc = OC_NOP;
  }

  if (oc != OC_NOP) {
    addOpCode(oc);
    getSym();
    if (Sym == ES_COLON)
      getSym();
    else
      setError(": expected");
  }

  addOpCode(ocQuant);

  parseExpr(idP);
  CurrExprP->entType = oldType;
  addOpCode(OC_END);
  if (idP->resType != ER_BOOL)
    setError("boolean expression expected");

  if (Sym == ES_RPAR)
    getSym();
  else
    setError(") expected");
}

static DataEntityType
parseQual(void)
{
  OpCode oc;
  DataEntityType entType;

  if (Sym == ES_SPEC) {
    oc = OC_ENT_SPEC;
    entType = DE_SPEC;
  } else if (Sym == ES_PEAK) {
    oc = OC_ENT_PEAK;
    entType = DE_PEAK;
    if (CurrExprP->entType != DE_PEAK)
      setError("peak only valid for peak");
  } else if (Sym == ES_PRIM) {
    oc = OC_ENT_PRIM;
    entType = DE_PRIM;
    if (CurrExprP->entType != DE_PRIM)
      setError("prim only valid for prim");
  } else {
    oc = OC_NOP;
    entType = CurrExprP->entType;
  }
  
  if (oc != OC_NOP) {
    addOpCode(oc);
    getSym();
    if (Sym == ES_DOT)
      getSym();
    else
      setError(". expected");
  }

  return entType;
}

static void
parseDesignator(Ident *idP)
{
  DataEntityType entType;
  PropRefP refP;

  entType = parseQual();

  if (Sym == ES_NUMBER) {
    idP->resType = ER_INT;
    addOpCode(OC_GET_NUMBER);
    if (entType != DE_SPEC && entType != DE_PEAK && entType != DE_PRIM)
      setError("number only defined for spec, peak and prim");
  } else if (Sym == ES_NAME) {
    idP->resType = ER_STR;
    addOpCode(OC_GET_NAME);
    if (entType != DE_SPEC)
      setError("name only defined for spec");
  } else if (Sym == ES_AMP) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_AMP);
    if (entType != DE_PEAK)
      setError("amp only defined for peak");
  } else if (Sym == ES_SYMM) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_SYMM);
    if (entType != DE_PEAK)
      setError("symm only defined for peak");
  } else if (Sym == ES_UNIF) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_UNIF);
    if (entType != DE_PEAK)
      setError("unif only defined for peak");
  } else if (Sym == ES_QUAL) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_QUAL);
    if (entType != DE_PEAK)
      setError("qual only defined for peak");
  } else if (Sym == ES_SHIFT0) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_SHIFT0);
    if (entType != DE_PEAK)
      setError("shift0 only defined for peak");
  } else if (Sym == ES_SHIFT1) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_SHIFT1);
    if (entType != DE_PEAK)
      setError("shift1 only defined for peak");
  } else if (Sym == ES_SHIFT2) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_SHIFT2);
    if (entType != DE_PEAK)
      setError("shift2 only defined for peak");
  } else if (Sym == ES_SHIFT3) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_SHIFT3);
    if (entType != DE_PEAK)
      setError("shift3 only defined for peak");
  } else if (Sym == ES_SHIFT4) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_SHIFT4);
    if (entType != DE_PEAK)
      setError("shift4 only defined for peak");
  } else if (Sym == ES_WIDTH0) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_WIDTH0);
    if (entType != DE_PEAK)
      setError("width0 only defined for peak");
  } else if (Sym == ES_WIDTH1) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_WIDTH1);
    if (entType != DE_PEAK)
      setError("width1 only defined for peak");
  } else if (Sym == ES_WIDTH2) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_WIDTH2);
    if (entType != DE_PEAK)
      setError("width2 only defined for peak");
  } else if (Sym == ES_WIDTH3) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_WIDTH3);
    if (entType != DE_PEAK)
      setError("width3 only defined for peak");
  } else if (Sym == ES_WIDTH4) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_WIDTH4);
    if (entType != DE_PEAK)
      setError("width4 only defined for peak");
  } else if (Sym == ES_ATTR) {
    idP->resType = ER_INT;
    addOpCode(OC_GET_ATTR);
    if (entType != DE_SPEC && entType != DE_PEAK && entType != DE_PRIM)
      setError("attr only defined for spec, peak and prim");
  } else if (Sym == ES_IDENT) {
    idP->resType = ER_BOOL;
    addOpCode(OC_GET_PROP);
    refP = PropGetRef(ExprScanGetStr(), FALSE);
    if (refP == NULL)
      setError("unknown property");
    else
      addBytes(&refP, sizeof(refP));
    if (entType == DE_NONE)
      setError("no access to property");
  } else {
    setError("identifier expected");
  }
  
  getSym();
}

static void
parsePrimary(Ident *idP)
{
  int i;
  float f;
  char *s;

  if (Sym == ES_INT) {
    idP->resType = ER_INT;
    addOpCode(OC_GET_INT);
    i = ExprScanGetInt();
    addBytes(&i, sizeof(i));
    getSym();
  } else if (Sym == ES_FLOAT) {
    idP->resType = ER_FLOAT;
    addOpCode(OC_GET_FLOAT);
    f = ExprScanGetFloat();
    addBytes(&f, sizeof(f));
    getSym();
  } else if (Sym == ES_STR) {
    idP->resType = ER_STR;
    addOpCode(OC_GET_STR);
    s = ExprScanGetStr();
    addBytes(s, strlen(s) + 1);
    getSym();
  } else if (Sym == ES_LPAR) {
    getSym();
    parseExpr(idP);
    if (Sym == ES_RPAR)
      getSym();
    else
      setError(") expected");
  } else if (Sym == ES_EXISTS || Sym == ES_ALL) {
    parseQuantExpr(idP);
  } else {
    parseDesignator(idP);
  }

  incStack();
}

static void
parseFactor(Ident *idP)
{
  ExprSym opSym;

  opSym = Sym;
  if (opSym == ES_MINUS || opSym == ES_SQRT ||
      opSym == ES_LOG || opSym == ES_NOT)
    getSym();

  parsePrimary(idP);

  if (opSym == ES_MINUS) {
    if (idP->resType == ER_INT)
      addOpCode(OC_NEG_INT);
    else if (idP->resType == ER_FLOAT)
      addOpCode(OC_NEG_FLOAT);
    else
      setError("int or float expected for -");
  } else if (opSym == ES_SQRT) {
    if (idP->resType == ER_INT) {
      addOpCode(OC_CAST);
      idP->resType = ER_FLOAT;
    }
    if (idP->resType == ER_FLOAT)
      addOpCode(OC_SQRT);
    else
      setError("int of float expected for sqrt");
  } else if (opSym == ES_LOG) {
    if (idP->resType == ER_INT) {
      addOpCode(OC_CAST);
      idP->resType = ER_FLOAT;
    }
    if (idP->resType == ER_FLOAT)
      addOpCode(OC_LOG);
    else
      setError("int of float expected for log");
  } else if (opSym == ES_NOT) {
    if (idP->resType == ER_BOOL)
      addOpCode(OC_NOT);
    else
      setError("boolean expected for !");
  }
}

static void
parseTerm(Ident *idP)
{
  ExprSym opSym;
  Ident id2;

  parseFactor(idP);
  while (Sym == ES_MUL || Sym == ES_DIV || Sym == ES_REM) {
    opSym = Sym;
    getSym();
    parseFactor(&id2);
    typeCast(idP, &id2);
    if (opSym == ES_MUL) {
      if (idP->resType == ER_INT)
	addOpCode(OC_MUL_INT);
      else if (idP->resType == ER_FLOAT)
	addOpCode(OC_MUL_FLOAT);
      else
	setError("int or float expected for *");
    } else if (opSym == ES_DIV) {
      if (idP->resType == ER_INT)
	addOpCode(OC_DIV_INT);
      else if (idP->resType == ER_FLOAT)
	addOpCode(OC_DIV_FLOAT);
      else
	setError("int or float expected for /");
    } else {
      if (idP->resType == ER_INT)
	addOpCode(OC_REM);
      else
	setError("int expected for %");
    }
    decStack();
  }
}

static void
parseSimpleExpr(Ident *idP)
{
  ExprSym opSym;
  Ident id2;

  parseTerm(idP);
  while (Sym == ES_PLUS || Sym == ES_MINUS) {
    opSym = Sym;
    getSym();
    parseTerm(&id2);
    typeCast(idP, &id2);
    if (idP->resType == ER_INT)
      if (opSym == ES_PLUS)
	addOpCode(OC_PLUS_INT);
      else
	addOpCode(OC_MINUS_INT);
    else if (idP->resType == ER_FLOAT)
      if (opSym == ES_PLUS)
	addOpCode(OC_PLUS_FLOAT);
      else
	addOpCode(OC_MINUS_FLOAT);
    else
      setError("int or float expected for + or -");
    decStack();
  }
}

static void
parseRangeExpr(Ident *idP)
{
  Ident id2;

  parseSimpleExpr(idP);
  if (Sym == ES_RANGE) {
    getSym();
    parseSimpleExpr(&id2);
    typeCast(idP, &id2);

    if (idP->resType == ER_INT)
      idP->resType = ER_INT_SET;
    else if (idP->resType == ER_FLOAT)
      idP->resType = ER_FLOAT_SET;
    else
      setError("int or float expected for ..");
  }
}

static void
parseSetExpr(Ident *idP)
{
  Ident id2;

  parseRangeExpr(idP);
  if (Sym == ES_COMMA) {
    if (idP->resType == ER_INT) {
      addOpCode(OC_DUP);
      incStack();
      idP->resType = ER_INT_SET;
    } else if (idP->resType == ER_FLOAT) {
      addOpCode(OC_DUP);
      incStack();
      idP->resType = ER_FLOAT_SET;
    } else if (idP->resType != ER_INT_SET && idP->resType != ER_FLOAT_SET) {
      setError("int or float expected for ,");
    }
  }


  idP->rangeNo = 1;

  while (Sym == ES_COMMA) {
    idP->rangeNo++;

    getSym();
    parseRangeExpr(&id2);
    if (idP->resType == ER_INT_SET) {
      if (id2.resType == ER_INT) {
        addOpCode(OC_DUP);
        incStack();
      } else if (idP->resType != ER_INT_SET) {
        setError("int expected for ,");
      }
    } else {
      if (id2.resType == ER_INT) {
        addOpCode(OC_CAST);
        addOpCode(OC_DUP);
        incStack();
      } else if (id2.resType == ER_FLOAT) {
        addOpCode(OC_DUP);
        incStack();
      } else if (id2.resType == ER_INT_SET) {
        addOpCode(OC_CAST_1);
        addOpCode(OC_CAST);
      } else if (id2.resType != ER_INT_SET) {
        setError("int or float expected for ,");
      }
    }
  }
}

static void
parseRelExpr(Ident *idP)
{
  ExprSym opSym;
  Ident id2;
  int i;

  parseSimpleExpr(idP);
  if (Sym == ES_EQU || Sym == ES_NEQ) {
    opSym = Sym;
    getSym();
    parseSetExpr(&id2);
    if (id2.resType == ER_INT_SET) {
      if (idP->resType == ER_INT) {
        if (opSym == ES_EQU)
          addOpCode(OC_INSET_INT);
        else
          addOpCode(OC_OUTSET_INT);
        addBytes(&id2.rangeNo, sizeof(id2.rangeNo));
      } else {
        setError("int expected for int set");
      }

      for (i = 0; i < 2 * id2.rangeNo; i++)
        decStack();
    } else if (id2.resType == ER_FLOAT_SET) {
      if (idP->resType == ER_FLOAT) {
        if (opSym == ES_EQU)
          addOpCode(OC_INSET_FLOAT);
        else
          addOpCode(OC_OUTSET_FLOAT);
        addBytes(&id2.rangeNo, sizeof(id2.rangeNo));
      } else {
        setError("float expected for float set");
      }

      for (i = 0; i < 2 * id2.rangeNo; i++)
        decStack();
    } else {
      typeCast(idP, &id2);


      if (idP->resType == ER_INT) {
        if (opSym == ES_EQU)
          addOpCode(OC_EQU_INT);
        else
          addOpCode(OC_NEQ_INT);
      } else if (idP->resType == ER_FLOAT) {
        if (opSym == ES_EQU)
          addOpCode(OC_EQU_FLOAT);
        else
          addOpCode(OC_NEQ_FLOAT);
      } else if (idP->resType == ER_STR) {
        if (opSym == ES_EQU)
          addOpCode(OC_EQU_STR);
        else
          addOpCode(OC_NEQ_STR);
      } else {
        if (opSym == ES_EQU)
          setError("int, float or string expected for =");
        else
          setError("int, float or string expected for !=");
      }

      decStack();
    }
    idP->resType = ER_BOOL;
  } else if (Sym == ES_LEQ) {
    getSym();
    parseSimpleExpr(&id2);
    typeCast(idP, &id2);
    if (idP->resType == ER_INT)
      addOpCode(OC_LEQ_INT);
    else if (idP->resType == ER_FLOAT)
      addOpCode(OC_LEQ_FLOAT);
    else
      setError("int or float expected for <=");
    idP->resType = ER_BOOL;
    decStack();
  } else if (Sym == ES_LSS) {
    getSym();
    parseSimpleExpr(&id2);
    typeCast(idP, &id2);
    if (idP->resType == ER_INT)
      addOpCode(OC_LSS_INT);
    else if (idP->resType == ER_FLOAT)
      addOpCode(OC_LSS_FLOAT);
    else
      setError("int or float expected for <");
    idP->resType = ER_BOOL;
    decStack();
  } else if (Sym == ES_GEQ) {
    getSym();
    parseSimpleExpr(&id2);
    typeCast(idP, &id2);
    if (idP->resType == ER_INT)
      addOpCode(OC_GEQ_INT);
    else if (idP->resType == ER_FLOAT)
      addOpCode(OC_GEQ_FLOAT);
    else
      setError("int or float expected for >=");
    idP->resType = ER_BOOL;
    decStack();
  } else if (Sym == ES_GTR) {
    getSym();
    parseSimpleExpr(&id2);
    typeCast(idP, &id2);
    if (idP->resType == ER_INT)
      addOpCode(OC_GTR_INT);
    else if (idP->resType == ER_FLOAT)
      addOpCode(OC_GTR_FLOAT);
    else
      setError("int or float expected for >");
    idP->resType = ER_BOOL;
    decStack();
  }
}

static void
parseAndExpr(Ident *idP)
{
  Ident id2;

  parseRelExpr(idP);
  while (Sym == ES_AND) {
    getSym();
    parseRelExpr(&id2);
    if (idP->resType != ER_BOOL || id2.resType != ER_BOOL)
      setError("bool expected for &");
    addOpCode(OC_AND);
    decStack();
  }
}

static void
parseExpr(Ident *idP)
{
  Ident id2;

  parseAndExpr(idP);
  while (Sym == ES_OR) {
    getSym();
    parseAndExpr(&id2);
    if (idP->resType != ER_BOOL || id2.resType != ER_BOOL)
      setError("bool expected for |");
    addOpCode(OC_OR);
    decStack();
  }
}

ExprP
ExprCompile(DataEntityType entType, char *exprStr, ExprResType *resTypeP)
{
  Ident id;

  CurrExprP = malloc(sizeof(*CurrExprP));
  CurrExprP->sp = 0;
  CurrExprP->maxStackSize = 0;
  CurrExprP->codeSize = INIT_CODE_SIZE;
  CurrExprP->code = malloc(CurrExprP->codeSize * sizeof(OpCode));
  CurrExprP->pcEnd = 0;
  CurrExprP->entType = entType;

  ErrorMsg = NULL;

  ExprScanStart(exprStr);
  getSym();
  parseExpr(&id);
  addOpCode(OC_END);
  if (Sym != ES_END)
    setError("end expected");

  if (ErrorMsg != NULL) {
    ExprFree(CurrExprP);
    return NULL;
  }

  CurrExprP->resType = id.resType;
  CurrExprP->codeSize = CurrExprP->pcEnd;
  CurrExprP->code = realloc(CurrExprP->code,
      CurrExprP->codeSize * sizeof(OpCode));

  *resTypeP = id.resType;
  if (Stack == NULL) {
    Stack = malloc(CurrExprP->maxStackSize * sizeof(StackElem));
    StackSize = CurrExprP->maxStackSize;
  } else if (CurrExprP->maxStackSize > StackSize) {
    Stack = realloc(Stack, CurrExprP->maxStackSize * sizeof(StackElem));
    StackSize = CurrExprP->maxStackSize;
  }

  return CurrExprP;
}

char *
ExprGetErrorMsg(void)
{
  return ErrorMsg;
}

static void evalOneExpr(ExprP);

static void
predSpec(DhSpecP specP, void *clientData)
{
  ExprP exprP = clientData;

  if (exprP->predOp == OC_EXISTS) {
    if (exprP->predResult == TRUE)
      return;
  } else {
    if (exprP->predResult == FALSE)
      return;
  }

  exprP->specP = specP;
  evalOneExpr(exprP);
  exprP->predResult = Stack[exprP->sp + 1].b;
}

static void
evalOneExpr(ExprP exprP)
{
  OpCode *code = exprP->code;
  int pc = exprP->pcStart;
  int sp = exprP->sp;
  DataEntityType entType = exprP->entType;
  int iVal;
  float fVal;
  int rangeNo, i;
  BOOL setRes;
  PropRefP refP;
  AppPeakP appPeakP;
  struct ExprS exprS;
  AttrP attrP;

  while (code[pc] != OC_END) {
    switch (code[pc++]) {
      case OC_PLUS_INT:
	sp--;
	Stack[sp].i = Stack[sp].i + Stack[sp + 1].i;
	break;
      case OC_MINUS_INT:
	sp--;
	Stack[sp].i = Stack[sp].i - Stack[sp + 1].i;
	break;
      case OC_MUL_INT:
	sp--;
	Stack[sp].i = Stack[sp].i * Stack[sp + 1].i;
	break;
      case OC_DIV_INT:
	sp--;
	Stack[sp].i = Stack[sp].i / Stack[sp + 1].i;
	break;
      case OC_REM:
	sp--;
	Stack[sp].i = Stack[sp].i % Stack[sp + 1].i;
	break;
      case OC_NEG_INT:
	Stack[sp].i = - Stack[sp].i;
	break;
      case OC_PLUS_FLOAT:
	sp--;
	Stack[sp].f = Stack[sp].f + Stack[sp + 1].f;
	break;
      case OC_MINUS_FLOAT:
	sp--;
	Stack[sp].f = Stack[sp].f - Stack[sp + 1].f;
	break;
      case OC_MUL_FLOAT:
	sp--;
	Stack[sp].f = Stack[sp].f * Stack[sp + 1].f;
	break;
      case OC_DIV_FLOAT:
	sp--;
	Stack[sp].f = Stack[sp].f / Stack[sp + 1].f;
	break;
      case OC_NEG_FLOAT:
	Stack[sp].f = - Stack[sp].f;
	break;
      case OC_SQRT:
	Stack[sp].f = sqrt(Stack[sp].f);
	break;
      case OC_LOG:
	Stack[sp].f = log(Stack[sp].f);
	break;
      case OC_DUP:
        sp++;
        Stack[sp] = Stack[sp - 1];
        break;
      case OC_CAST:
	Stack[sp].f = (float) Stack[sp].i;
	break;
      case OC_CAST_1:
	Stack[sp - 1].f = (float) Stack[sp - 1].i;
	break;
      case OC_EQU_INT:
	sp--;
	Stack[sp].b = (Stack[sp].i == Stack[sp + 1].i);
	break;
      case OC_NEQ_INT:
	sp--;
	Stack[sp].b = (Stack[sp].i != Stack[sp + 1].i);
	break;
      case OC_LEQ_INT:
	sp--;
	Stack[sp].b = (Stack[sp].i <= Stack[sp + 1].i);
	break;
      case OC_LSS_INT:
	sp--;
	Stack[sp].b = (Stack[sp].i < Stack[sp + 1].i);
	break;
      case OC_GEQ_INT:
	sp--;
	Stack[sp].b = (Stack[sp].i >= Stack[sp + 1].i);
	break;
      case OC_GTR_INT:
	sp--;
	Stack[sp].b = (Stack[sp].i > Stack[sp + 1].i);
	break;
      case OC_INSET_INT:
        rangeNo = * (int *) (code + pc);
        pc += OP_CODE_NO(sizeof(int));
        iVal = Stack[sp - 2 * rangeNo].i;
        setRes = FALSE;
        for (i = 0; i < rangeNo; i++) {
          if (iVal >= Stack[sp - 1].i && iVal <= Stack[sp].i)
            setRes = TRUE;
          sp -= 2;
        }
        Stack[sp].b = setRes;
        break;
      case OC_OUTSET_INT:
        rangeNo = * (int *) (code + pc);
        pc += OP_CODE_NO(sizeof(int));
        iVal = Stack[sp - 2 * rangeNo].i;
        setRes = TRUE;
        for (i = 0; i < rangeNo; i++) {
          if (iVal >= Stack[sp - 1].i && iVal <= Stack[sp].i)
            setRes = FALSE;
          sp -= 2;
        }
        Stack[sp].b = setRes;
        break;
      case OC_EQU_FLOAT:
	sp--;
	Stack[sp].b = (Stack[sp].f == Stack[sp + 1].f);
	break;
      case OC_NEQ_FLOAT:
	sp--;
	Stack[sp].b = (Stack[sp].f != Stack[sp + 1].f);
	break;
      case OC_LEQ_FLOAT:
	sp--;
	Stack[sp].b = (Stack[sp].f <= Stack[sp + 1].f);
	break;
      case OC_LSS_FLOAT:
	sp--;
	Stack[sp].b = (Stack[sp].f < Stack[sp + 1].f);
	break;
      case OC_GEQ_FLOAT:
	sp--;
	Stack[sp].b = (Stack[sp].f >= Stack[sp + 1].f);
	break;
      case OC_GTR_FLOAT:
	sp--;
	Stack[sp].b = (Stack[sp].f > Stack[sp + 1].f);
	break;
      case OC_INSET_FLOAT:
        rangeNo = * (int *) (code + pc);
        pc += OP_CODE_NO(sizeof(int));
        fVal = Stack[sp - 2 * rangeNo].f;
        setRes = FALSE;
        for (i = 0; i < rangeNo; i++) {
          if (fVal >= Stack[sp - 1].f && fVal <= Stack[sp].f)
            setRes = TRUE;
          sp -= 2;
        }
        Stack[sp].b = setRes;
        break;
      case OC_OUTSET_FLOAT:
        rangeNo = * (int *) (code + pc);
        pc += OP_CODE_NO(sizeof(int));
        fVal = Stack[sp - 2 * rangeNo].f;
        setRes = TRUE;
        for (i = 0; i < rangeNo; i++) {
          if (fVal >= Stack[sp - 1].f && fVal <= Stack[sp].f)
            setRes = FALSE;
          sp -= 2;
        }
        Stack[sp].b = setRes;
        break;
      case OC_EQU_STR:
	sp--;
	Stack[sp].b = StrMatch(Stack[sp].s, Stack[sp + 1].s);
	break;
      case OC_NEQ_STR:
	sp--;
	Stack[sp].b = ! StrMatch(Stack[sp].s, Stack[sp + 1].s);
	break;
      case OC_OR:
	sp--;
	Stack[sp].b = Stack[sp].b || Stack[sp + 1].b;
	break;
      case OC_AND:
	sp--;
	Stack[sp].b = Stack[sp].b && Stack[sp + 1].b;
	break;
      case OC_NOT:
	Stack[sp].b = ! Stack[sp].b;
	break;
      case OC_GET_INT:
	sp++;
	Stack[sp].i = * (int *) (code + pc);
	pc += OP_CODE_NO(sizeof(int));
	break;
      case OC_GET_FLOAT:
	sp++;
	Stack[sp].f = * (float *) (code + pc);
	pc += OP_CODE_NO(sizeof(float));
	break;
      case OC_GET_STR:
	sp++;
	Stack[sp].s = (char *) (code + pc);
	pc += OP_CODE_NO(strlen(Stack[sp].s) + 1);
	break;
      case OC_GET_PROP:
	sp++;
	refP = * (PropRefP *) (code + pc);
	pc += OP_CODE_NO(sizeof(PropRefP));
	if (entType == DE_SPEC)
	  if (exprP->specP == NULL)
	    Stack[sp].b = FALSE;
	  else
	    Stack[sp].b = DhSpecGetProp(refP, exprP->specP);
	else if (entType == DE_PEAK)
	  Stack[sp].b = DhPeakGetProp(refP, exprP->peakP);
	else
	  Stack[sp].b = PrimGetProp(refP, exprP->primP);
	entType = exprP->entType;
	break;
      case OC_EXISTS:
      case OC_ALL:
	exprS = *exprP;
	exprS.pcStart = pc;
	exprS.sp = sp;
	exprS.entType = entType;
	exprS.predOp = code[pc - 1];
	if (exprS.predOp == OC_EXISTS)
	  exprS.predResult = FALSE;
	else
	  exprS.predResult = TRUE;
	refP = PropGetRef(PROP_ALL, FALSE);
	DhApplySpec(refP, predSpec, &exprS);
	pc = exprS.pcEnd + 1;
	break;
      case OC_ENT_SPEC:
	entType = DE_SPEC;
	break;
      case OC_ENT_PEAK:
	entType = DE_PEAK;
	break;
      case OC_ENT_PRIM:
	entType = DE_PRIM;
	break;
      case OC_GET_NUMBER:
	sp++;
	if (entType == DE_SPEC)
	  if (exprP->specP == NULL)
	    Stack[sp].i = 0;
	  else
	    Stack[sp].i = DhSpecGetNumber(exprP->specP);
	else if (entType == DE_PEAK)
	  Stack[sp].i = DhPeakGetNumber(exprP->peakP);
	else
	  Stack[sp].i = PrimGetNumber(exprP->primP);
	entType = exprP->entType;
	break;
      case OC_GET_NAME:
	sp++;
	if (exprP->specP == NULL)
	  Stack[sp].s = "";
	else
	  Stack[sp].s = DStrToStr(DhSpecGetName(exprP->specP));
	entType = exprP->entType;
	break;
      case OC_GET_AMP:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->amplitude;
	entType = exprP->entType;
	break;
      case OC_GET_SYMM:
	sp++;
	appPeakP = DhPeakGetApp(exprP->peakP);
	Stack[sp].f = appPeakP->symmetryErrorA[0];
	for (i = 1; i < appPeakP->dimensionNo; i++)
	  if (appPeakP->symmetryErrorA[i] > Stack[sp].f)
	    Stack[sp].f = appPeakP->symmetryErrorA[i];
	entType = exprP->entType;
	break;
      case OC_GET_UNIF:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->uniformError;
	entType = exprP->entType;
	break;
      case OC_GET_QUAL:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->quality;
	entType = exprP->entType;
	break;
      case OC_GET_SHIFT0:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->positionA[0];
	entType = exprP->entType;
	break;
      case OC_GET_SHIFT1:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->positionA[1];
	entType = exprP->entType;
	break;
      case OC_GET_SHIFT2:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->positionA[2];
	entType = exprP->entType;
	break;
      case OC_GET_SHIFT3:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->positionA[3];
	entType = exprP->entType;
	break;
      case OC_GET_SHIFT4:
	sp++;
	Stack[sp].f = DhPeakGetApp(exprP->peakP)->positionA[4];
	entType = exprP->entType;
	break;
      case OC_GET_WIDTH0:
	sp++;
	appPeakP = DhPeakGetApp(exprP->peakP);
	if (appPeakP->dimensionNo > 0)
	  Stack[sp].f = appPeakP->shapePA[0]->width;
	else
	  Stack[sp].f = 0.0;
	entType = exprP->entType;
	break;
      case OC_GET_WIDTH1:
	sp++;
	appPeakP = DhPeakGetApp(exprP->peakP);
	if (appPeakP->dimensionNo > 1)
	  Stack[sp].f = appPeakP->shapePA[1]->width;
	else
	  Stack[sp].f = 0.0;
	entType = exprP->entType;
	break;
      case OC_GET_WIDTH2:
	sp++;
	appPeakP = DhPeakGetApp(exprP->peakP);
	if (appPeakP->dimensionNo > 2)
	  Stack[sp].f = appPeakP->shapePA[2]->width;
	else
	  Stack[sp].f = 0.0;
	entType = exprP->entType;
	break;
      case OC_GET_WIDTH3:
	sp++;
	appPeakP = DhPeakGetApp(exprP->peakP);
	if (appPeakP->dimensionNo > 3)
	  Stack[sp].f = appPeakP->shapePA[3]->width;
	else
	  Stack[sp].f = 0.0;
	entType = exprP->entType;
	break;
      case OC_GET_WIDTH4:
	sp++;
	appPeakP = DhPeakGetApp(exprP->peakP);
	if (appPeakP->dimensionNo > 4)
	  Stack[sp].f = appPeakP->shapePA[4]->width;
	else
	  Stack[sp].f = 0.0;
	entType = exprP->entType;
	break;
      case OC_GET_ATTR:
        sp++;
        if (entType == DE_SPEC)
          attrP = DhSpecGetAttr(exprP->specP);
        else if (entType == DE_PEAK)
          attrP = DhPeakGetAttr(exprP->peakP);
        else
          attrP = PrimGetAttr(exprP->primP);
        Stack[sp].i = AttrGetIndex(attrP);
        entType = exprP->entType;
        break;
    }  /* switch */
  }  /* for */

  exprP->pcEnd = pc;
}

void
ExprEval(void *entP, ExprP exprP, ExprRes *resultP)
{
  if (exprP->entType == DE_SPEC) {
    exprP->specP = entP;
  } else if (exprP->entType == DE_PEAK) {
    exprP->peakP = entP;
    exprP->specP = DhPeakGetSpec(exprP->peakP);
  } else if (exprP->entType == DE_PRIM) {
    exprP->primP = entP;
  }

  exprP->pcStart = 0;
  exprP->sp = -1;
  evalOneExpr(exprP);

  resultP->resType = exprP->resType;

  if (exprP->resType == ER_BOOL)
    resultP->u.boolVal = Stack[0].b;
  else if (exprP->resType == ER_INT)
    resultP->u.intVal = Stack[0].i;
  else if (exprP->resType == ER_FLOAT)
    resultP->u.floatVal = Stack[0].f;
}

void
ExprFree(ExprP exprP)
{
  free(exprP->code);
  free(exprP);
}
