/*
************************************************************************
*
*   ExTexture.c - Texture* commands
*
*   Copyright (c) 1994-95
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/13
*   Pathname of SCCS file     : /sgiext/autopsy/src/cmdattr/SCCS/s.ExTexture.c
*   SCCS identification       : 1.1
*
************************************************************************
*/

#include <cmd_attr.h>

#include <string.h>

#include <arg.h>
#include <data_hand.h>
#include <prim_hand.h>
#include <attr_struc.h>
#include <attr_mng.h>
#include <graph_draw.h>

#define FLOAT_AS_IS -1.0
#define TEX_AS_IS "as is"

typedef struct {
  int entNo;
  char name[SG_TEXTURE_NAME_SIZE];
  ExprP texScExprP, bumpExprP, bumpScExprP;
  float texSc, bump, bumpSc;
} TexDescr;

static void
getTex(AttrP attrP, TexDescr *texP)
{
  if (texP->entNo == 0) {
    (void) strcpy(texP->name, attrP->texture);
    texP->texSc = attrP->textureScale;
    texP->bump = attrP->bumpDepth;
    texP->bumpSc = attrP->bumpScale;
  } else {
    if (strcmp(texP->name, attrP->texture) != 0)
      (void) strcpy(texP->name, TEX_AS_IS);
    if (texP->texSc != attrP->textureScale)
      texP->texSc = FLOAT_AS_IS;
    if (texP->bump != attrP->bumpDepth)
      texP->bump = FLOAT_AS_IS;
    if (texP->bumpSc != attrP->bumpScale)
      texP->bumpSc = FLOAT_AS_IS;
  }

  texP->entNo++;
}

static void
getTexSpec(DhSpecP specP, void *clientData)
{
  getTex(DhSpecGetAttr(specP), clientData);
}

static void
getTexPeak(DhPeakP peakP, void *clientData)
{
  getTex(DhPeakGetAttr(peakP), clientData);
}

static void
getTexPrim(PrimObjP primP, void *clientData)
{
  getTex(PrimGetAttr(primP), clientData);
}

static void
evalTex(void *entP, TexDescr *texP)
{
  ExprRes exprRes;

  ExprEval(entP, texP->texScExprP, &exprRes);
  if (exprRes.resType == ER_INT)
    texP->texSc = (float) exprRes.u.intVal;
  else
    texP->texSc = exprRes.u.floatVal;

  ExprEval(entP, texP->bumpExprP, &exprRes);
  if (exprRes.resType == ER_INT)
    texP->bump = (float) exprRes.u.intVal;
  else
    texP->bump = exprRes.u.floatVal;

  ExprEval(entP, texP->bumpScExprP, &exprRes);
  if (exprRes.resType == ER_INT)
    texP->bumpSc = (float) exprRes.u.intVal;
  else
    texP->bumpSc = exprRes.u.floatVal;
}

static AttrP
changeTex(AttrP attrP, TexDescr *texP)
{
  struct AttrS attr;

  AttrCopy(&attr, attrP);
  AttrReturn(attrP);
  if (strcmp(texP->name, TEX_AS_IS) != 0)
    (void) strcpy(attr.texture, texP->name);
  if (texP->texSc > 0.0)
    attr.textureScale = texP->texSc;
  if (texP->bump >= 0.0)
    attr.bumpDepth = texP->bump;
  if (texP->bumpSc > 0.0)
    attr.bumpScale = texP->bumpSc;
  return AttrGet(&attr);
}

static void
texSpec(DhSpecP specP, void *clientData)
{
  evalTex(specP, clientData);
  DhSpecSetAttr(specP, changeTex(DhSpecGetAttr(specP), clientData));
}

static void
texPeak(DhPeakP peakP, void *clientData)
{
  evalTex(peakP, clientData);
  DhPeakSetAttr(peakP, changeTex(DhPeakGetAttr(peakP), clientData));
}

static void
texPrim(PrimObjP primP, void *clientData)
{
  evalTex(primP, clientData);
  PrimSetAttr(primP, changeTex(PrimGetAttr(primP), clientData));
}

#define ARG_NO 4

ErrCode
ExTexture(char *cmd)
{
  DataEntityType entType;
  ArgDescr arg[ARG_NO];
  ErrCode errCode;
  TexDescr texDescr;
  PropRefP refP;
  int i;

  if (strcmp(cmd, "TextureSpec") == 0)
    entType = DE_SPEC;
  else if (strcmp(cmd, "TexturePeak") == 0)
    entType = DE_PEAK;
  else
    entType = DE_PRIM;

  arg[0].type = AT_STR;
  arg[1].type = AT_DOUBLE;
  arg[2].type = AT_DOUBLE;
  arg[3].type = AT_DOUBLE;

  ArgInit(arg, ARG_NO);

  arg[0].prompt = "Name";
  arg[1].prompt = "Scale";
  arg[2].prompt = "Bumpiness";
  arg[3].prompt = "Bump Scale";

  arg[1].entType = entType;
  arg[2].entType = entType;
  arg[3].entType = entType;

  texDescr.entNo = 0;
  texDescr.name[0] = '\0';
  texDescr.texSc = FLOAT_AS_IS;
  texDescr.bump = FLOAT_AS_IS;
  texDescr.bumpSc = FLOAT_AS_IS;

  refP = PropGetRef(PROP_SELECTED, FALSE);

  switch (entType) {
    case DE_SPEC:
      DhApplySpec(refP, getTexSpec, &texDescr);
      break;
    case DE_PEAK:
      DhApplyPeak(refP, getTexPeak, &texDescr);
      break;
    case DE_PRIM:
      PrimApply(PT_ALL, refP, getTexPrim, &texDescr);
      break;
  }

  DStrAssignStr(arg[0].v.strVal, texDescr.name);
  arg[1].v.doubleVal = texDescr.texSc;
  arg[2].v.doubleVal = texDescr.bump;
  arg[3].v.doubleVal = texDescr.bumpSc;

  errCode = ArgGet(arg, ARG_NO);
  if (errCode != EC_OK) {
    ArgCleanup(arg, ARG_NO);
    return errCode;
  }

  DStrPart(arg[0].v.strVal, 0, SG_TEXTURE_NAME_SIZE - 2);
  (void) strcpy(texDescr.name, DStrToStr(arg[0].v.strVal));
  texDescr.texScExprP = arg[1].v.exprP;
  texDescr.bumpExprP = arg[2].v.exprP;
  texDescr.bumpScExprP = arg[3].v.exprP;

  switch (entType) {
    case DE_SPEC:
      DhApplySpec(refP, texSpec, &texDescr);
      break;
    case DE_PEAK:
      DhApplyPeak(refP, texPeak, &texDescr);
      break;
    case DE_PRIM:
      PrimApply(PT_ALL, refP, texPrim, &texDescr);
      break;
  }

  ArgCleanup(arg, ARG_NO);

  GraphSpecChanged(PROP_SELECTED);
  GraphRedrawNeeded();

  return EC_OK;
}
