/*
************************************************************************
*
*   VRMLShape.c - VRML spheres and cones
*
*   Copyright (c) 1996
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/07/23
*   Pathname of SCCS file     : /sgiext/molmol/sg/src/vrml/SCCS/s.VRMLShape.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include "vrml_shape.h"

#include <math.h>

#include <mat_vec.h>
#include <sg_map.h>
#include <sg_plot.h>
#include "vrml_util.h"
#include "vrml_color.h"
#include "vrml_light.h"

static void
drawSphere(float x[3], float r, BOOL plotSep)
{
  if (plotSep)
    SgPlotOutputStr("TransformSeparator {\n");

  SgPlotOutputStr("Transform {\n");
  SgPlotOutputStr("translation ");
  VRMLPlotVect(x);
  SgPlotOutputStr("\n");
  SgPlotOutputStr("}\n");

  SgPlotOutputStr("Sphere {");
  SgPlotOutputStr("radius ");
  SgPlotOutputFloat(r);
  SgPlotOutputStr("}\n");

  if (plotSep)
    SgPlotOutputStr("}\n");
}

void
SgVRMLDrawSphere(float x[3], float r)
{
  float xm[3], v[3], vm[3];

  VRMLUpdateLight();
  SgPlotOutputStr("Separator {\n");
  VRMLPlotColor();

  SgMapModelPoint(xm, x);

  v[0] = r;
  v[1] = 0.0;
  v[2] = 0.0;
  SgMapModelVect(vm, v);
  r = Vec3Abs(vm);

  drawSphere(xm, r, FALSE);

  SgPlotOutputStr("}\n");
}

void
SgVRMLDrawCone(float x0[3], float x1[3], float r0, float r1,
    SgConeEnd end0, SgConeEnd end1)
{
  float xm0[3], xm1[3], v[3], vm[3], x[3];
  float r, h;
  BOOL isCylinder;

  VRMLUpdateLight();
  SgPlotOutputStr("Separator {\n");
  VRMLPlotColor();

  /* only support for cylinders */
  r = 0.5 * (r0 + r1);

  SgMapModelPoint(xm0, x0);
  SgMapModelPoint(xm1, x1);

  v[0] = r;
  v[1] = 0.0;
  v[2] = 0.0;
  SgMapModelVect(vm, v);
  r = Vec3Abs(vm);

  Vec3Copy(vm, xm1);
  Vec3Sub(vm, xm0);
  h = Vec3Abs(vm);
  Vec3Scale(vm, 1.0 / h);

  SgPlotOutputStr("Transform {\n");
  SgPlotOutputStr("translation ");
  VRMLPlotVect(x0);
  SgPlotOutputStr("\n");
  SgPlotOutputStr("rotation ");
  SgPlotOutputFloat(- vm[1]);
  SgPlotOutputStr(" ");
  SgPlotOutputFloat(vm[0]);
  SgPlotOutputStr(" 0 ");
  SgPlotOutputFloat(atan2(vm[0] * vm[0] + vm[1] * vm[1], vm[2]));
  SgPlotOutputStr("\n");
  SgPlotOutputStr("}\n");

  SgPlotOutputStr("Cylinder {\n");
  if (end0 == SG_CONE_CAP_FLAT && end1 == SG_CONE_CAP_FLAT)
    SgPlotOutputStr("parts ALL\n");
  else if (end0 == SG_CONE_CAP_FLAT)
    SgPlotOutputStr("parts (SIDES | BOTTOM)\n");
  else if (end1 == SG_CONE_CAP_FLAT)
    SgPlotOutputStr("parts (SIDES | TOP)\n");
  SgPlotOutputStr("radius ");
  SgPlotOutputFloat(r);
  SgPlotOutputStr("\n");
  SgPlotOutputStr("height ");
  SgPlotOutputFloat(h);
  SgPlotOutputStr("\n");
  SgPlotOutputStr("\n");
  SgPlotOutputStr("}\n");

  if (end0 == SG_CONE_CAP_ROUND)
    drawSphere(xm0, r0, TRUE);

  if (end1 == SG_CONE_CAP_ROUND)
    drawSphere(xm1, r1, TRUE);
}
