#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <values.h>

#include <app/reg_symm.h>
#include <app/reg_shape.h>

#define START0 200
#define START1 100
#define SIZE0 30
#define SIZE1 20
#define AMP 1000.0
#define C0 16.7
#define C1 8.3
#define W0 6.2
#define W1 4.3

static void
fillTestReg(AppRegionDataP regDataP)
{
  float *v;
  int i0, i1;
  float z0, z1;

  regDataP->dimensionNo = 2;
  regDataP->dataType = ADT_FLT32;

  regDataP->rangeA[0][0] = START0;
  regDataP->rangeA[0][1] = START0 + SIZE0 - 1;
  regDataP->rangeA[1][0] = START1;
  regDataP->rangeA[1][1] = START1 + SIZE1 - 1;

  regDataP->subRangeA[0][0] = 1;
  regDataP->subRangeA[0][1] = SIZE0 - 2;
  regDataP->subRangeA[1][0] = 1;
  regDataP->subRangeA[1][1] = SIZE1 - 2;

  regDataP->dataA = malloc(SIZE0 * SIZE1 * sizeof(float));
  regDataP->validA = malloc(SIZE0 * SIZE1 * sizeof(*regDataP->validA));

  v = regDataP->dataA;

  for (i0 = 0; i0 < SIZE0; i0++) {
    z0 = 2.0 * (i0 - C0) / W0;

    for (i1 = 0; i1 < SIZE1; i1++) {
      z1 = 2.0 * (i1 - C1) / W1;

      if (i0 == 0 || i0 == SIZE0 - 1 || i1 == 0 || i1 == SIZE1 - 1) {
	regDataP->validA[i1 * SIZE0 + i0] = FALSE;
      } else {
	v[i1 * SIZE0 + i0] = AMP *
	    exp(- M_LN2 * z0 * z0) *
	    exp(- M_LN2 * z1 * z1);

	regDataP->validA[i1 * SIZE0 + i0] = TRUE;
      }
    }
  }

  regDataP->maxAmplitude = AMP;

  regDataP->noiseMagnitudeMax = 1.0;
}

static void
freeReg(AppRegionDataP regDataP)
{
  free(regDataP->dataA);
  free(regDataP->validA);
}

static void
testSymm(void)
{
  struct AppRegionDataS regData1S;
  struct AppRegionDataS regData2S;
  int maxPosA[2];
  float maxSplitA[2];
  float posA[2], errA[2];
  BOOL dirSelA[2];
  float *v1, *v2, maxDiff;
  int i;

  (void) printf("\nCalcSymmetry\n");

  fillTestReg(&regData1S);
  maxPosA[0] = START0 + (int) (C0 + 0.5);
  maxPosA[1] = START1 + (int) (C1 + 0.5);
  maxSplitA[0] = 1.0;
  maxSplitA[1] = 1.0;
  AppBestSymmetry(&regData1S, maxPosA,
      maxSplitA, 0.0, regData1S.noiseMagnitudeMax, posA);
  AppCalcSymmetry(&regData1S, posA, errA);

  (void) printf("pos: %8.5f %8.5f\n", posA[0], posA[1]);
  (void) printf("err: %5.2f %5.2f\n", errA[0], errA[1]);

  dirSelA[0] = TRUE;
  dirSelA[1] = TRUE;
  AppSymmetrize(&regData1S, posA, dirSelA);

  fillTestReg(&regData2S);

  v1 = regData1S.dataA;
  v2 = regData2S.dataA;
  maxDiff = 0.0;
  for (i = 0; i < SIZE0 * SIZE1; i++) {
    if (! regData2S.validA[i])
      continue;

    if (v1[i] - v2[i] > maxDiff)
      maxDiff = v1[i] - v2[i];
    else if (v2[i] - v1[i] > maxDiff)
      maxDiff = v2[i] - v1[i];
  }

  (void) printf("maxDiff: %5.2f\n", maxDiff);

  freeReg(&regData1S);
  freeReg(&regData2S);
}

static void
printShape(AppShapeP shapeP)
{
  int i;

  AppShapeCalcPar(shapeP);

  (void) printf("w: %5.2f, lor: %5.2f\n",
      shapeP->width, shapeP->lorentzFraction);
  
  for (i = 0; i < shapeP->valueNo; i++)
    (void) printf("%5.2f ", shapeP->valueA[i]);
  (void) printf("\n");
}

static void
testShape(void)
{
  struct AppRegionDataS regDataS;
  struct AppShapeS shapeA[2];
  float amp, err;

  (void) printf("\nCalcShapes\n");

  fillTestReg(&regDataS);
  AppCalcShapes(&regDataS, NULL, shapeA, &amp, &err);
  freeReg(&regDataS);

  printShape(&shapeA[0]);
  printShape(&shapeA[1]);
  (void) printf("amp: %5.2f   err: %5.2f\n", amp, err);

  AppShapeDestroy(&shapeA[0]);
  AppShapeDestroy(&shapeA[1]);
}

int
main(int argc, char *argv[])
{
  testSymm();
  testShape();

  return 0;
}
