/*
************************************************************************
*
*   BlockTab.c - block managment for N dimensional data i/o module
*
*   Copyright (c) 1991-96
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification   : 97/01/24
*   Pathname of SCCS file       : /local/home/kor/autopsy/app/src/ndio/SCCS/s.BlockTab.c
*   SCCS identification         : 1.2
*
*   Modifications :
*   <NAME>  <DATE>  <COMMENTS>
*
************************************************************************
*/

#include "block_tab.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>

#include <linlist.h>
#include <hashtab.h>

#include <ndio/ndio_error.h>
#include "byte_swap.h"

/*
  elements of local (hidden) of linear list of blocks
*/

typedef struct tabEntry **ListEntryPtr;

typedef struct tabEntry {
  HASHTABLE hashTab;
  int     idx;                          /* block-index */
  BOOL    dirty;                        /* true, false */
  int     refCount;                     /* counter for to read */
  BlockPtr data;                        /* pointer data */
  ListEntryPtr lentryP;                 /* pointer to list element */
} TabEntryRec;

/*
  global handle on block tab (BLOCKTABHANDLE)
*/
typedef struct blockTabRec {
  int     fd;                           /* file descriptor */
  NdioFileType type;
  int     headerSize;
  BOOL    byteSwap;
  int     blockDiskSize;                /* size of block */
  int     blockMemSize;                 /* size of block */
  NdioFileMode mode;
  HASHTABLE hashTab;                    /* pointer to hash-tab */
} BlockTabRec;

/***********************************************************************
        STATICS
***********************************************************************/

static int MemoryLimit = 0;
static int MemoryUse;
static LINLIST BlockList = NULL;
static BOOL ExpTabFilled = FALSE;
static float ExpTab[96];


/**********************************************************************/
void
BlockSetMemLimit(
    int memLimit                         /* specify buffer (in Bytes) */
)
/*----------------------------------------------------------------------
Define space in the RAM for data blocks.
----------------------------------------------------------------------*/
{
  MemoryLimit = memLimit;
}  /* BlockSetMemLimit() */


/**********************************************************************/
void
BlockInit(void)
/*----------------------------------------------------------------------
Initializes the buffer management.
----------------------------------------------------------------------*/
{
  if (BlockList == NULL) {
    if (MemoryLimit == 0)
      MemoryLimit = 4 * 1024 * 1024;        /* default 4M */
    MemoryUse = 0;
    BlockList = ListOpen(sizeof(TabEntryRec *));
  }
}  /* BlockInit() */


/**********************************************************************/
static int
getDataSize(NdioFileType type)
{
  switch (type) {
    case NdioEASY8:
      return 1;
    case NdioEASY16:
      return 2;
    default:
      return 4;
  }
}  /* getDataSize() */


/**********************************************************************/
static unsigned
hashFunc(
    TabEntryRec * p_entry,              /* specify a TabEntryRec */
    unsigned tabSize                    /* specify the hashTab size */
)
/*----------------------------------------------------------------------
calculate the hashval for the hashTab
----------------------------------------------------------------------*/
{
  return p_entry->idx % tabSize;
}  /* hashFunc() */


/**********************************************************************/
static int
compFunc(
    TabEntryRec * p_entry1,         /* specify a TabEntryRecord */
    TabEntryRec * p_entry2          /* specify another TabEntryRecord */
)
/*----------------------------------------------------------------------
compare the keys(idx) in hashTab, return 0 if equal
----------------------------------------------------------------------*/
{
  return p_entry1->idx != p_entry2->idx;
}  /* compFunc() */


/**********************************************************************/
static void
destroyBlock(
    TabEntryRec * p_entry,          /* specify TabEntryRec to destroy */
    char *clientdta                 /* specified with HaddDestroyCB */
)
/*----------------------------------------------------------------------
if dirty flag TRUE, write data to disk.
free memory of data block and remove list element.
----------------------------------------------------------------------*/
{
  off_t   posB;
  BLOCKTABHANDLE blockHandle = (BLOCKTABHANDLE) clientdta;

  if (p_entry->dirty) {
    /* give pos of block */
    if (blockHandle->type != NdioFLOAT) {
      NdioError("can only write FLOAT files");
      return;
    }

    posB = (off_t) p_entry->idx * blockHandle->blockDiskSize;
    if (lseek(blockHandle->fd, posB + blockHandle->headerSize, 0) < 0) {
      NdioError("lseek failed in destroyBlock: %s", NdioErrorGetSys());
      return;
    }

    if (blockHandle->byteSwap)
      Reverse4ByteOrder(blockHandle->blockDiskSize / 4, p_entry->data);

    if (write(blockHandle->fd, (char *) p_entry->data,
            blockHandle->blockDiskSize) < 0) {
      NdioError("write failed in destroyBlock: %s", NdioErrorGetSys());
      return;
    }
  }
  free(p_entry->data);
  (void) ListRemove(BlockList, p_entry->lentryP);
  MemoryUse -= blockHandle->blockMemSize;

}  /* destroyBlock() */



/**********************************************************************/
BLOCKTABHANDLE
BlockOpen(
    char *fileName,                  /* Pointer fileName of data file */
    NdioFileType type,               /* type of data in file */
    int headerSize,                  /* size of header (in bytes) */
    int fileSize,                    /* number of data items in file */
    int blockSize,                   /* size of block */
    NdioFileMode mode,               /* read, write or read_write */
    BOOL byteSwap                    /* byte swapping necessary */
)
/*----------------------------------------------------------------------
give mode read,write or read_write
define blockSize
define size of hashTab
opens file to read from, opens hashtable with a destroyCB function
initializes a BLOCKTABHANDLE
----------------------------------------------------------------------*/

{
  int     openMode;
  int     fd;
  struct stat stbuf;
  int     hashTabSize;               /* size of hash-tab */
  BLOCKTABHANDLE blockHandle;        /* specify block table */

  if (mode == NdioREAD) {
    openMode = O_RDONLY;
  } else {
    openMode = O_RDWR;
  }

  fd = open(fileName, openMode);

  if (fd < 0) {
    NdioError("can't open file %s: %s", fileName, NdioErrorGetSys());
    return NULL;
  }
  /* check correct length of file ; refuse if smaller */
  if (fstat(fd, &stbuf) == -1) {     /* stat failed */
    NdioError("can't stat file %s: %s", fileName, NdioErrorGetSys());
    return NULL;
  }
  if (stbuf.st_size < headerSize + getDataSize(type) * fileSize) {
    NdioError("file %s is too small", fileName);
    return NULL;
  }
  /* get a blockHandle */
  blockHandle = malloc(sizeof(BlockTabRec));
  if (blockHandle == NULL) {
    NdioError("memory allocation failed in BlockOpen");
    return NULL;
  }
  /* fill the fields of the BlockTabRec */

  blockHandle->fd = fd;
  blockHandle->type = type;
  blockHandle->headerSize = headerSize;
  blockHandle->byteSwap = byteSwap;
  blockHandle->blockDiskSize = blockSize * getDataSize(type);
  blockHandle->blockMemSize = blockSize * DATA_SIZE;
  blockHandle->mode = mode;

  /* init size of hash-tab */
  hashTabSize = MemoryLimit / blockHandle->blockMemSize;
  hashTabSize = (hashTabSize % 13 + 1) * 13;   /* next multiple of 13 */

  blockHandle->hashTab = HashOpen(hashTabSize, sizeof(TabEntryRec),
      (HashFunc) hashFunc, (HashCompareFunc) compFunc);
  if (blockHandle->hashTab == NULL) {
    NdioError("hash table creation failed in BlockOpen");
    return NULL;
  }
  (void) HashAddDestroyCB(blockHandle->hashTab,
      (DestroyCB) destroyBlock, blockHandle, NULL);

  return blockHandle;

}  /* BlockOpen() */


/**********************************************************************/
static void
fillExpTab(void)
{
  float sqrt2;
  int i;

  if (ExpTabFilled)
    return;
  
  sqrt2 = sqrt(2.0);

  for (i = 1; i <= 47; i++) {
    ExpTab[i] = pow(sqrt2, (double) (i - 1));
    ExpTab[95 - i] = - pow(sqrt2, (double) i);
  }

  ExpTabFilled = TRUE;
}  /* fillExpTab() */


/**********************************************************************/
static void
convertInt(char *dataP, int n)
{
  int i;

  for (i = 0; i < n; i++)
    ((FLOAT32 *) dataP)[i] = (float) ((INT32 *) dataP)[i];
}  /* convertInt() */


/**********************************************************************/
static void
convertEasy8(FLOAT32 *floatP, char *dataP, int n)
{
  int i;

  fillExpTab();

  for (i = 0; i < n; i++)
    floatP[i] = ExpTab[dataP[i]];
}  /* convertEasy8() */


/**********************************************************************/
static void
convertEasy16(FLOAT32 *floatP, char *dataP, int n)
{
  unsigned char *uDataP = (unsigned char *) dataP;
  float scal;
  int i;

  fillExpTab();
  scal = 1.0 / 721.0;

  for (i = 0; i < n; i++)
    floatP[i] = scal * (uDataP[2 * i] + 615) * ExpTab[uDataP[2 * i + 1]];
}  /* convertEasyInt() */


/**********************************************************************/
BlockPtr
BlockGet(
    BLOCKTABHANDLE blockHandle,         /* specify block table */
    int idx,                            /* number of block */
    NdioFileMode mode
)
/*----------------------------------------------------------------------
if block in hash-tab puts block on top of list and returns block (data)
if not in hash-tab, examine memory limit, if not enough memory,
write back oldest block (if dirty), clean up, insert new block at first
position of list, put data in buffer, return block
----------------------------------------------------------------------*/
{
  TabEntryRec searchS;
  TabEntryRec **lastP;
  TabEntryRec **prevP;
  TabEntryRec *found;      /* Pointer TabEntryRec to next TabEntryRec */
  TabEntryRec *newP;
  TabEntryRec **p;
  off_t   posB;
  char *dataP;


  /* search for the block idx */
  searchS.idx = idx;
  found = HashSearch(blockHandle->hashTab, &searchS);

  if (found != NULL) {
    /* block already in hash-tab */
    ListMoveFirst(BlockList, found->lentryP);
    found->refCount++;
    return found->data;
  }

  /* not found */
  lastP = ListLast(BlockList);
  while (lastP && MemoryUse + blockHandle->blockMemSize > MemoryLimit) {
    /* not enough memory, delete oldest block */
    prevP = ListPrev(BlockList, lastP);
    if ((*lastP)->refCount == 0) {
      /* destroyCB removes also data */
      HashRemove((*lastP)->hashTab, *lastP);
    }
    lastP = prevP;
  }

  /* insert block in hashtab */
  newP = HashInsert(blockHandle->hashTab, &searchS, FALSE);
  p = ListInsertFirst(BlockList, &newP);

  /* calculate the position of block */
  posB = (off_t) idx * blockHandle->blockDiskSize;

  if (lseek(blockHandle->fd, posB + blockHandle->headerSize, 0) < 0) {
    NdioError("lseek failed in BlockRead: %s", NdioErrorGetSys());
    return NULL;
  }

  dataP = malloc(blockHandle->blockDiskSize);
  if (dataP == NULL) {
    NdioError("memory allocation failed in BlockGet");
    return NULL;
  }

  if (read(blockHandle->fd,
	  dataP, blockHandle->blockDiskSize) < 0) {
    NdioError("read failed in BlockRead: %s", NdioErrorGetSys());
    return NULL;
  }

  if (blockHandle->blockDiskSize == blockHandle->blockMemSize) {
    if (blockHandle->byteSwap)
      Reverse4ByteOrder(blockHandle->blockMemSize / 4, dataP);

    if (blockHandle->type == NdioINT)
      convertInt(dataP, blockHandle->blockMemSize / 4);

    newP->data = (BlockPtr) dataP;
  } else {
    newP->data = malloc(blockHandle->blockMemSize);
    if (newP->data == NULL) {
      NdioError("memory allocation failed in BlockGet");
      return NULL;
    }

    if (blockHandle->type == NdioEASY8)
      convertEasy8(newP->data, dataP, blockHandle->blockMemSize / 4);
    else
      convertEasy16(newP->data, dataP, blockHandle->blockMemSize / 4);

    free(dataP);
  }

  MemoryUse += blockHandle->blockMemSize;

  newP->hashTab = blockHandle->hashTab;
  newP->idx = idx;
  newP->refCount = 1;
  newP->dirty = FALSE;
  newP->lentryP = p;

  return newP->data;
}  /* BlockGet() */


/**********************************************************************/
void
BlockPut(
    BLOCKTABHANDLE blockHandle,
    int idx,                            /* number of block */
    NdioFileMode mode
)
/*----------------------------------------------------------------------
sets dirty of the specified block to TRUE, if mode is NdioWRITE or
NdioREAD_WRITE (to make sure that this block will be written back
when leaving the buffer).
----------------------------------------------------------------------*/

{
  TabEntryRec *p_entry;
  TabEntryRec searchS;


  /* search for the block idx */
  searchS.idx = idx;
  p_entry = HashSearch(blockHandle->hashTab, &searchS);

#if DEBUG
  if (blockHandle->mode != NdioREAD_WRITE && blockHandle->mode != mode) {
    (void) fprintf(stderr,
        "BlockPut() vs. BlockOpen(): error: mode mismatch\n");
  };
#endif

  if (mode != NdioREAD)
    p_entry->dirty = TRUE;

  p_entry->refCount--;

  if (p_entry->refCount < 0) {
#if DEBUG
    (void) fprintf(stderr, "BlockPut(): error: refCount < 0\n");
#endif
    p_entry->refCount = 0;
  };
}  /* BlockPut() */




/**********************************************************************/
void
BlockClose(BLOCKTABHANDLE blockHandle)
/*----------------------------------------------------------------------
close hashtab, close file
free memory of BLOCKTABHANDLE
----------------------------------------------------------------------*/
{
  HashClose(blockHandle->hashTab);
  /* data blocks and list elements removed by destroyBlock       */

  (void) close(blockHandle->fd);

  free(blockHandle);

}  /* BlockClose() */


/**********************************************************************/
void
BlockCleanup(void)
/*----------------------------------------------------------------------
----------------------------------------------------------------------*/
{
  ListClose(BlockList);
  BlockList = NULL;
}
