/*
************************************************************************
*
*   SpecSegm.c - spectrum segmentation
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/21
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.SpecSegm.c
*   SCCS identification       : 1.4
*
************************************************************************
*/

#include <app/spec_segm.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>

#include <linlist.h>
#include <hashtab.h>
#include "reg_struc.h"
#include "spec.h"
#include "conv.h"
#include "segment.h"

#define READ_POINT_NO 10

typedef struct {
  AppSpectrumP specP;
  AppMaskP inclP, exclP;
  AppNoiseP noiseP;
  float noiseLevel;
  AppRegionP regP;
  int validSize;
  AppDataP dataA;
  float *rowP;
  BOOL isCurr;
  int rowLeft, rowRight;
  int currLeft, currRight;
} SegData;

typedef struct {
  AppRegionP regP;
  float level;
  float *dataA;
  BOOL *oldValidA;
  float *rowP;
  BOOL *validP, *oldValidP;
} SubsegData;

typedef struct {
  int dim;
  int coord[APP_MAX_DIM];
} MaxData;

static LINLIST RegList = NULL;

static unsigned
hashMax(void *p, unsigned size)
{
  MaxData *dataP = p;
  int key, i;

  key = dataP->coord[0];
  for (i = 1; i < dataP->dim; i++)
    key = 37 * key + dataP->coord[i];
  
  return key % size;
}

static int
compMax(void *p1, void *p2)
{
  MaxData *data1P = p1;
  MaxData *data2P = p2;
  int i;

  for (i = 0; i < data1P->dim; i++)
    if (data1P->coord[i] != data2P->coord[i])
      return 1;
  
  return 0;
}

static AppRegionP
newReg(AppSpectrumP specP)
{
  struct AppRegionS regS;

  if (RegList == NULL)
    RegList = ListOpen(sizeof(struct AppRegionS));

  regS.d.dimensionNo = specP->dim;
  regS.d.dataType = specP->type;
  regS.d.maxAmplitude = - MAXFLOAT;
  regS.specP = specP;
  regS.maxL = NULL;
  regS.copyP = NULL;

  return ListInsertLast(RegList, &regS);
}

static BOOL
segCheck(int *coord, BOOL sameRow, void *clientData)
{
  SegData *dataP = clientData;
  int dim = dataP->specP->dim;
  AppDataRange rangeA[APP_MAX_DIM];
  BOOL inside;
  int spanSize;
  float *v;
  int ind, i;

  if (! sameRow)
    dataP->isCurr = FALSE;

  if (! AppMaskIsInside(dataP->inclP, coord))
    return FALSE;

  inside = TRUE;
  for (i = 0; i < dim; i++)
    if (coord[i] <= dataP->regP->d.rangeA[i][0] ||
        coord[i] >= dataP->regP->d.rangeA[i][1]) {
      inside = FALSE;
      break;
    }

  if (inside) {
    ind = 0;
    for (i = dim - 1; i >= 0; i--)
      ind = ind * (dataP->regP->d.rangeA[i][1] -
	  dataP->regP->d.rangeA[i][0] + 1) +
	  coord[i] - dataP->regP->d.rangeA[i][0];

    if (dataP->regP->d.validA[ind])
      return FALSE;
  }

  if (dataP->exclP != NULL && AppMaskIsInside(dataP->exclP, coord))
    return FALSE;

  if (! dataP->isCurr)
    AppMaskGetRange(dataP->inclP, coord, &dataP->rowLeft, &dataP->rowRight);

  if (! dataP->isCurr ||
      coord[0] < dataP->currLeft || coord[0] > dataP->currRight) {
    dataP->currLeft = coord[0] - READ_POINT_NO;
    if (dataP->currLeft < dataP->rowLeft)
      dataP->currLeft = dataP->rowLeft;

    dataP->currRight = coord[0] + READ_POINT_NO;
    if (dataP->currRight > dataP->rowRight)
      dataP->currRight = dataP->rowRight;

    rangeA[0][0] = dataP->currLeft;
    rangeA[0][1] = dataP->currRight;

    for (i = 1; i < dim; i++) {
      rangeA[i][0] = coord[i];
      rangeA[i][1] = coord[i];
    }

    spanSize = dataP->currRight - dataP->currLeft + 1;
    AppReadSpectrum(dataP->specP, rangeA, dataP->dataA);
    v = ConvGetFloat(dataP->dataA, spanSize, dataP->specP->type);
    (void) memcpy(dataP->rowP, v, spanSize * sizeof(*v));

    dataP->isCurr = TRUE;
  }

  if (dataP->rowP[coord[0] - dataP->currLeft] < 
      dataP->noiseLevel * AppNoiseGetMagnitude(dataP->noiseP, coord))
    return FALSE;

  return TRUE;
}

static void
segAddSpan(int *coord, int left, int right, void *clientData)
{
  SegData *dataP = clientData;
  int dim = dataP->specP->dim;
  int start[APP_MAX_DIM], end[APP_MAX_DIM];
  int oldSizeA[APP_MAX_DIM], newSizeA[APP_MAX_DIM];
  int boxSize, oldInd, newInd, offs;
  BOOL inside;
  int newCoord[APP_MAX_DIM];
  int i;

  if (left <= dataP->regP->d.rangeA[0][0])
    start[0] = left - 1;
  else
    start[0] = dataP->regP->d.rangeA[0][0];

  if (right >= dataP->regP->d.rangeA[0][1])
    end[0] = right + 1;
  else
    end[0] = dataP->regP->d.rangeA[0][1];

  for (i = 0; i < dim; i++)
    oldSizeA[i] = dataP->regP->d.rangeA[i][1] -
	dataP->regP->d.rangeA[i][0] + 1;

  newSizeA[0] = end[0] - start[0] + 1;
  boxSize = newSizeA[0];

  for (i = 1; i < dim; i++) {
    if (coord[i] <= dataP->regP->d.rangeA[i][0])
      start[i] = coord[i] - 1;
    else
      start[i] = dataP->regP->d.rangeA[i][0];

    if (coord[i] >= dataP->regP->d.rangeA[i][1])
      end[i] = coord[i] + 1;
    else
      end[i] = dataP->regP->d.rangeA[i][1];

    newSizeA[i] = end[i] - start[i] + 1;
    boxSize *= newSizeA[i];
  }

  if (boxSize > dataP->regP->boxSize) {
    if (boxSize > dataP->validSize) {
      while (dataP->validSize < boxSize)
        dataP->validSize *= 2;
      dataP->regP->d.validA = realloc(dataP->regP->d.validA,
          dataP->validSize * sizeof(*dataP->regP->d.validA));
    }

    for (i = 1; i < dim; i++)
      newCoord[i] = end[i];

    for (;;) {
      newInd = 0;
      inside = TRUE;
      for (i = dim - 1; i >= 1; i--) {
        newInd = newInd * newSizeA[i] + newCoord[i] - start[i];
        if (newCoord[i] < dataP->regP->d.rangeA[i][0] ||
            newCoord[i] > dataP->regP->d.rangeA[i][1])
	  inside = FALSE;
      }
      newInd *= newSizeA[0];

      if (inside) {
	oldInd = 0;
	for (i = dim - 1; i >= 1; i--)
	  oldInd = oldInd * oldSizeA[i] +
	      newCoord[i] - dataP->regP->d.rangeA[i][0];
	oldInd *= oldSizeA[0];
	offs = dataP->regP->d.rangeA[i][0] - start[i];

	(void) memset(dataP->regP->d.validA + newInd + offs + oldSizeA[0], 0,
	    (newSizeA[0] - oldSizeA[0] - offs) *
	    sizeof(*dataP->regP->d.validA));
	(void) memmove(dataP->regP->d.validA + newInd + offs,
	    dataP->regP->d.validA + oldInd,
	    oldSizeA[0] * sizeof(*dataP->regP->d.validA));
	(void) memset(dataP->regP->d.validA + newInd, 0,
	    offs * sizeof(*dataP->regP->d.validA));
      } else {
	(void) memset(dataP->regP->d.validA + newInd, 0,
	    newSizeA[0] * sizeof(*dataP->regP->d.validA));
      }

      for (i = 1; i < dim; i++) {
        if (newCoord[i] == start[i]) {
          newCoord[i] = end[i];
        } else {
          newCoord[i]--;
          break;
        }
      }

      if (i == dim)
        break;
    }
  }

  newInd = 0;
  for (i = dim - 1; i >= 1; i--)
    newInd = newInd * newSizeA[i] + coord[i] - start[i];
  newInd = newInd * newSizeA[0] + left - start[0];
  for (i = 0; i <= right - left; i++)
    dataP->regP->d.validA[newInd + i] = TRUE;

  for (i = 0; i < dim; i++) {
    dataP->regP->d.rangeA[i][0] = start[i];
    dataP->regP->d.rangeA[i][1] = end[i];
  }

  dataP->regP->boxSize = boxSize;
}

static void
calcRegNoise(AppRegionP regP, AppNoiseP noiseP, float noiseMagn)
{
  int coord[APP_MAX_DIM];
  float lev;
  int ind, i;

  regP->d.noiseMagnitudeMax = - MAXFLOAT;
  regP->d.baseLevel = MAXFLOAT;

  for (i = 0; i < regP->d.dimensionNo; i++)
    coord[i] = regP->d.rangeA[i][0];

  ind = 0;
  for (;;) {
    if (regP->d.validA[ind]) {
      lev = AppNoiseGetMagnitude(noiseP, coord);
      if (lev > regP->d.noiseMagnitudeMax)
        regP->d.noiseMagnitudeMax = lev;
      if (lev < regP->d.baseLevel)
        regP->d.baseLevel = lev;
    }

    for (i = 0; i < regP->d.dimensionNo; i++) {
      if (coord[i] >= regP->d.rangeA[i][1]) {
        coord[i] = regP->d.rangeA[i][0];
      } else {
        coord[i]++;
        break;
      }
    }

    if (i == regP->d.dimensionNo)
      break;

    ind++;
  }

  regP->d.baseLevel *= noiseMagn;
}

static BOOL
getNextCoord(int *sizeA, int dim, int *coord)
{
  int i;

  for (i = 1; i < dim; i++)
    if (coord[i] >= sizeA[i] - 1) {
      coord[i] = 1;
    } else {
      coord[i]++;
      break;
    }

  return (i == dim);
}

int
AppSegment(AppSpectrumP specP, AppMaskP inclP, AppMaskP exclP,
    AppNoiseP noiseP, float noiseLevel, float minLevel, int minSizeA[])
{
  SegData data;
  HASHTABLE maxTab;
  int rowSize, rowNo, rowI;
  float **rowA, *v;
  int regNo;
  int coord[APP_MAX_DIM];
  AppDataRange rowRangeA[APP_MAX_DIM];
  float amp;
  MaxData maxData;
  BOOL localMax;
  AppMaxP maxP;
  int left, right, c0, i;

  if (noiseLevel < 0.0) {
    AppSetSpectrumSign(specP, -1);
    noiseLevel = - noiseLevel;
  }

  rowSize = specP->sizeA[0];

  data.specP = specP;
  data.inclP = inclP;
  data.exclP = exclP;
  data.noiseP = noiseP;
  data.noiseLevel = noiseLevel;
  data.dataA = malloc(4 * rowSize);
  data.rowP = malloc(rowSize * sizeof(*data.rowP));

  maxTab = HashOpen(9029, sizeof(MaxData), hashMax, compMax);

  rowNo = 1;
  for (i = 1; i < specP->dim; i++)
    rowNo *= 3;

  rowA = malloc(rowNo * sizeof(*rowA));
  for (i = 0; i < rowNo; i++)
    rowA[i] = malloc(rowSize * sizeof(*rowA[i]));

  for (i = 1; i < specP->dim; i++)
    coord[i] = 1;

  regNo = 0;

  for (;;) {
    AppMaskGetRange(inclP, coord, &left, &right);
    if (left > right) {
      if (getNextCoord(specP->sizeA, specP->dim, coord))
	break;
      continue;
    }

    rowRangeA[0][0] = left;
    rowRangeA[0][1] = right;

    for (i = 1; i < specP->dim; i++) {
      rowRangeA[i][0] = coord[i] - 1;
      rowRangeA[i][1] = coord[i] - 1;
    }

    rowI = 0;
    for(;;) {
      AppReadSpectrum(specP, rowRangeA, data.dataA);
      v = ConvGetFloat(data.dataA, rowSize, specP->type);
      (void) memcpy(rowA[rowI], v, rowSize * sizeof(*v));

      for (i = 1; i < specP->dim; i++) {
        if (rowRangeA[i][0] == coord[i] + 1) {
          rowRangeA[i][0] = coord[i] - 1;
          rowRangeA[i][1] = coord[i] - 1;
        } else {
          rowRangeA[i][0]++;
          rowRangeA[i][1]++;
          break;
        }
      }

      if (i == specP->dim)
        break;

      rowI++;
    }

    for (c0 = left + 1; c0 < right; c0++) {
      coord[0] = c0;

      amp = rowA[(rowNo - 1) / 2][c0 - left];
      if (amp < minLevel * AppNoiseGetMagnitude(noiseP, coord))
        continue;
  
      localMax = TRUE;
      for (rowI = 0; rowI < rowNo; rowI++)
        if (amp < rowA[rowI][c0 - left - 1] ||
            amp < rowA[rowI][c0 - left] ||
            amp < rowA[rowI][c0 - left + 1]) {
          localMax = FALSE;
          break;
        }
  
      if (! localMax)
        continue;

      if (exclP != NULL && AppMaskIsInside(exclP, coord))
	continue;

      maxData.dim = specP->dim;
      for (i = 0; i < specP->dim; i++)
	maxData.coord[i] = coord[i];
      if (HashSearch(maxTab, &maxData) != NULL)
        continue;

      data.regP = newReg(specP);

      for (i = 0; i < specP->dim; i++) {
        data.regP->d.rangeA[i][0] = coord[i];
        data.regP->d.rangeA[i][1] = coord[i];
      }

      data.validSize = 10;
      data.regP->boxSize = 0;
      data.regP->d.validA = malloc(data.validSize *
          sizeof(*data.regP->d.validA));

      SegFill(specP->dim, coord, segCheck, segAddSpan, &data);

      for (i = 0; i < specP->dim; i++) {
	data.regP->d.subRangeA[i][0] = 1;
	data.regP->d.subRangeA[i][1] = 
	    data.regP->d.rangeA[i][1] - data.regP->d.rangeA[i][0] - 1;
	if (data.regP->d.subRangeA[i][1] < minSizeA[i])
	  break;
      }

      if (i < specP->dim) {
	free(data.regP->d.validA);
	ListRemove(RegList, data.regP);
	continue;
      }

      regNo++;

      data.regP->d.validA = realloc(data.regP->d.validA,
          data.regP->boxSize * sizeof(*data.regP->d.validA));

      data.regP->d.dataA = malloc(4 * data.regP->boxSize);
      AppReadSpectrum(specP, data.regP->d.rangeA, data.regP->d.dataA);

      if (noiseP == NULL) {
	data.regP->d.noiseMagnitudeMax = noiseLevel;
	data.regP->d.baseLevel = noiseLevel;
      } else {
	calcRegNoise(data.regP, noiseP, noiseLevel);
      }

      /* these are used for AppMaxUpdate */
      data.regP->noiseP = noiseP;
      data.regP->segLevel = noiseLevel;
      data.regP->minLevel = minLevel;

      AppMaxUpdate(data.regP);
      maxP = ListFirst(data.regP->maxL);
      while (maxP != NULL) {
	maxData.dim = specP->dim;
	for (i = 0; i < specP->dim; i++)
	  maxData.coord[i] = maxP->coordinates[i];
	(void) HashInsert(maxTab, &maxData, FALSE);
	maxP = ListNext(data.regP->maxL, maxP);
      }
    }

    if (getNextCoord(specP->sizeA, specP->dim, coord))
      break;
  }

  free(data.dataA);
  free(data.rowP);
  for (i = 0; i < rowNo; i++)
    free(rowA[i]);
  free(rowA);

  HashClose(maxTab);

  AppSetSpectrumSign(specP, 1);

  return regNo;
}

void
AppRegionApply(AppSpectrumP specP, AppRegionApplyF applyF, void *clientData)
{
  AppRegionP regP;

  regP = ListFirst(RegList);
  while (regP != NULL) {
    if (regP->specP == specP)
      applyF(regP, clientData);
    regP = ListNext(RegList, regP);
  }
}

void
AppRegionRemove(AppRegionP regP)
{
  RegionStrucFree(regP);
  ListRemove(RegList, regP);
}

static BOOL
subsegCheck(int *coord, BOOL sameRow, void *clientData)
{
  SubsegData *dataP = clientData;
  int dim = dataP->regP->d.dimensionNo;
  BOOL inside;
  int ind, i;

  if (! sameRow) {
    inside = TRUE;
    for (i = 1; i < dim; i++)
      if (coord[i] <= 0 || coord[i] >=
	  dataP->regP->d.rangeA[i][1] - dataP->regP->d.rangeA[i][0])
        inside = FALSE;

    if (inside) {
      ind = 0;
      for (i = dim - 1; i >= 1; i--)
        ind = ind * (dataP->regP->d.rangeA[i][1] -
            dataP->regP->d.rangeA[i][0] + 1) + coord[i];
      ind *= dataP->regP->d.rangeA[0][1] - dataP->regP->d.rangeA[0][0] + 1;
      dataP->rowP = dataP->dataA + ind;
      dataP->validP = dataP->regP->d.validA + ind;
      dataP->oldValidP = dataP->oldValidA + ind;
    } else {
      dataP->rowP = NULL;
    }
  }

  if (dataP->rowP == NULL)
    return FALSE;

  if (coord[0] <= 0 ||
      coord[0] >= dataP->regP->d.rangeA[0][1] - dataP->regP->d.rangeA[0][0])
    return FALSE;

  if (! dataP->oldValidP[coord[0]])
    return FALSE;

  if (dataP->rowP[coord[0]] < dataP->level)
    return FALSE;

  if (dataP->validP[coord[0]])
    return FALSE;

  return TRUE;
}

static void
subsegAddSpan(int *coord, int left, int right, void *clientData)
{
  SubsegData *dataP = clientData;
  int ind, i;

  ind = 0;
  for (i = dataP->regP->d.dimensionNo - 1; i >= 1; i--)
    ind = ind * (dataP->regP->d.rangeA[i][1] -
	dataP->regP->d.rangeA[i][0] + 1) + coord[i];
  ind = ind * (dataP->regP->d.rangeA[0][1] -
      dataP->regP->d.rangeA[0][0] + 1) + left;

  for (i = 0; i <= right - left; i++)
    dataP->regP->d.validA[ind + i] = TRUE;

  for (i = 1; i < dataP->regP->d.dimensionNo; i++) {
    if (coord[i] < dataP->regP->d.subRangeA[i][0])
      dataP->regP->d.subRangeA[i][0] = coord[i];
    if (coord[i] > dataP->regP->d.subRangeA[i][1])
      dataP->regP->d.subRangeA[i][1] = coord[i];
  }

  if (left < dataP->regP->d.subRangeA[0][0])
    dataP->regP->d.subRangeA[0][0] = left;
  if (right > dataP->regP->d.subRangeA[0][1])
    dataP->regP->d.subRangeA[0][1] = right;
}

void
AppRegionSubsegment(AppRegionP regP, float level, int coord[])
{
  int seedA[APP_MAX_DIM];
  SubsegData data;
  int i;

  for (i = 0; i < regP->d.dimensionNo; i++) {
    seedA[i] = coord[i] - regP->d.rangeA[i][0];
    regP->d.subRangeA[i][0] = seedA[i] + 1;
    regP->d.subRangeA[i][1] = seedA[i] - 1;
  }

  data.oldValidA = malloc(regP->boxSize * sizeof(*data.oldValidA));
  (void) memcpy(data.oldValidA, regP->d.validA,
      regP->boxSize * sizeof(*data.oldValidA));
  (void) memset(regP->d.validA, 0, regP->boxSize * sizeof(*regP->d.validA));

  data.regP = regP;
  data.level = level;
  data.dataA = ConvGetRegionFloat((AppRegionDataP) regP);
  SegFill(regP->d.dimensionNo, seedA, subsegCheck, subsegAddSpan, &data);

  free(data.oldValidA);
}
