/*
************************************************************************
*
*   SpecNoise.c - local noise determination
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/21
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.SpecNoise.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <app/spec_noise.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>

#include "spec.h"
#include "conv.h"

#define MIN_LEN 5

struct AppNoiseS {
  int dim;
  AppDataRange rangeA[APP_MAX_DIM];
  float baseS;
  float *magnS[APP_MAX_DIM];
};

static float
getVecNoise(float *v, int n)
{
  int len;
  double sum;  /* for better precision */
  float bestSum;
  int i;

  if (n < 2)
    return 0.0;

  len = n / 20;
  if (len < MIN_LEN) {
    len = MIN_LEN;
    if (len > n / 2)
      len = n / 2;
  }

  sum = 0;
  for (i = 0; i < len; i++)
    sum += v[i] * v[i];
  
  bestSum = sum;

  for (i = len; i < n; i++) {
    sum -= v[i - len] * v[i - len];
    sum += v[i] * v[i];

    if (sum < bestSum)
      bestSum = sum;
  }

  return 5.0 * (bestSum / len);
}

AppNoiseP
AppNoisePrepare(AppSpectrumP specP, AppDataRange rangeA[])
{
  AppNoiseP noiseP;
  int sizeA[APP_MAX_DIM];
  int totSize;
  AppDataP dataA;
  AppDataRange coord[APP_MAX_DIM];
  float *v, valS;
  int dom, ind, i;

  noiseP = malloc(sizeof(*noiseP));

  totSize = 1;
  for (dom = 0; dom < specP->dim; dom++) {
    sizeA[dom] = rangeA[dom][1] - rangeA[dom][0] + 1;
    totSize *= sizeA[dom];
  }

  noiseP->dim = specP->dim;

  noiseP->baseS = MAXFLOAT;

  for (dom = 0; dom < specP->dim; dom++) {
    noiseP->rangeA[dom][0] = rangeA[dom][0];
    noiseP->rangeA[dom][1] = rangeA[dom][1];

    noiseP->magnS[dom] = malloc((totSize / sizeA[dom]) * sizeof(float));
    
    for (i = 0; i < specP->dim; i++) {
      coord[i][0] = rangeA[i][0];
      coord[i][1] = rangeA[i][0];
    }

    coord[dom][0] = rangeA[dom][0];
    coord[dom][1] = rangeA[dom][1];

    dataA = malloc(4 * sizeA[dom]);

    ind = 0;

    for (;;) {
      AppReadSpectrum(specP, coord, dataA);
      v = ConvGetFloat(dataA, sizeA[dom], specP->type);

      valS = getVecNoise(v, sizeA[dom]);
      noiseP->magnS[dom][ind] = valS;
      ind++;

      if (valS < noiseP->baseS)
	noiseP->baseS = valS;

      for (i = 0; i < specP->dim; i++) {
	if (i == dom)
	  continue;
	if (coord[i][0] == rangeA[i][1]) {
	  coord[i][0] = rangeA[i][0];
	  coord[i][1] = rangeA[i][0];
	} else {
	  coord[i][0]++;
	  coord[i][1]++;
	  break;
	}
      }

      if (i == specP->dim)
	break;
    }

    free(dataA);
  }

  noiseP->baseS *= noiseP->dim - 1;

  return noiseP;
}

float
AppNoiseGetMagnitude(AppNoiseP noiseP, int coord[])
{
  int i0, i, k;
  float valS;

  if (noiseP == NULL)
    return 1.0;

  valS = - noiseP->baseS;

  for (i = 0; i < noiseP->dim; i++) {
    i0 = 0;
    for (k = noiseP->dim - 1; k >= 0; k--) {
      if (k == i)
	continue;
      
      i0 = i0 * (noiseP->rangeA[k][1] - noiseP->rangeA[k][0] + 1) +
	  coord[k] - noiseP->rangeA[k][0];
    }

    valS += noiseP->magnS[i][i0];
  }

  return sqrt(valS);
}

float
AppNoiseGetMagnitudeFloat(AppNoiseP noiseP, float posA[])
{
  AppDataRange rangeA[APP_MAX_DIM];
  int coord[APP_MAX_DIM];
  float magn, maxMagn;
  int i;

  if (noiseP == NULL)
    return 1.0;

  for (i = 0; i < noiseP->dim; i++) {
    rangeA[i][0] = (int) posA[i];
    rangeA[i][1] = rangeA[i][0] + 1;
    coord[i] = rangeA[i][0];
  }

  maxMagn = - MAXFLOAT;

  for (;;) {
    magn = AppNoiseGetMagnitude(noiseP, coord);
    if (magn > maxMagn)
      maxMagn = magn;

    for (i = 0; i < noiseP->dim; i++)
      if (coord[i] == rangeA[i][1]) {
        coord[i] = rangeA[i][0];
      } else {
        coord[i]++;
        break;
      }

    if (i == noiseP->dim)
      break;
  }

  return maxMagn;
}

float
AppNoiseGetRow(AppNoiseP noiseP, int coord[])
{
  int dom, i0, k;

  dom = -1;
  i0 = 0;
  for (k = noiseP->dim - 1; k >= 0; k--) {
    if (coord[k] < 0) {
      dom = k;
      continue;
    }
    
    i0 = i0 * (noiseP->rangeA[k][1] - noiseP->rangeA[k][0] + 1) +
	coord[k] - noiseP->rangeA[k][0];
  }

  if (dom < 0)  /* error! */
    return 0.0;

  return sqrt(noiseP->magnS[dom][i0]);
}

void
AppNoiseDestroy(AppNoiseP noiseP)
{
  int i;

  for (i = 0; i < noiseP->dim; i++)
    free(noiseP->magnS[i]);

  free(noiseP);
}
