/*
************************************************************************
*
*   SpecMask.c - description of spectrum parts (boxes, diagonals etc.)
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/25
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.SpecMask.c
*   SCCS identification       : 1.7
*
************************************************************************
*/

#include <app/spec_mask.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "spec.h"
#include "conv.h"
#include "segment.h"

#define READ_POINT_NO 10

typedef struct {
  int startI, endI;
} SpanDescr;

typedef struct {
  int spanNo;
  SpanDescr *spanA;
} SpanSetDescr;

struct AppMaskS {
  int dim;
  AppSpectrumP specP;
  int sizeA[APP_MAX_DIM];
  SpanSetDescr *spanSetA;
};

typedef struct {
  AppMaskP maskP;
  AppMaskP inclP;
  AppNoiseP noiseP;
  float noiseLevel;
  AppDataP dataA;
  float *rowP;
  SpanSetDescr *currSetP;
  BOOL isCurr;
  int currLeft, currRight;
} MaskData;

AppMaskP
AppMaskPrepare(AppSpectrumP specP)
{
  AppMaskP maskP;
  int spanSetNo;
  int dom, i;

  maskP = malloc(sizeof(*maskP));
  maskP->dim = specP->dim;
  maskP->specP = specP;

  for (dom = 0; dom < maskP->dim; dom++)
    maskP->sizeA[dom] = specP->sizeA[dom];

  spanSetNo = 1;
  for (dom = 1; dom < maskP->dim; dom++)
    spanSetNo *= maskP->sizeA[dom];

  maskP->spanSetA = malloc(spanSetNo * sizeof(*maskP->spanSetA));

  for (i = 0; i < spanSetNo; i++)
    maskP->spanSetA[i].spanNo = 0;

  return maskP;
}

void
AppMaskBox(AppMaskP maskP, AppDataRange rangeA[])
{
  int coord[APP_MAX_DIM];
  SpanSetDescr *setP;
  BOOL inside;
  int ind, dom;

  for (dom = 1; dom < maskP->dim; dom++)
    coord[dom] = 0;

  ind = 0;
  for (;;) {
    setP = maskP->spanSetA + ind;

    if (setP->spanNo > 0)
      free(setP->spanA);

    inside = TRUE;
    for (dom = 1; dom < maskP->dim; dom++)
      if (coord[dom] < rangeA[dom][0] || coord[dom] > rangeA[dom][1]) {
	inside = FALSE;
	break;
      }
    
    if (inside) {
      setP->spanA = malloc(sizeof(*setP->spanA));
      setP->spanNo = 1;
      setP->spanA[0].startI = rangeA[0][0];
      setP->spanA[0].endI = rangeA[0][1];
    } else {
      setP->spanNo = 0;
    }

    for (dom = 1; dom < maskP->dim; dom++)
      if (coord[dom] == maskP->sizeA[dom] - 1) {
	coord[dom] = 0;
      } else {
	coord[dom]++;
	break;
      }
    
    if (dom == maskP->dim)
      break;
    
    ind++;
  }
}

static BOOL
maskCheck(int *coord, BOOL sameRow, void *clientData)
{
  MaskData *dataP = clientData;
  AppMaskP maskP = dataP->maskP;
  AppDataRange rangeA[APP_MAX_DIM];
  int spanSize;
  float *v;
  int ind, dom, i;

  if (! sameRow)
    dataP->isCurr = FALSE;

  for (dom = 0; dom < maskP->dim; dom++)
    if (coord[dom] < 0 || coord[dom] >= maskP->sizeA[dom])
      return FALSE;

  if (dataP->inclP != NULL &&
      ! AppMaskIsInside(dataP->inclP, coord))
    return FALSE;

  if (! dataP->isCurr) {
    ind = 0;
    for (dom = maskP->dim - 1; dom >= 1; dom--)
      ind = ind * maskP->sizeA[dom] + coord[dom];
    dataP->currSetP = maskP->spanSetA + ind;
  }

  for (i = 0; i < dataP->currSetP->spanNo; i++)
    if (coord[0] >= dataP->currSetP->spanA[i].startI &&
	coord[0] <= dataP->currSetP->spanA[i].endI)
      return FALSE;

  if (! dataP->isCurr ||
      coord[0] < dataP->currLeft || coord[0] > dataP->currRight) {
    dataP->currLeft = coord[0] - READ_POINT_NO;
    if (dataP->currLeft < 0)
      dataP->currLeft = 0;

    dataP->currRight = coord[0] + READ_POINT_NO;
    if (dataP->currRight >= maskP->sizeA[0])
      dataP->currRight = maskP->sizeA[0] - 1;

    rangeA[0][0] = dataP->currLeft;
    rangeA[0][1] = dataP->currRight;

    for (dom = 1; dom < maskP->dim; dom++) {
      rangeA[dom][0] = coord[dom];
      rangeA[dom][1] = coord[dom];
    }

    spanSize = dataP->currRight - dataP->currLeft + 1;
    AppReadSpectrum(maskP->specP, rangeA, dataP->dataA);
    v = ConvGetFloat(dataP->dataA, spanSize, maskP->specP->type);
    (void) memcpy(dataP->rowP, v, spanSize * sizeof(*v));

    dataP->isCurr = TRUE;
  }

  if (dataP->rowP[coord[0] - dataP->currLeft] <
      dataP->noiseLevel * AppNoiseGetMagnitude(dataP->noiseP, coord))
    return FALSE;

  return TRUE;
}

static void
maskAddSpan(int *coord, int left, int right, void *clientData)
{
  MaskData *dataP = clientData;
  AppMaskP maskP = dataP->maskP;
  SpanSetDescr *setP;
  int dom, ind;

  ind = 0;
  for (dom = maskP->dim - 1; dom >= 1; dom--)
    ind = ind * maskP->sizeA[dom] + coord[dom];

  setP = maskP->spanSetA + ind;

  setP->spanNo++;
  if (setP->spanNo == 1)
    setP->spanA = malloc(sizeof(*setP->spanA));
  else
    setP->spanA = realloc(setP->spanA, setP->spanNo * sizeof(*setP->spanA));
  
  setP->spanA[setP->spanNo - 1].startI = left;
  setP->spanA[setP->spanNo - 1].endI = right;
}

static void
sortSpans(SpanDescr *spanA, int leftI, int rightI)
{
  int i, j;
  int x;
  SpanDescr w;

  /* quicksort (Wirth) */
  i = leftI;
  j = rightI;
  x = spanA[(leftI + rightI) / 2].startI;
  for (;;) {
    while (spanA[i].startI < x)
      i++;
    while (x < spanA[j].startI)
      j--;

    if (i <= j) {
      w = spanA[i];
      spanA[i] = spanA[j];
      spanA[j] = w;
      i++;
      j--;
    }

    if (i > j)
      break;
  }

  if (leftI < j)
    sortSpans(spanA, leftI, j);
  if (i < rightI)
    sortSpans(spanA, i, rightI);
}

static void
sortSpanSets(AppMaskP maskP)
{
  int spanSetNo, dom, i;

  spanSetNo = 1;
  for (dom = 1; dom < maskP->dim; dom++)
    spanSetNo *= maskP->sizeA[dom];

  for (i = 0; i < spanSetNo; i++)
    if (maskP->spanSetA[i].spanNo > 1)
      sortSpans(maskP->spanSetA[i].spanA, 0, maskP->spanSetA[i].spanNo - 1);
}

void
AppMaskDiagonal(AppMaskP maskP, AppMaskP inclP,
    AppNoiseP noiseP, float noiseLevel,
    char *unit, int dom1, int dom2)
{
  MaskData data;
  int coord[APP_MAX_DIM];
  float posA[APP_MAX_DIM], symmPosA[APP_MAX_DIM];
  int dom, i;

  if (dom1 == dom2 || dom1 >= maskP->dim || dom2 >= maskP->dim)
    return;

  if (noiseLevel < 0.0) {
    AppSetSpectrumSign(maskP->specP, -1);
    noiseLevel = - noiseLevel;
  }

  data.maskP = maskP;
  data.inclP = inclP;
  data.noiseP = noiseP;
  data.noiseLevel = noiseLevel;
  data.dataA = malloc(4 * maskP->sizeA[0]);
  data.rowP = malloc(maskP->sizeA[0] * sizeof(*data.rowP));

  if (maskP->sizeA[dom1] > maskP->sizeA[dom2]) {
    dom = dom1;
    dom1 = dom2;
    dom2 = dom;
  }

  for (dom = 0; dom < maskP->dim; dom++)
    coord[dom] = 0;

  for (;;) {
    for (dom = 0; dom < maskP->dim; dom++)
      posA[dom] = (float) coord[dom];

    AppConvertSpectrumUnit(maskP->specP, posA, "point", symmPosA, unit);
    symmPosA[dom1] = symmPosA[dom2];
    AppConvertSpectrumUnit(maskP->specP, symmPosA, unit, posA, "point");
    coord[dom1] = (float) (posA[dom1] + 0.5);

    if (coord[dom1] >= 0 && coord[dom1] < maskP->sizeA[dom1])
      SegFill(maskP->dim, coord, maskCheck, maskAddSpan, &data);

    for (dom = 0; dom < maskP->dim; dom++) {
      if (dom == dom1)
	continue;

      if (coord[dom] == maskP->sizeA[dom] - 1) {
	coord[dom] = 0;
      } else {
	coord[dom]++;
	break;
      }
    }
    
    if (dom == maskP->dim)
      break;
  }

  free(data.dataA);
  free(data.rowP);

  AppSetSpectrumSign(maskP->specP, 1);

  sortSpanSets(maskP);
}

void
AppMaskBorder(AppMaskP maskP, AppMaskP inclP,
    AppNoiseP noiseP, float noiseLevel, int width)
{
  MaskData data;
  int coord[APP_MAX_DIM];
  AppDataRange rangeA[APP_MAX_DIM];
  float *v;
  BOOL isBorder;
  int dom, i;

  if (width <= 0)
    return;

  if (noiseLevel < 0.0) {
    AppSetSpectrumSign(maskP->specP, -1);
    noiseLevel = - noiseLevel;
  }

  data.maskP = maskP;
  data.inclP = inclP;
  data.noiseP = noiseP;
  data.noiseLevel = noiseLevel;
  data.dataA = malloc(4 * maskP->sizeA[0]);
  data.rowP = malloc(maskP->sizeA[0] * sizeof(*data.rowP));

  for (dom = 1; dom < maskP->dim; dom++) {
    coord[dom] = 0;
    rangeA[dom][0] = 0;
    rangeA[dom][1] = 0;
  }

  for (;;) {
    isBorder = FALSE;
    for (dom = 1; dom < maskP->dim; dom++)
      if (coord[dom] < width ||
	  coord[dom] > maskP->sizeA[dom] - 1 - width) {
	isBorder = TRUE;
	break;
      }

    if (isBorder) {
      rangeA[0][0] = 0;
      rangeA[0][1] = maskP->sizeA[0] - 1;

      AppReadSpectrum(maskP->specP, rangeA, data.dataA);
      v = ConvGetFloat(data.dataA, maskP->sizeA[0], maskP->specP->type);

      for (i = 0; i < maskP->sizeA[0]; i++) {
	coord[0] = i;
	if (v[i] > noiseLevel * AppNoiseGetMagnitude(noiseP, coord))
	  SegFill(maskP->dim, coord, maskCheck, maskAddSpan, &data);
      }
    } else {
      rangeA[0][0] = 0;
      rangeA[0][1] = width - 1;

      AppReadSpectrum(maskP->specP, rangeA, data.dataA);
      v = ConvGetFloat(data.dataA, maskP->sizeA[0], maskP->specP->type);

      for (i = 0; i < width; i++) {
	coord[0] = i;
	if (v[i] > noiseLevel * AppNoiseGetMagnitude(noiseP, coord))
	  SegFill(maskP->dim, coord, maskCheck, maskAddSpan, &data);
      }

      rangeA[0][0] = maskP->sizeA[0] - width;
      rangeA[0][1] = maskP->sizeA[0] - 1;

      AppReadSpectrum(maskP->specP, rangeA, data.dataA);
      v = ConvGetFloat(data.dataA, maskP->sizeA[0], maskP->specP->type);

      for (i = 0; i < width; i++) {
	coord[0] = maskP->sizeA[0] - width + i;
	if (v[i] > noiseLevel * AppNoiseGetMagnitude(noiseP, coord))
	  SegFill(maskP->dim, coord, maskCheck, maskAddSpan, &data);
      }
    }

    for (dom = 1; dom < maskP->dim; dom++)
      if (coord[dom] == maskP->sizeA[dom] - 1) {
	coord[dom] = 0;
	rangeA[dom][0] = 0;
	rangeA[dom][1] = 0;
      } else {
	coord[dom]++;
	rangeA[dom][0]++;
	rangeA[dom][1]++;
	break;
      }
    
    if (dom == maskP->dim)
      break;
  }

  free(data.dataA);
  free(data.rowP);

  AppSetSpectrumSign(maskP->specP, 1);

  sortSpanSets(maskP);
}

BOOL
AppMaskIsInside(AppMaskP maskP, int coord[])
{
  SpanSetDescr *setP;
  int dom, ind, jl, jm, ju;

  if (maskP == NULL)  /* NULL is considered an empty mask */
    return FALSE;

  for (dom = 0; dom < maskP->dim; dom++)
    if (coord[dom] < 0 || coord[dom] >= maskP->sizeA[dom])
      return FALSE;

  ind = 0;
  for (dom = maskP->dim - 1; dom >= 1; dom--)
    ind = ind * maskP->sizeA[dom] + coord[dom];

  setP = maskP->spanSetA + ind;

  /* binary search */
  jl = -1;
  ju = setP->spanNo;
  while (ju - jl > 1) {
    jm = (jl + ju) / 2;
    if (coord[0] >= setP->spanA[jm].startI)
      jl = jm;
    else
      ju = jm;
  }

  if (jl < 0)
    return FALSE;

  return (coord[0] <= setP->spanA[jl].endI);
}

BOOL
AppMaskIsEmpty(AppMaskP maskP)
{
  int spanSetNo;
  int dom, i;

  spanSetNo = 1;
  for (dom = 1; dom < maskP->dim; dom++)
    spanSetNo *= maskP->sizeA[dom];

  for (i = 0; i < spanSetNo; i++)
    if (maskP->spanSetA[i].spanNo > 0)
      return FALSE;

  return TRUE;
}

void
AppMaskGetRange(AppMaskP maskP, int coord[], int *leftP, int *rightP)
{
  SpanSetDescr *setP;
  int dom, ind;

  if (maskP == NULL) {  /* NULL is considered an empty mask */
    *leftP = 1;
    *rightP = 0;
    return;
  }

  for (dom = 1; dom < maskP->dim; dom++)
    if (coord[dom] < 0 || coord[dom] >= maskP->sizeA[dom]) {
      *leftP = 1;
      *rightP = 0;
      return;
    }

  ind = 0;
  for (dom = maskP->dim - 1; dom >= 1; dom--)
    ind = ind * maskP->sizeA[dom] + coord[dom];

  setP = maskP->spanSetA + ind;

  if (setP->spanNo == 0) {
    *leftP = 1;
    *rightP = 0;
  } else {
    *leftP = setP->spanA[0].startI;
    *rightP = setP->spanA[setP->spanNo - 1].endI;
  }
}

void
AppMaskDestroy(AppMaskP maskP)
{
  int spanSetNo;
  int dom, i;

  spanSetNo = 1;
  for (dom = 1; dom < maskP->dim; dom++)
    spanSetNo *= maskP->sizeA[dom];

  for (i = 0; i < spanSetNo; i++)
    if (maskP->spanSetA[i].spanNo > 0)
      free(maskP->spanSetA[i].spanA);

  free(maskP->spanSetA);
  free(maskP);
}
