/*
************************************************************************
*
*   ShapeList.c - manage list of shapes
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/13
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.ShapeList.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <app/shape_list.h>

#include <stdio.h>
#include <stdlib.h>

#include <linlist.h>
#include <cluster.h>

typedef struct {
  struct AppShapeS d;  /* must be first member, type cast! */
  AppSpectrumP specP;
} ShapeInfo;

typedef struct {
  float maxDiff;
  int clusterNo;
  ShapeInfo *firstP;
} ClusterData;

static LINLIST ShapeList = NULL;

AppShapeP
AppAddShape(AppSpectrumP specP, AppShapeP shapeP)
{
  ShapeInfo infoS;
  int i;

  if (ShapeList == NULL)
    ShapeList = ListOpen(sizeof(ShapeInfo));

  infoS.d = *shapeP;
  infoS.d.valueA = malloc(infoS.d.valueNo * sizeof(*infoS.d.valueA));
  infoS.d.errorA = malloc(infoS.d.valueNo * sizeof(*infoS.d.errorA));
  for (i = 0; i < shapeP->valueNo; i++) {
    infoS.d.valueA[i] = shapeP->valueA[i];
    infoS.d.errorA[i] = shapeP->errorA[i];
  }

  infoS.specP = specP;

  return ListInsertLast(ShapeList, &infoS);
}

void
AppShapeApply(AppSpectrumP specP, AppShapeApplyF applyF, void *clientData)
{
  ShapeInfo *infoP, *nextInfoP;

  infoP = ListFirst(ShapeList);
  while (infoP != NULL) {
    nextInfoP = ListNext(ShapeList, infoP);  /* infoP may be removed */
    if (infoP->specP == specP)
      applyF((AppShapeP) infoP, clientData);
    infoP = nextInfoP;
  }
}

static int
compareShape(void *p1, void *p2)
{
  ShapeInfo *info1P = p1;
  ShapeInfo *info2P = p2;

  if (info1P->d.startIndex < info2P->d.startIndex)
    return -1;

  if (info1P->d.startIndex > info2P->d.startIndex)
    return 1;
  
  return 0;
}

static void
buildCluster(int idx, int clusterSize, void *entryP,
    float clusterDiff, float otherDiff, void *clientData)
{
  ShapeInfo *infoP = entryP;
  ClusterData *dataP = clientData;

  if (idx == 0) {
    infoP->d.clusterDiff = clusterDiff / dataP->maxDiff;
    if (otherDiff < infoP->d.otherDiff * dataP->maxDiff)
      infoP->d.otherDiff = otherDiff / dataP->maxDiff;

    dataP->firstP = infoP;
    dataP->clusterNo++;
  } else {
    AppShapeCombine((AppShapeP) dataP->firstP, (AppShapeP) infoP);
    AppRemoveShape((AppShapeP) infoP);
  }
}

int
AppClusterShapes(AppSpectrumP specP, int domain,
    int clusterNo, float minDiff, float maxDiff)
{
  int shapeNo;
  ShapeInfo *infoP;
  ShapeInfo **infoPA;
  ClusterEntryP *entryPA;
  float diff;
  ClusterData clusterData;
  int idx, idx1, idx2, endIdx;

  shapeNo = 0;
  infoP = ListFirst(ShapeList);
  while (infoP != NULL) {
    if (infoP->specP == specP && infoP->d.domain == domain)
      shapeNo++;
    infoP = ListNext(ShapeList, infoP);
  }
  if (shapeNo == 0)
    return 0;

  ListSort(ShapeList, compareShape);

  infoPA = malloc(shapeNo * sizeof(*infoPA));
  entryPA = malloc(shapeNo * sizeof(*entryPA));

  infoP = ListFirst(ShapeList);
  idx = 0;
  while (infoP != NULL) {
    if (infoP->specP == specP && infoP->d.domain == domain) {
      infoP->d.otherDiff = 100.0;
      infoPA[idx] = infoP;
      entryPA[idx] = ClusterAddEntry(infoP);
      idx++;
    }
    infoP = ListNext(ShapeList, infoP);
  }

  for (idx1 = 0; idx1 < shapeNo - 1; idx1++) {
    endIdx = infoPA[idx1]->d.startIndex + infoPA[idx1]->d.valueNo;
    for (idx2 = idx1 + 1; idx2 < shapeNo; idx2++) {
      if (infoPA[idx2]->d.startIndex >= endIdx)
	break;
      diff = AppShapeCompare((AppShapeP) infoPA[idx1],
	  (AppShapeP) infoPA[idx2]);
      if (diff <= maxDiff) {
        ClusterAddPair(entryPA[idx1], entryPA[idx2], diff);
      } else {
	if (diff < infoPA[idx1]->d.otherDiff * maxDiff)
	  infoPA[idx1]->d.otherDiff = diff / maxDiff;
	if (diff < infoPA[idx2]->d.otherDiff * maxDiff)
	  infoPA[idx2]->d.otherDiff = diff / maxDiff;
      }
    }
  }

  free(infoPA);
  free(entryPA);

  clusterData.maxDiff = maxDiff;
  clusterData.clusterNo = 0;
  ClusterCalc(clusterNo, minDiff, buildCluster, &clusterData);

  return clusterData.clusterNo;
}

void
AppRemoveShape(AppShapeP shapeP)
{
  AppShapeDestroy(shapeP);
  ListRemove(ShapeList, shapeP);
}
