/*
************************************************************************
*
*   RegShape.c - calculate line shapes of spectrum region
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/21
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.RegShape.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <app/reg_shape.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>

#include "conv.h"
#include "shape.h"

static void
getFactor(float *v1, BOOL *validA, float *noiseA, float *v2, int n,
    float *factP, float *factErrP, float *fitErrP)
{
  float a12, a22;
  float s, factErr, fitErr, d;
  int validNo, i;

  validNo = 0;
  a12 = 0.0;
  a22 = 0.0;

  for (i = 0; i < n; i++) {
    if (! validA[i])
      continue;

    validNo++;
    a12 += v1[i] * v2[i];
    a22 += v2[i] * v2[i];
  }
  if (validNo == 0)
    s = 0.0;
  else
    s = a12 / a22;

  factErr = 0.0;
  fitErr = 0.0;
  for (i = 0; i < n; i++) {
    if (! validA[i])
      continue;

    d = s * v2[i] - v1[i];
    if (d < 0.0)
      d = - d;

    factErr += v2[i] * v2[i] * (d + noiseA[i]) * (d + noiseA[i]);

    if (d > fitErr)
      fitErr = d;
  }

  factErr = sqrt(factErr) / a22;

  *factP = s;
  *factErrP = factErr;
  *fitErrP = fitErr;
}

void
AppCalcShapes(AppRegionDataP regDataP, float *fullNoiseA,
    AppShapeP shapeA, float *ampP, float *errP)
{
  float *fullDataA, *dataA, *subDataA, *shapeVal;
  BOOL *validA, *subValidA;
  float *noiseA, *subNoiseA;
  int fullSizeA[APP_MAX_DIM], sizeA[APP_MAX_DIM], coord[APP_MAX_DIM];
  int totSize, minSize;
  BOOL hasPoints;
  float sf, factErr, fitErr, prod;
  int dom, step, fullInd, ind, subInd, i;

  fullDataA = ConvGetRegionFloat(regDataP);

  totSize = 1;
  hasPoints = TRUE;
  for (i = 0; i < regDataP->dimensionNo; i++) {
    fullSizeA[i] = regDataP->rangeA[i][1] - regDataP->rangeA[i][0] + 1;
    sizeA[i] = regDataP->subRangeA[i][1] - regDataP->subRangeA[i][0] + 1;
    if (sizeA[i] < 0) {
      hasPoints = FALSE;
      break;
    }

    totSize *= sizeA[i];
    if (i == 0 || sizeA[i] < minSize)
      minSize = sizeA[i];
  }

  if (! hasPoints) {
    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      shapeA[dom].valueNo = 0;
      shapeA[dom].valueA = NULL;
      shapeA[dom].errorA = NULL;
    }
    return;
  }

  for (i = 0; i < regDataP->dimensionNo; i++)
    coord[i] = 0;
  
  dataA = malloc(totSize * sizeof(*dataA));
  validA = malloc(totSize * sizeof(*validA));
  noiseA = malloc(totSize * sizeof(*noiseA));

  ind = 0;
  for (;;) {
    fullInd = 0;
    for (i = regDataP->dimensionNo - 1; i >= 0; i--)
      fullInd = fullInd * fullSizeA[i] + coord[i] + regDataP->subRangeA[i][0];

    dataA[ind] = fullDataA[fullInd];
    validA[ind] = regDataP->validA[fullInd];
    if (fullNoiseA == NULL)
      noiseA[ind] = regDataP->noiseMagnitudeMax;
    else
      noiseA[ind] = fullNoiseA[fullInd];

    for (i = 0; i < regDataP->dimensionNo; i++) {
      if (coord[i] == sizeA[i] - 1) {
	coord[i] = 0;
      } else {
	coord[i]++;
	break;
      }
    }

    if (i == regDataP->dimensionNo)
      break;

    ind++;
  }

  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    shapeA[dom].dimensionNo = regDataP->dimensionNo;
    shapeA[dom].domain = dom;
    shapeA[dom].startIndex = regDataP->rangeA[dom][0] +
	regDataP->subRangeA[dom][0];
    shapeA[dom].valueNo = sizeA[dom];
    shapeA[dom].origLeft = 0;
    shapeA[dom].origValueNo = shapeA[dom].valueNo;
    shapeA[dom].valueA = malloc(sizeA[dom] * sizeof(*shapeA[dom].valueA));
    shapeA[dom].errorA = malloc(sizeA[dom] * sizeof(*shapeA[dom].errorA));
    for (i = 0; i < sizeA[dom]; i++) {
      shapeA[dom].valueA[i] = 0.0;
      shapeA[dom].errorA[i] = 0.0;
    }
    shapeA[dom].pointerList = NULL;
  }

  for (i = 0; i < regDataP->dimensionNo; i++)
    coord[i] = 0;
  
  ind = 0;
  for (;;) {
    if (validA[ind])
      for (i = 0; i < regDataP->dimensionNo; i++)
	if (dataA[ind] > shapeA[i].valueA[coord[i]])
	  shapeA[i].valueA[coord[i]] = dataA[ind];

    for (i = 0; i < regDataP->dimensionNo; i++) {
      if (coord[i] == sizeA[i] - 1) {
	coord[i] = 0;
      } else {
	coord[i]++;
	break;
      }
    }

    if (i == regDataP->dimensionNo)
      break;

    ind++;
  }

  for (i = 0; i < regDataP->dimensionNo; i++)
    (void) ShapeNorm(shapeA + i);

  subDataA = malloc((totSize / minSize) * sizeof(*subDataA));
  subValidA = malloc((totSize / minSize) * sizeof(*subValidA));
  subNoiseA = malloc((totSize / minSize) * sizeof(*subNoiseA));
  shapeVal = malloc(totSize * sizeof(*shapeVal));

  for (step = 0; step < 3; step++) {
    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      for (coord[dom] = 0; coord[dom] < sizeA[dom]; coord[dom]++) {
	for (i = 0; i < regDataP->dimensionNo; i++)
	  if (i != dom)
	    coord[i] = 0;

	subInd = 0;
	for (;;) {
	  ind = 0;
	  for (i = regDataP->dimensionNo - 1; i >= 0; i--)
	    ind = ind * sizeA[i] + coord[i];

	  subValidA[subInd] = validA[ind];
	  subDataA[subInd] = dataA[ind];
	  subNoiseA[subInd] = noiseA[ind];

	  shapeVal[subInd] = 1.0;
	  for (i = 0; i < regDataP->dimensionNo; i++)
	    if (i != dom)
	      shapeVal[subInd] *= shapeA[i].valueA[coord[i]];

	  for (i = 0; i < regDataP->dimensionNo; i++) {
	    if (i == dom)
	      continue;

	    if (coord[i] == sizeA[i] - 1) {
	      coord[i] = 0;
	    } else {
	      coord[i]++;
	      break;
	    }
	  }

	  if (i == regDataP->dimensionNo)
	    break;
	  
	  subInd++;
	}

	getFactor(subDataA, subValidA, subNoiseA,
	    shapeVal, totSize / sizeA[dom], &sf, &factErr, &fitErr);
	shapeA[dom].valueA[coord[dom]] = sf;
	shapeA[dom].errorA[coord[dom]] = factErr;
      }

      (void) ShapeNorm(shapeA + dom);
    }
  }

  for (i = 0; i < regDataP->dimensionNo; i++)
    coord[i] = 0;

  ind = 0;
  for (;;) {
    if (validA[ind]) {
      prod = 1.0;
      for (i = 0; i < regDataP->dimensionNo; i++)
	prod *= shapeA[i].valueA[coord[i]];

      shapeVal[ind] = prod;
    }

    for (i = 0; i < regDataP->dimensionNo; i++) {
      if (coord[i] == sizeA[i] - 1) {
	coord[i] = 0;
      } else {
	coord[i]++;
	break;
      }
    }

    if (i == regDataP->dimensionNo)
      break;

    ind++;
  }

  getFactor(dataA, validA, noiseA,
      shapeVal, totSize, ampP, &factErr, errP);

  free(dataA);
  free(validA);
  free(noiseA);
  free(subValidA);
  free(subDataA);
  free(subNoiseA);
  free(shapeVal);
}

void
AppExtendShape(AppRegionDataP regDataP, float *noiseA,
    AppShapeP shapeA, float amp,
    AppShapeP shapeP, int dom)
{
  float *dataA;
  int shapeSize;
  int sizeA[APP_MAX_DIM], offsA[APP_MAX_DIM], coord[APP_MAX_DIM];
  BOOL *validA;
  float *dataVal, *noiseVal, *shapeVal;
  int validNo, leftInc, rightInc;
  float sf, factErr, fitErr;
  int dim, ind, subInd, i;

  dim = regDataP->dimensionNo;
  dataA = ConvGetRegionFloat(regDataP);

  shapeSize = 1;
  for (i = 0; i < dim; i++) {
    sizeA[i] = regDataP->rangeA[i][1] - regDataP->rangeA[i][0] + 1;
    offsA[i] = - regDataP->rangeA[i][0];

    if (i == dom)
      continue;
    
    offsA[i] += shapeA[i].startIndex;
    shapeSize *= shapeA[i].valueNo;
  }

  validA = malloc(shapeSize * sizeof(*validA));
  dataVal = malloc(shapeSize * sizeof(*dataVal));
  noiseVal = malloc(shapeSize * sizeof(*noiseVal));
  shapeVal = malloc(shapeSize * sizeof(*shapeVal));

  for (i = 0; i < dim; i++)
    coord[i] = 0;

  subInd = 0;
  for (;;) {
    shapeVal[subInd] = 1.0;
    for (i = 0; i < dim; i++) {
      if (i == dom)
	continue;
    
      shapeVal[subInd] *= shapeA[i].valueA[coord[i]];
    }

    for (i = 0; i < dim; i++) {
      if (i == dom)
	continue;
      
      if (coord[i] == shapeA[i].valueNo - 1) {
	coord[i] = 0;
      } else {
	coord[i]++;
	break;
      }
    }

    if (i == dim)
      break;
    
    subInd++;
  }

  AppShapeComplete(shapeP,
      regDataP->rangeA[dom][0], regDataP->rangeA[dom][1], 0.0);

  coord[dom] = shapeA[dom].startIndex - 1;
  leftInc = 0;
  rightInc = 0;

  for (;;) {
    if (coord[dom] < regDataP->rangeA[dom][0])
      coord[dom] = shapeA[dom].startIndex + shapeA[dom].valueNo;
    if (coord[dom] > regDataP->rangeA[dom][1])
      break;

    for (i = 0; i < dim; i++) {
      if (i == dom)
	continue;

      coord[i] = 0;
    }

    validNo = 0;
    subInd = 0;
    for (;;) {
      ind = 0;
      for (i = dim - 1; i >= 0; i--)
	ind = ind * sizeA[i] + coord[i] + offsA[i];

      validA[subInd] = regDataP->validA[ind];
      dataVal[subInd] = dataA[ind];
      if (noiseA == NULL)
	noiseVal[subInd] = regDataP->noiseMagnitudeMax;
      else
	noiseVal[subInd] = noiseA[ind];

      if (validA[subInd])
	validNo++;

      for (i = 0; i < dim; i++) {
	if (i == dom)
	  continue;
	
	if (coord[i] == shapeA[i].valueNo - 1) {
	  coord[i] = 0;
	} else {
	  coord[i]++;
	  break;
	}
      }

      if (i == dim)
	break;
      
      subInd++;
    }

    if (validNo == 0) {
      if (coord[dom] < shapeA[dom].startIndex) {
	coord[dom] = shapeA[dom].startIndex + shapeA[dom].valueNo;
	continue;
      } else {
	break;
      }
    }

    getFactor(dataVal, validA, noiseVal,
	shapeVal, shapeSize, &sf, &factErr, &fitErr);
    sf /= amp;
    factErr /= amp;

    if (sf > shapeP->valueA[coord[dom] - shapeP->startIndex]) {
      if (coord[dom] < shapeA[dom].startIndex) {
	coord[dom] = shapeA[dom].startIndex + shapeA[dom].valueNo;
	continue;
      } else {
	break;
      }
    }

    shapeP->valueA[coord[dom] - shapeP->startIndex] = sf;
    shapeP->errorA[coord[dom] - shapeP->startIndex] = factErr;

    if (coord[dom] < shapeA[dom].startIndex) {
      leftInc++;
      coord[dom]--;
    } else {
      rightInc++;
      coord[dom]++;
    }
  }

  free(validA);
  free(dataVal);
  free(noiseVal);
  free(shapeVal);

  shapeP->origLeft -= leftInc;
  shapeP->origValueNo += leftInc + rightInc;

  AppShapeOriginal(shapeP);

  if (leftInc + rightInc > 0)
    AppShapeCalcPar(shapeP);
}

void
AppSubtractShapes(AppRegionDataP regDataP, AppShapeP shapeA, float amp)
{
  int sizeA[APP_MAX_DIM], offsA[APP_MAX_DIM], coord[APP_MAX_DIM];
  float prod;
  int ind, i;

  for (i = 0; i < regDataP->dimensionNo; i++) {
    sizeA[i] = regDataP->rangeA[i][1] - regDataP->rangeA[i][0] + 1;
    offsA[i] = shapeA[i].startIndex - regDataP->rangeA[i][0];
    coord[i] = 0;
  }
  
  for (;;) {
    ind = 0;
    for (i = regDataP->dimensionNo - 1; i >= 0; i--) {
      if (coord[i] + offsA[i] < 0 || coord[i] + offsA[i] >= sizeA[i])
	break;
      ind = ind * sizeA[i] + coord[i] + offsA[i];
    }

    if (i < 0 && regDataP->validA[ind]) {
      prod = amp;
      for (i = 0; i < regDataP->dimensionNo; i++)
	prod *= shapeA[i].valueA[coord[i]];

      if (regDataP->dataType == ADT_FLT32)
	((float *) regDataP->dataA)[ind] -= prod;
      else
	((int *) regDataP->dataA)[ind] -= (int) prod;
    }

    for (i = 0; i < regDataP->dimensionNo; i++) {
      if (coord[i] == shapeA[i].valueNo - 1) {
	coord[i] = 0;
      } else {
	coord[i]++;
	break;
      }
    }

    if (i == regDataP->dimensionNo)
      break;
  }
}
