/*
************************************************************************
*
*   Reg.c - management of spectrum regions
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/21
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.Reg.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <app/reg.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#include <memory.h>

#include "conv.h"
#include "reg_struc.h"

struct RegCopyS {
  AppDataP dataA;
  BOOL *validA;
  AppDataRange subRangeA[APP_MAX_DIM];
  LINLIST maxL;
  RegCopyP nextP;
};

AppRegionP
AppRegionNew(AppRegionDataP regDataP)
{
  AppRegionP regP;
  int i;

  regP = malloc(sizeof(*regP));

  regP->d = *regDataP;

  regP->boxSize = 1;
  for (i = 0; i < regP->d.dimensionNo; i++)
    regP->boxSize *= regP->d.rangeA[i][1] - regP->d.rangeA[i][0] + 1;
  
  regP->d.dataA = malloc(4 * regP->boxSize);
  (void) memcpy(regP->d.dataA, regDataP->dataA, 4 * regP->boxSize);

  regP->d.validA = malloc(regP->boxSize * sizeof(*regP->d.validA));
  (void) memcpy(regP->d.validA, regDataP->validA,
      regP->boxSize * sizeof(*regDataP->validA));

  regP->specP = NULL;
  regP->maxL = ListOpen(sizeof(struct AppMaxS));
  regP->noiseP = NULL;
  regP->segLevel = regP->d.noiseMagnitudeMax;
  regP->minLevel = regP->d.noiseMagnitudeMax;
  regP->copyP = NULL;

  return regP;
}

void
AppRegionPushState(AppRegionP regP, int mask)
{
  RegCopyP copyP;
  AppMaxP maxP;
  int i;

  copyP = malloc(sizeof(*copyP));

  copyP->nextP = regP->copyP;
  regP->copyP = copyP;

  if (mask & APP_PUSH_DATA) {
    copyP->dataA = regP->d.dataA;
    regP->d.dataA = malloc(4 * regP->boxSize);
    (void) memcpy(regP->d.dataA, copyP->dataA, 4 * regP->boxSize);
  } else {
    copyP->dataA = NULL;
  }

  if (mask & APP_PUSH_VALID) {
    copyP->validA = regP->d.validA;
    regP->d.validA = malloc(regP->boxSize * sizeof(*copyP->validA));
    (void) memcpy(regP->d.validA, copyP->validA,
	regP->boxSize * sizeof(*copyP->validA));
    for (i = 0; i < regP->d.dimensionNo; i++) {
      copyP->subRangeA[i][0] = regP->d.subRangeA[i][0];
      copyP->subRangeA[i][1] = regP->d.subRangeA[i][1];
    }
  } else {
    copyP->validA = NULL;
  }

  if (mask & APP_PUSH_MAX_LIST) {
    copyP->maxL = regP->maxL;
    regP->maxL = ListOpen(sizeof(struct AppMaxS));
    maxP = ListFirst(copyP->maxL);
    while (maxP != NULL) {
      (void) ListInsertLast(regP->maxL, maxP);
      maxP = ListNext(copyP->maxL, maxP);
    }
  } else {
    copyP->maxL = NULL;
  }
}

void
AppRegionPopState(AppRegionP regP)
{
  RegCopyP copyP;
  int i;

  copyP = regP->copyP;

  if (copyP->dataA != NULL) {
    free(regP->d.dataA);
    regP->d.dataA = copyP->dataA;
  }

  if (copyP->validA != NULL) {
    free(regP->d.validA);
    regP->d.validA = copyP->validA;
    for (i = 0; i < regP->d.dimensionNo; i++) {
      regP->d.subRangeA[i][0] = copyP->subRangeA[i][0];
      regP->d.subRangeA[i][1] = copyP->subRangeA[i][1];
    }
  }

  if (copyP->maxL != NULL) {
    ListClose(regP->maxL);
    regP->maxL = copyP->maxL;
  }

  regP->copyP = copyP->nextP;
  free(copyP);
}

void
RegionStrucFree(AppRegionP regP)
{
  RegCopyP copyP, nextCopyP;

  copyP = regP->copyP;
  while (copyP != NULL) {
    nextCopyP = copyP->nextP;

    if (copyP->dataA != NULL)
      free(copyP->dataA);
    if (copyP->validA != NULL)
      free(copyP->validA);
    if (copyP->maxL != NULL)
      ListClose(copyP->maxL);
    free(copyP);

    copyP = nextCopyP;
  }

  free(regP->d.dataA);
  free(regP->d.validA);
  ListClose(regP->maxL);
}

void
AppRegionDestroy(AppRegionP regP)
{
  RegionStrucFree(regP);
  free(regP);
}

BOOL
AppRegionIsInside(AppRegionDataP regDataP, int coord[])
{
  int ind, i;

  for (i = 0; i < regDataP->dimensionNo; i++)
    if (coord[i] < regDataP->rangeA[i][0] ||
	coord[i] > regDataP->rangeA[i][1])
      return FALSE;

  ind = 0;
  for (i = regDataP->dimensionNo - 1; i >= 0; i--)
    ind = ind * (regDataP->rangeA[i][1] - regDataP->rangeA[i][0] + 1) +
        coord[i] - regDataP->rangeA[i][0];
 
  return regDataP->validA[ind];
}

void
AppMaxApply(AppRegionP regP, AppMaxApplyF applyF, void *clientData)
{
  AppMaxP maxP;

  maxP = ListFirst(regP->maxL);
  while (maxP != NULL) {
    applyF(maxP, clientData);
    maxP = ListNext(regP->maxL, maxP);
  }
}

void
addMax(AppRegionP regP, int *coord, float amp, float noiseMagn)
{
  struct AppMaxS maxS;
  int i;

  for (i = 0; i < regP->d.dimensionNo; i++)
    maxS.coordinates[i] = coord[i];

  maxS.amplitude = amp;
  if (amp > regP->d.maxAmplitude)
    regP->d.maxAmplitude = amp;
  
  maxS.noiseMagnitude = noiseMagn;

  (void) ListInsertLast(regP->maxL, &maxS);
}

static int
compareMax(void *p1, void *p2)
{
  AppMaxP max1P = p1, max2P = p2;

  if (max1P->amplitude > max2P->amplitude)
    return -1;

  return 1;
}

void
RegionSortMax(AppRegionP regP)
{
  ListSort(regP->maxL, compareMax);
}

float *
RegionGetNoise(AppRegionP regP)
{
  float *noiseA;
  int coord[APP_MAX_DIM];
  int ind, i;

  noiseA = malloc(regP->boxSize * sizeof(*noiseA));

  if (regP->noiseP == NULL) {
    for (ind = 0; ind < regP->boxSize; ind++)
      noiseA[ind] = regP->d.noiseMagnitudeMax;

    return noiseA;
  }

  for (i = 0; i < regP->d.dimensionNo; i++)
    coord[i] = regP->d.rangeA[i][0];

  ind = 0;
  for (;;) {
    if (regP->d.validA[ind])
      noiseA[ind] = AppNoiseGetMagnitude(regP->noiseP, coord);
    else
      noiseA[ind] = 0.0;

    for (i = 0; i < regP->d.dimensionNo; i++) {
      if (coord[i] == regP->d.rangeA[i][1]) {
        coord[i] = regP->d.rangeA[i][0];
      } else {
        coord[i]++;
        break;
      }
    }

    if (i == regP->d.dimensionNo)
      break;
    
    ind++;
  }

  return noiseA;
}

void
AppMaxUpdate(AppRegionP regP)
{
  float *dataA;
  int incSum;
  int sizeA[APP_MAX_DIM], incA[APP_MAX_DIM];
  int coord[APP_MAX_DIM], neigh[APP_MAX_DIM];
  BOOL canBeMax;
  float noiseMagn, minLevel, amp;
  int ind, neighInd, i;

  regP->d.maxAmplitude = - MAXFLOAT;
  ListClose(regP->maxL);
  regP->maxL = ListOpen(sizeof(struct AppMaxS));

  for (i = 0; i < regP->d.dimensionNo; i++) {
    sizeA[i] = regP->d.rangeA[i][1] - regP->d.rangeA[i][0] + 1;
    if (sizeA[i] < 3)
      return;
  }

  dataA = ConvGetRegionFloat((AppRegionDataP) regP);

  incA[0] = 1;
  incSum = 1;
  for (i = 1; i < regP->d.dimensionNo; i++) {
    incA[i] = incA[i - 1] * sizeA[i - 1];
    incSum += incA[i];
  }

  for (i = 0; i < regP->d.dimensionNo; i++)
    coord[i] = regP->d.rangeA[i][0] + 1;

  ind = incSum;

  for (;;) {
    canBeMax = regP->d.validA[ind];

    if (canBeMax) {
      if (regP->noiseP == NULL) {
	noiseMagn = regP->d.noiseMagnitudeMax;
        minLevel = regP->minLevel;
      } else {
	noiseMagn = AppNoiseGetMagnitude(regP->noiseP, coord);
	minLevel = regP->minLevel * noiseMagn;
      }

      if (dataA[ind] < minLevel)
        canBeMax = FALSE;
    }

    for (i = 0; i < regP->d.dimensionNo; i++)
      neigh[i] = coord[i] - 1;
    neighInd = ind - incSum;

    while (canBeMax) {
      if (dataA[ind] < dataA[neighInd])
        canBeMax = FALSE;

      for (i = 0; i < regP->d.dimensionNo; i++) {
        if (neigh[i] == coord[i] + 1) {
          neigh[i] = coord[i] - 1;
          neighInd -= 2 * incA[i];
        } else {
          neigh[i]++;
          neighInd += incA[i];
          break;
        }
      }

      if (i == regP->d.dimensionNo)
        break;
    }

    if (canBeMax)
      addMax(regP, coord, dataA[ind], noiseMagn);

    for (i = 0; i < regP->d.dimensionNo; i++) {
      if (coord[i] == regP->d.rangeA[i][1] - 1) {
        coord[i] = regP->d.rangeA[i][0] + 1;
        ind -= (sizeA[i] - 3) * incA[i];
      } else {
        coord[i]++;
        ind += incA[i];
        break;
      }
    }

    if (i == regP->d.dimensionNo)
      break;
  }

  RegionSortMax(regP);
}
