/*
************************************************************************
*
*   PeakSymm.c - peak list symmetrization
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/13
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.PeakSymm.c
*   SCCS identification       : 1.5
*
************************************************************************
*/

#include <app/peak_symm.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>

#include "spec.h"
#include "conv.h"
#include "peak_info.h"

void
AppPeakSetSymmetric(AppSpectrumP specP, char *unit,
    int dom1, int dom2)
{
  LINLIST peakList;
  PeakInfo *info1P, *info2P;
  float shift1A[APP_MAX_DIM], shift2A[APP_MAX_DIM];
  float dist, d1, d2;

  if (dom1 == dom2 || dom1 >= specP->dim || dom2 >= specP->dim)
    return;

  peakList = PeakListGet();

  info1P = ListFirst(peakList);
  while (info1P != NULL) {
    if (info1P->specP == specP) {
      AppConvertSpectrumUnit(info1P->specP, info1P->d.positionA, "point",
	  shift1A, unit);
      info1P->d.symmetricDist = MAXFLOAT;

      /* stupid linear search, not very efficient... */
      info2P = ListFirst(peakList);
      while (info2P != NULL) {
	if (info2P->specP == specP) {
	  AppConvertSpectrumUnit(info2P->specP, info2P->d.positionA, "point",
	      shift2A, unit);
	  d1 = shift1A[dom1] - shift2A[dom2];
	  d2 = shift1A[dom2] - shift2A[dom1];
	  dist = d1 * d1 + d2 * d2;
	  if (dist < info1P->d.symmetricDist) {
	    info1P->d.symmetricDist = dist;
	    info1P->d.symmetricPeakP = (AppPeakP) info2P;
	  }
	}
	info2P = ListNext(peakList, info2P);
      }

      info1P->d.symmetricDist = sqrt(info1P->d.symmetricDist);
    }

    info1P = ListNext(peakList, info1P);
  }
}

static BOOL
getSymmVal(AppSpectrumP specP, char *unit, int dom1, int dom2,
    float *posA, AppNoiseP noiseP,
    float *valP, float *noiseLevelP)
{
  float shiftA[APP_MAX_DIM], symmShiftA[APP_MAX_DIM];
  float symmPosA[APP_MAX_DIM];
  AppDataRange rangeA[APP_MAX_DIM];
  AppDataP dataA;
  float *v;
  int rangeSize, dom, i;

  AppConvertSpectrumUnit(specP, posA, "point", shiftA, unit);

  for (dom = 0; dom < specP->dim; dom++)
    symmShiftA[dom] = shiftA[dom];

  symmShiftA[dom1] = shiftA[dom2];
  symmShiftA[dom2] = shiftA[dom1];

  AppConvertSpectrumUnit(specP, symmShiftA, unit, symmPosA, "point");

  rangeSize = 1;
  for (dom = 0; dom < specP->dim; dom++) {
    rangeA[dom][0] = (int) symmPosA[dom];
    if (rangeA[dom][0] < 0)
      break;

    rangeA[dom][1] = rangeA[dom][0] + 1;
    if (rangeA[dom][1] >= specP->sizeA[dom])
      break;

    rangeSize *= 2;
  }

  if (dom < specP->dim)
    return FALSE;

  dataA = malloc(4 * rangeSize);
  AppReadSpectrum(specP, rangeA, dataA);
  v = ConvGetFloat(dataA, rangeSize, specP->type);
  
  *valP = - MAXFLOAT;
  for (i = 0; i < rangeSize; i++)
    if (v[i] > *valP)
      *valP = v[i];

  free(dataA);

  *noiseLevelP = AppNoiseGetMagnitudeFloat(noiseP, symmPosA);

  return TRUE;
}

void
AppPeakSymmetrize(AppSpectrumP specP, char *unit,
    int dom1, int dom2, float dist, AppNoiseP noiseP, float noiseLevel)
{
  LINLIST peakList;
  PeakInfo *infoP;
  int peakNo, peakI;
  float *newQualA;
  float q1, q2, d, symmVal, noiseVal;

  peakList = PeakListGet();

  infoP = ListFirst(peakList);
  peakNo = 0;
  while (infoP != NULL) {
    if (infoP->specP == specP)
      peakNo++;

    infoP = ListNext(peakList, infoP);
  }

  /* We need to temporarily store the new qualities, and only set
     them at the end. Otherwise already newly calculated values would
     influence the calculation of others. */
  newQualA = malloc(peakNo * sizeof(*newQualA));

  infoP = ListFirst(peakList);
  peakI = 0;
  while (infoP != NULL) {
    if (infoP->specP == specP) {
      q1 = infoP->d.quality;
      q2 = infoP->d.symmetricPeakP->quality;
      d = infoP->d.symmetricDist / dist;

      if (d > 1.0) {
	if (getSymmVal(specP, unit, dom1, dom2, infoP->d.positionA, noiseP,
	    &symmVal, &noiseVal)) {
	  noiseVal *= noiseLevel;
	  symmVal += noiseVal;
	  if (symmVal < infoP->d.amplitude) {
	    if (symmVal < 0.0)
	      newQualA[peakI] = 0.0;
	    else
	      newQualA[peakI] = q1 * (symmVal / infoP->d.amplitude);
	  } else {
	    newQualA[peakI] = q1;
	  }
	} else {
	  newQualA[peakI] = q1;
	}
      } else {
	q2 *= 1.0 - d;
	newQualA[peakI] = q1 + q2 - q1 * q2;
      }

      peakI++;
    }

    infoP = ListNext(peakList, infoP);
  }

  infoP = ListFirst(peakList);
  peakI = 0;
  while (infoP != NULL) {
    if (infoP->specP == specP) {
      infoP->d.quality = newQualA[peakI];
      peakI++;
    }

    infoP = ListNext(peakList, infoP);
  }

  free(newQualA);
}
