/****************************************************************************/ 
/* INFIT 1.1                                                                */
/* Copyright (c) 2002 by Peter Guntert. All rights reserved.                */
/****************************************************************************/ 


#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define PI      3.14159265358979
#define PI2     1.570796327
#define TWOPI   6.28318530717959
#define MAXN    65536
#define TWOMAXN 131072
#define TMAX    0.2
#define FSHIFT  90.0
#define NFID    1024   
#define NZFID   2048
#define DT      0.0004
#define NPERIOD 20   
#define SMALL   0.005
#define SIGMA   0.02
#define MAXIT   20
#define STARTJ  0.5
#define ENDJ    12.0
#define STEPJ	23

#define nint(r) (((r)>=0.0) ? (int)((r)+0.5) : (int)((r)-0.5))

/* subroutines **************************************************************/

void simenv(float cc, float gamma, float *filter, float taq, float tau, int m, 
            int mz, int n1, int n2, int nz, int nenv, float *x, float *y, float *dy, 
            float *dg, float *data, float *datb); /* simulate envelope      */
void fft(float *data, int nn, int isign);         /* fast Fourier transform */

/****************************************************************************/

int main (int argc, char **argv)
{
    float data[TWOMAXN],datb[TWOMAXN],s[MAXN],
          ppmmin,ppmmax,hzmin,hzmax,dr,ppmhz,
          smin,smax,s1,s2,dt,tmax,t,u,xmin,ymin,
	  filter[MAXN],x[MAXN],y[MAXN],yfit[MAXN],dyfit[MAXN],dgfit[MAXN],
	  x0,x1,x2,x01,x02,x12,y0,y1,y2,cc,fwhm,gamma,center,taq,tzfid,
	  a11,a12,a22,b1,b2,a11t,a12t,a22t,b1t,b2t,chisq,chisqt,alamda,
	  cct,gammat,sigma[MAXN],ystart[MAXN],yt[MAXN],dcc,dgamma,dcct,
	  dgammat,cc0,gamma0,fshift,tau,jtest,gammaest;
    int   i,m,n,nin,n2,nr,nl,nz,it,mfid,mzfid,jit,chiplot;
/*    
    unsigned long t0;
*/
    char  filnam[80],grffil[80];
    float jhist[STEPJ],chihist[STEPJ],maxchi,minchi;
    FILE  *f;

    /* check command line parameters ****************************************/

/*
    t0=clock(); 
*/
    tmax=TMAX; tau=0.0; mfid=NFID; mzfid=NZFID; grffil[0]='\0'; 
    m=0;           
    chiplot=0;
    fshift=90.0;
    for (i=1;i<argc;i++) {
	if (!strcmp(argv[i],"-c")) { chiplot=1; }
	else if (!strcmp(argv[i],"-h")) {
	    m=0; break; }
	else if (!strcmp(argv[i],"-t")) {
	    if (argc>i+1) sscanf(argv[++i],"%f",&tmax); }
	else if (!strcmp(argv[i],"-f")) {
	    if (argc>i+1) sscanf(argv[++i],"%f",&fshift); }
	else if (!strcmp(argv[i],"-b")) {
	    if (argc>i+1) sscanf(argv[++i],"%f",&tau); }
	else if (!strcmp(argv[i],"-a")) {
	    if (argc>i+1) sscanf(argv[++i],"%d",&mfid);
	    if (argc>i+1) sscanf(argv[++i],"%d",&mzfid); }
	else if (!strcmp(argv[i],"-g")) {
	    if (argc>i+1) sscanf(argv[++i],"%s",grffil); }
	else { m++; if (m==1) sscanf(argv[i],"%s",filnam); else break; }}
    if (m!=1) { 
	printf(
	"Usage: infit [-ch] [-a <m> <n>] [-t <tmax>] [-f <phi>] [-b <tau>] "); 
	printf("[-g <grf-file>] <slc-file>\n\n");
        printf(
        "       -c          plot chi^2 versus J\n");
        printf(
        "       -h          print this help message\n");
	printf(
	"       <m>, <n>    Number of points in FID and after zero-filling\n");
	printf(
	"       <tmax>      Maximal time to consider [s]\n");
	printf(
	"       <phi>       Phase shift of window function [degrees]\n");
	printf(
	"       <tau>       Delay tau in Eq. [3] of the paper [s]\n");
        printf(
        "                   (set to 0 for NOESY or [15N,1H]-COSY with purge pulse)\n");
	printf(
	"       <grf-file>  Output file for the program GRAF (no extension)\n");
	printf(
	"       <slc-file>  Input data file from XEASY (no extension)\n\n");
	printf(
	"       Defaults: m = %d, n = %d, tmax = %.3f s, phi = %.1f deg, tau = 0\n",
	NFID,NZFID,TMAX,FSHIFT);
	printf(
	"                 grf-file = same as slc-file (extension .grf)\n\n");
	printf(
	"       Window-function: f(t) = sin ((180-phi)*(t/taq)+phi)\n");
	printf(
	"                        taq = aquisition time, phi = 180/SSB on X32\n");
	exit(1); }
    if (grffil[0]=='\0') strcpy(grffil,filnam); 
    strcat(filnam,".slc"); strcat(grffil,".grf");
/*    printf("filnam=%s, grffil=%s, tmax=%.3f, mfid=%d, mzfid=%d\n",
           filnam,grffil,tmax,mfid,mzfid);   */
    
    /* open input data file *************************************************/

    f=fopen(filnam,"r"); 
    if (f==0) {
	printf("Input file %s cannot be opened. Program aborted\n",filnam); 
	exit(1); }
    
    /* read input data file *************************************************/

    if (fscanf(f,"%*s %*s %*s %*s %*s %d",&n)!=1) { 
	printf("First line of data file cannot be read. Program aborted.\n");
	exit(1); }
    if (fscanf(f,"%*s %*s %*s %*s %*s %f %f",&ppmmin,&ppmmax)!=2) { 
	printf("Second line of data file cannot be read. Program aborted.\n");
	exit(1); }
    printf("Number of data points  : %d (from %s)\n",n,filnam);
    if (fscanf(f,"%*s %*s %*s %*s %*s %f %f",&hzmin,&hzmax)!=2) { 
	printf("Third line of data file cannot be read. Program aborted.\n");
	exit(1); }
    ppmmin=(-ppmmin); ppmmax=(-ppmmax); hzmin=(-hzmin); hzmax=(-hzmax); 
    printf("Frequency range        : %.3f...%.3f ppm (%.2f...%.2f Hz)\n",
            ppmmax,ppmmin,hzmax,hzmin);
    dr=(hzmax-hzmin)/(n-1); ppmhz=(ppmmax-ppmmin)/(hzmax-hzmin);
    printf(
      "Digital resolution     : %.2f Hz (spectrometer frequency %.2f MHz)\n",
      dr,1.0/ppmhz);
    for (i=1;i<=n;i++) if (fscanf(f,"%f",&s[i])!=1) {
	printf("Data point %d cannot be read. Program aborted.\n",i); 
	exit(1); }
    fclose(f);

    /* invert the slice if predominantly negative ***************************/

    t=0.0; for (i=1;i<=n;i++) t=t+s[i];
    if (t<0.0) {
	printf ("Warning: Slice is predominantly negative. Sign changed.\n");
	for (i=1;i<=n;i++) s[i]=(-s[i]); }

    /* baseline correction **************************************************/

    m=n/4; if (m<1) m=1;
    s1=1.0e20; for (i=1;i<=m;i++) if (s[i]<s1) s1=s[i];
    s2=1.0e20; for (i=n;i>n-m;i--) if (s[i]<s2) s2=s[i];
    t=(s2-s1)/(n-1); for (i=1;i<=n;i++) s[i]=s[i]-(s1+t*(i-1));
    smin=1.0e20; smax=-1.0e20;
    for (i=1;i<=n;i++) {
	if (s[i]<smin) smin=s[i]; if (s[i]>smax) smax=s[i]; }
    printf("Corrected data range   : %d...%d\n",(int)smin,(int)smax);

    /* estimate full width at half maximum **********************************/

    t=0.5*smax; 
    for (i=2;i<=n;i++) if (s[i]>t) break; 
    x1=(t-s[i-1]*i+s[i]*(i-1))/(s[i]-s[i-1]);
    for (i=n-1;i>=1;i--) if (s[i]>t) break; 
    x2=(t+s[i+1]*i-s[i]*(i+1))/(s[i+1]-s[i]); 
    fwhm=(x2-x1)*dr; center=0.5*(x1+x2); t=(center-1.0)*dr;
    printf("Peak center position   : %.3f ppm (%.2f Hz)\n",
           ppmmin+t*ppmhz,hzmin+t);
    nl=nint(center)-1; nr=n-nint(center); 	    

    /* open output file for GRAF ********************************************/
    
    f=fopen(grffil,"w");
    if (f==0) {
	printf("Input file %s cannot be opened. Program aborted\n",filnam); 
	exit(1); }

    /* write first part of the output file for GRAF *************************/

    fprintf(f,"#RECTANGLE y\n#LETTERSIZE 0.02\n");
    fprintf(f,"#VIEWPORT -0.9 0.9 0.1 1.0\n#MINMAX %.5f %.5f %.5e %.5e\n",
            ppmmin,ppmmax,smin-0.05*(smax-smin),smax+0.05*(smax-smin));
    fprintf(f,"#LEFTTEXT %.4f %.5e 1 %d data points from %s\n",
            ppmmin+0.03*(ppmmax-ppmmin),smin+0.97*(smax-smin),n,filnam);
    fprintf(f,"#LEFTTEXT %.4f %.5e 1 Digital resolution = %.2f Hz\n",
            ppmmin+0.03*(ppmmax-ppmmin),smin+0.91*(smax-smin),dr);
    fprintf(f,"#XTEXT ppm\n#REPRES d\n#CONNECT l\n1\n");
    for (i=1;i<=n;i++) fprintf(f,"%.5f %.1f\n",ppmmin+dr*ppmhz*(i-1),s[i]); 
	    
    /* apply zero-filling ***************************************************/

    nin=n; for (n=16;n<MAXN;n=n*2) if (n>nin && 1.0/(dr*n)<DT) break; 
    dt=1.0/(dr*n); tzfid=dt*(n-1);
    printf("Time resolution        : %.5f s\n",dt);
    printf("Time limit             : %.3f s (maximal time limit %.3f s)\n",
           tmax,tzfid);
    printf("Zero-filling to        : %d (%d points in time range)\n",
           n,(int)(tmax/dt)+1); 
    printf(
    "New peak frequency     : %.2f Hz (frequency range %.2f...%.2f Hz)\n",
    1.0/(NPERIOD*dt),-dr*(n/2),dr*(n/2));
    hzmin=1.0/(NPERIOD*dt)-0.5*(hzmax-hzmin); hzmax=hzmin+(nin-1)*dr;
    m=(int)(hzmin/dr)+n/2; for (i=nin;i>=1;i--) s[i+m]=s[i];
    for (i=1;i<=m;i++) s[i]=0.0; for (i=nin+m+1;i<=n;i++) s[i]=0.0;
      
    /* inverse fast Fourier transform ***************************************/

    n2=n/2; for (i=1;i<=n;i++) data[2*i]=0.0;
    for (i=1;i<=n2;i++) { data[2*i-1]=s[n2+i]; data[2*(n2+i)-1]=s[i]; }
    fft(data,n,-1); 
    
    /* Scale time domain data ***********************************************/

    nz=n; n=(int)(tmax/dt)+1;
    smax=-1.0e20; for (i=1;i<=n;i++) s[i]=data[2*i-1];
    for (i=1;i<=n;i++) if (fabs(s[i])>smax) smax=fabs(s[i]);
    if (smax!=0.0) for (i=1;i<=n;i++) s[i]=s[i]/smax;
    
    /* find envelope ********************************************************/

    m=1; x[1]=0.0; y[1]=1.0;
    for (i=2;i<n;i++) if (s[i]>s[i-1] && s[i]>s[i+1]) {
	t=s[i-1]-2.0*s[i]+s[i+1]; u=0.5*(s[i-1]-s[i+1]);
	x[++m]=dt*(u/t+(i-1)); y[m]=s[i]-0.5*u*u/t; }
    
    /* find first local minimum of envelope *********************************/

    for (i=3;i<=m-2;i++) 
        if (y[i]<y[i-1] && y[i]<y[i+1] && 
	    y[i-1]<y[i-2] && y[i+1]<y[i+2]) break;
    if (y[i]<y[i-1] && y[i]<y[i+1] && y[i-1]<y[i-2] && y[i+1]<y[i+2]) {
	x0=x[i-1]; x1=x[i]; x2=x[i+1]; y0=y[i-1]; y1=y[i]; y2=y[i+1];
	x01=x0-x1; x02=x0-x2; x12=x1-x2; t=x12*y0-x02*y1+x01*y2;
	xmin=0.5*((x1*x1-x2*x2)*y0+(x2*x2-x0*x0)*y1+(x0*x0-x1*x1)*y2)/t;
	ymin=y1+(xmin-x1)*(x12*x12*(y0-y1)+x01*x01*(y1-y2)+t*(xmin-x1))/
                (x01*x02*x12); }
    else {
	for (i=1;i<m;i++) if (y[i]<SMALL) break; xmin=x[i]; ymin=y[i]; }
    cc=0.5/xmin; gamma=PI*(fwhm-cc); cc0=cc; gamma0=gamma;
    printf("Estimated J-coupling   : %.2f Hz (height %.3f)\n",cc,ymin);
    printf("Estimated linewidth    : %.2f Hz (decay time %.4f s)\n",
               gamma/PI,1.0/gamma);
    gammaest=gamma;

    /* simulate envelope ****************************************************/

    taq=(tzfid/mzfid)*mfid;
    printf("Aquisition time for fit: %.3f s\n",taq);
    /* for (i=1;i<=mfid;i++) filter[i]=cos((PI2*(i-1))/(mfid-1)); */
	fshift=fshift*PI/180.0;
	for (i=1;i<=mfid;i++) filter[i]=sin(((PI-fshift)*(i-1))/(mfid-1)+fshift);

    if (chiplot) {
    /* determine chi square for different J's */
       printf("\nJ-coupling  linewidth     RMSD\n");
    for(jit=0;jit<STEPJ;jit++) {
       jtest=STARTJ+(ENDJ-STARTJ)/(float)STEPJ*jit;
       jhist[jit]=jtest;
       for (i=1;i<=m;i++) sigma[i]=SIGMA;
       alamda=10.0; cct=jtest; gammat=gamma; chisq=1.0e10;
       for (it=1;it<=MAXIT;it++) {
	simenv(cct,gammat,filter,taq,tau,mfid,mzfid,nl,nr,nz,m,
	        x,yt,dyfit,dgfit,data,datb);
	if (it==1) for (i=1;i<=m;i++) ystart[i]=yt[i];
	a11t=0.0; a12t=0.0; a22t=0.0; b1t=0.0; b2t=0.0; chisqt=0.0;
	for (i=1;i<=m;i++) {
	    t=1.0/sigma[i]; t=t*t; a11t=a11t+t*dyfit[i]*dyfit[i]; 
	    a22t=a22t+t*dgfit[i]*dgfit[i]; a12t=a12t+t*dyfit[i]*dgfit[i];
	    if (yt[i]>0.0) u=y[i]-yt[i]; else u=(-y[i]-yt[i]); 
	    t=t*u; b1t=b1t+t*dyfit[i]; b2t=b2t+t*dgfit[i]; 
	    chisqt=chisqt+t*u; }
        if (chisqt<chisq) {
	    alamda=0.1*alamda; for (i=1;i<=m;i++) yfit[i]=yt[i];
	    gamma=gammat; t=chisq; chisq=chisqt;
	    a11=a11t; a12=a12t; a22=a22t; b1=b1t; b2=b2t;  
	    dcc=a22/(a11*a22-a12*a12); dgamma=a11/(a11*a22-a12*a12);
            if (chisqt/m<0.001) break;
	    if ((t-chisqt)/t<0.01) break;
        }
	else { alamda=10.0*alamda; }
        t=1.0+alamda; a11t=a11*t; a22t=a22*t; t=1.0/(a11t*a22t-a12*a12);
	gammat=gamma+t*(a11t*b2-a12*b1);
       }
       chihist[jit]=sqrt(chisqt/m);
       if(jit==0||chihist[jit]<minchi) {
          minchi=chihist[jit];
          cc=cct;
          gammaest=gammat;
          cc0=cc; gamma0=gammaest;
       }
       if(jit==0||chihist[jit]>maxchi) maxchi=chihist[jit];
       printf("%8.3f %10.3f %12.2e\n",
	       cct,gammat/PI,sqrt(chisqt/m));
    }}

    /* find optimal J */
    gamma=gammaest;cc0=cc;
    for (i=1;i<=m;i++) sigma[i]=SIGMA;
    printf("\n   J-coupling  linewidth     RMSD        |grad|\n");
    alamda=10.0; cct=cc; gammat=gamma; chisq=1.0e10;
    for (it=1;it<=MAXIT;it++) {
	simenv(cct,gammat,filter,taq,tau,mfid,mzfid,nl,nr,nz,m,
	        x,yt,dyfit,dgfit,data,datb);
	if (it==1) for (i=1;i<=m;i++) ystart[i]=yt[i];
	a11t=0.0; a12t=0.0; a22t=0.0; b1t=0.0; b2t=0.0; chisqt=0.0;
	for (i=1;i<=m;i++) {
	    t=1.0/sigma[i]; t=t*t; a11t=a11t+t*dyfit[i]*dyfit[i]; 
	    a22t=a22t+t*dgfit[i]*dgfit[i]; a12t=a12t+t*dyfit[i]*dgfit[i];
	    if (yt[i]>0.0) u=y[i]-yt[i]; else u=(-y[i]-yt[i]); 
	    t=t*u; b1t=b1t+t*dyfit[i]; b2t=b2t+t*dgfit[i]; 
	    chisqt=chisqt+t*u; }
	printf("%2d %8.3f %10.3f %12.2e %12.2e",
	       it,cct,gammat/PI,sqrt(chisqt/m),sqrt((b1t*b1t+b2t*b2t)/m));
        if (chisqt<chisq) {
	    alamda=0.1*alamda; for (i=1;i<=m;i++) yfit[i]=yt[i];
	    cc=cct; gamma=gammat; t=chisq; chisq=chisqt;
	    a11=a11t; a12=a12t; a22=a22t; b1=b1t; b2=b2t;  
	    dcc=a22/(a11*a22-a12*a12); dgamma=a11/(a11*a22-a12*a12);
            if (chisqt/m<0.001) { printf("  RMSD < 0.001\n"); break; }
	    if ((t-chisqt)/t<0.01) {
		printf("  RMSD change < 1%%\n"); break; }
	    else printf("  successful\n"); }
	else { printf("  not successful\n"); alamda=10.0*alamda; }
        t=1.0+alamda; a11t=a11*t; a22t=a22*t; t=1.0/(a11t*a22t-a12*a12);
	cct=cc+t*(a22t*b1-a12*b2); gammat=gamma+t*(a11t*b2-a12*b1); }
    printf("\nFitted J-coupling      : %.2f +/- %.2f Hz\n",cc,sqrt(dcc));
    printf("Fitted linewidth       : %.2f +/- %.2f Hz\n",
           gamma/PI,sqrt(dgamma)/PI);

    /* set sign of experimental envelope ************************************/

    for (i=1;i<=m;i++) if (fabs(y[i]-yfit[i])>fabs(y[i]+yfit[i])) y[i]=(-y[i]);

    /* write second part of the output file for GRAF ************************/

    smin=0.0; for (i=1;i<=m;i++) if (yfit[i]<smin) smin=yfit[i]; u=1.1-smin;
    fprintf(f,"#NEXTPLOT\n#VIEWPORT -0.9 0.9 -1.0 -0.1\n");
    fprintf(f,"#MINMAX 0 %.5f %.3f 1.1\n",tmax,smin-0.1);
    fprintf(f,"#LEFTTEXT %.4f %.3f 1 Experiment (from %s)\n",
              0.4*tmax,1.1-0.1*u,filnam);
    fprintf(f,"#LEFTTEXT %.4f %.3f 1 Estimate: J = %.2f Hz, FWHM = %.2f Hz\n",
              0.4*tmax,1.1-0.17*u,cc0,gamma0/PI);
    fprintf(f,"#LEFTTEXT %.4f %.3f 1 Fit: ",0.4*tmax,1.1-0.24*u);
    fprintf(f,"J = %.2f +/- %.2f Hz, FWHM = %.2f +/- %.2f Hz\n",
              cc,sqrt(dcc),gamma/PI,sqrt(dgamma)/PI);
    fprintf(f,"#LEFTTEXT %.4f %.3f 1 RMS deviation = %.3f\n",
              0.4*tmax,1.1-0.31*u,sqrt(chisq/m));
    for (i=9;i>=2;i--) {
	t=0.5/i; if (t<tmax) fprintf(f,"#CENTERTEXT %.3f 1.15 1 %d\n",t,i); 
	if (0.5/(i-1)>tmax) break; }  
    fprintf(f,"#LEFTTEXT %.3f 1.15 1 Hz\n#XTEXT sec\n",t+tmax*0.05);
/*    for (i=1;i<=n;i++) fprintf(f,"%.5f %.4e\n",dt*(i-1),s[i]); */
    fprintf(f,"#REPRES -\n#CONNECT l\n#LINES 012\n3\n");
    for (i=1;i<=m;i++)
        fprintf(f,"%.5f %.4f %.4f %.4f\n",x[i],y[i],ystart[i],yfit[i]);
    fprintf(f,"#NEXTCURVE\n1\n0 0\n%.4f 0\n",tmax);	
    for (i=2;i<=12;i++) {
	t=0.5/i; u=0.5/(i+0.5);
	if (t<tmax) fprintf(f,"#NEXTCURVE\n1\n%.4f 1.06\n%.4f 1.0999\n",t,t);
	if (u<tmax) fprintf(f,"#NEXTCURVE\n1\n%.4f 1.08\n%.4f 1.0999\n",u,u); }
    fprintf(f,"#NEXTCURVE\n#LINES 012\n3\n"); u=1.1-smin;
    fprintf(f,"%.4f %.4f %.4f %.4f\n%.4f %.4f %.4f %.4f\n",
            0.30*tmax,1.1-0.1*u,1.1-0.17*u,1.1-0.24*u,	
            0.37*tmax,1.1-0.1*u,1.1-0.17*u,1.1-0.24*u);

    /* write third part of the output file for GRAF ************************/

    if (chiplot) {
    fprintf(f,"#NEXTPLOT\n#VIEWPORT -0.0 0.65 -0.83 -0.42\n");
    fprintf(f,"#LETTERSIZE 0.035\n");
    fprintf(f,"#MINMAX %.1f %.1f %.5f %.5f\n",STARTJ,ENDJ,
       0.0,maxchi+0.01*maxchi);
    fprintf(f,"#XTEXT Hz\n#YTEXT RMSD\n");
/*    fprintf(f,"#LEFTTEXT %.4f %.5f 1 RMS deviation\n",
            STARTJ+0.03*(ENDJ-STARTJ),0.97*maxchi);
*/
    fprintf(f,"#REPRES -\n#CONNECT l\n#LINES 0\n1\n");
    for (i=0;i<STEPJ;i++)
        fprintf(f,"%.2f %.4f\n",jhist[i],chihist[i]); }
/*
    t=1.0e-6*(float)(clock()-t0);
    printf("CPU time used          : %.2f s\n",t);
*/
    exit(0); }


/****************************************************************************/ 
/* Simulate envelope and its derivative with respect to the coupling const. */
/* PG, 28-6-1990                                                            */
/****************************************************************************/ 
    
void simenv(float cc, float gamma, float *filter, float taq, float tau, int m, 
            int mz, int n1, int n2, int nz, int nenv, float *x, float *y, float *dy, 
            float *dg, float *data, float *datb)

    /* cc        coupling constant [Hz]                              */
    /* gamma     inverse relaxation time [Hz]                        */
    /* filter    filter[i] = filter function at time taq*(i-1)/(n-1) */
    /* taq       aquisition time [s]                                 */
    /* tau       delay of Eq. [3] [s]                                */
    /* x         x[i] = time of point i on the envelope              */
    /* y         y[i] = point i on the envelope                      */
    /* dy        dy[i] = point i on the derivative of the envelope   */
    /* dg        dy[i] = point i on the derivative of the envelope   */
    /* data      workspace of size >= 2*max(mz,nz)+2                 */
    /* datb      workspace of size >= 2*max(mz,nz)+2                 */
    /* m         data points of FID                                  */
    /* mz        data points of zero-filled FID                      */
    /* n1        data points to the left (n1) and right (n2) of the  */
    /* n2        central frequency domain data point                 */
    /* nz        data points in frequency domain after zero-filling  */
    /* nenv      number of points on the envelope                    */
    
    {
    float  dt,c,t,u,ct,s1,s2,cost,sint,costau,sintau;
    int    i,j,k,m2,nn;

    /* simulate FID and its derivative with respect to the J-coupling (cc) */

    dt=taq/(m-1); c=PI*cc; data[1]=0.5*filter[1]; data[2]=0.0;
    datb[1]=0.0; for (i=2;i<=2*m;i=i+2) datb[i]=0.0;
    costau=cos(c*tau); sintau=sin(c*tau);
    for (i=2;i<=m;i++) {
	t=dt*(i-1); u=exp(-gamma*t)*filter[i]; ct=c*t;  
/*	data[2*i-1]=cos(ct)*u; data[2*i]=(-PI)*t*sin(ct)*u; */
        cost=cos(ct); sint=sin(ct);
        data[2*i-1]=(cost*costau-sint*sintau)*u;
        data[2*i]=-PI*(t+tau)*(sintau*cost+costau*sint)*u;
 	datb[2*i-1]=(-t)*data[2*i-1]; }

    /* zero-fill FID *******************************************************/

    for (i=2*m+1;i<=2*mz;i++) { data[i]=0.0; datb[i]=0.0; }

    /* Fourier transformation of the FID ***********************************/

    fft(data,mz,1); fft(datb,mz,1); 
    for (i=2;i<=2*mz;i=i+2) datb[i]=0.0;

    /* Get real parts of the transformed FID and its derivative ************/

    m2=2*(mz+2);
    for (i=4;i<=mz;i=i+2) {
	j=m2-i; data[i]=0.5*(data[i]+data[j]); data[j]=data[i]; 
	data[i-1]=0.5*(data[i-1]+data[j-1]); data[j-1]=data[i-1]; }

    /* shift zero frequency to the center of the spectrum ******************/

    for (i=1;i<=mz;i++) { 
	t=data[i]; data[i]=data[mz+i]; data[mz+i]=t;
	t=datb[i]; datb[i]=datb[mz+i]; datb[mz+i]=t; }

    /* linear baseline correction of the peak region in the spectrum *******/

/*printf("%d %d",n1,n2);*/
    n1=mz/2+1-n1; n2=mz/2+1+n2; nn=(n2-n1+1)/4; if (nn<1) nn=1;
    s1=1.0e20; for (i=n1;i<=n1+nn;i++) if (data[2*i-1]<s1) s1=data[2*i-1];
    s2=1.0e20; for (i=n2;i>=n2-nn;i--) if (data[2*i-1]<s2) s2=data[2*i-1];
    t=(s2-s1)/(n2-n1);
    for (i=n1;i<=n2;i++) data[2*i-1]=data[2*i-1]-(s1+t*(i-n1));
/*printf(" %d %d\n",n1,n2);
for (i=n1;i<=n2;i++) printf("%3d %4d %10.2f\n",i-n1+1,i,data[2*i-1]);*/

    /* zero-fill spectrum to the left and to the right *********************/
   
    nn=(nz-mz)/2+n1-1; j=2*(nn+1-n1);
    if (nn>n1) for (i=2*n2;i>=2*n1-1;i--) {
	data[j+i]=data[i]; datb[j+i]=datb[i]; }
    else if (nn<n1) for (i=2*n1-1;i<=2*n2;i++) {
	data[j+i]=data[i]; datb[j+i]=datb[i]; }
    for (i=1;i<=2*nn;i++) { data[i]=0.0; datb[i]=0.0; }
    for (i=2*(nn+n2-n1+2)-1;i<=2*nz;i++) { data[i]=0.0; datb[i]=0.0; }
/*printf("%d %d\n",n1,n2);
for (i=nz/2+1-10;i<=nz/2+1+10;i++) 
printf("%3d %4d %10.1f\n",i-nz/2-1,i,data[2*i-1]);     */

    /* restore original order of frequencies and Fourier tranform **********/

    for (i=1;i<=nz;i++) { 
	t=data[i]; data[i]=data[nz+i]; data[nz+i]=t;
	t=datb[i]; datb[i]=datb[nz+i]; datb[nz+i]=t; }
    fft(data,nz,-1); fft(datb,nz,-1);

    /* normalize envelope and get real parts of it and its derivative ******/

    nn=2*(nz+2); t=1.0/data[1]; data[1]=1.0; data[2]=data[2]*t;
    for (i=1;i<2*nz;i=i+2) datb[i]=datb[i]*t;
    data[nz+1]=data[nz+1]*t; data[nz+2]=data[nz+2]*t; t=0.5*t;
    for (i=4;i<=nz;i=i+2) {
	j=nn-i; data[i]=t*(data[i]+data[j]); data[j]=data[i]; 
	data[i-1]=t*(data[i-1]+data[j-1]); data[j-1]=data[i-1]; }
    t=data[2]; for (i=2;i<=2*nz;i=i+2) data[i]=data[i]-data[i-1]*t;
    t=datb[1]; for (i=1;i<=2*nz;i=i+2) datb[i]=datb[i]-data[i]*t;
/*for (i=1020;i<=1030;i++) printf("%d %.4f %.4f\n",i,data[2*i-1],data[2*i]);*/

    /* get points on the envelope by quadratic interpolation ***************/

    dt=(dt*mz)/nz; 
    for (i=1;i<=nenv;i++) {
	t=1.0+x[i]/dt; j=nint(t);
	if (j>1 && j<nz) {
	    k=2*j-1;
	    y[i]=data[k]+0.5*(t-j)*((t-j)*(data[k-2]-2.0*data[k]+data[k+2])+
	                            (data[k+2]-data[k-2]));
	    dg[i]=datb[k]+0.5*(t-j)*((t-j)*(datb[k-2]-2.0*datb[k]+datb[k+2])+
	                            (datb[k+2]-datb[k-2]));
            k++;				
            dy[i]=data[k]+0.5*(t-j)*((t-j)*(data[k-2]-2.0*data[k]+data[k+2])+
	                            (data[k+2]-data[k-2])); }
        else if (j==1) {
	    y[i]=data[1]+(t-1.0)*(data[3]-data[1]);				
	    dg[i]=datb[1]+(t-1.0)*(datb[3]-datb[1]);				
	    dy[i]=data[2]+(t-1.0)*(data[4]-data[2]); }				
        else if (j==nz) {
	    k=2*nz-1; y[i]=data[k]+(t+1.0)*(data[k-2]-data[k]);
	    dg[i]=datb[k]+(t+1.0)*(datb[k-2]-datb[k]);
	    k++; dy[i]=data[k]+(t+1.0)*(data[k-2]-data[k]); }
	else {
	    printf ("SIMENV: Illegal value x[%d] = %.5f\n",i,x[i]); 
	    exit(1); }}
    return; }


/****************************************************************************/ 
/* Fast Fourier transform                                                   */
/* PG, 5-6-1990                                                             */
/****************************************************************************/ 

void fft(float *data, int nn, int isign)
{
    double wr,wi,wpr,wpi,wtemp,theta;
    float  tempr,tempi;
    int    i,istep,j,m,mmax,n;

    n=2*nn; j=1;
    for (i=1;i<=n;i=i+2) {
	if (j>i) {
	    tempr=data[j]; tempi=data[j+1]; data[j]=data[i]; 
	    data[j+1]=data[i+1]; data[i]=tempr; data[i+1]=tempi; }
	m=n/2; while (m>=2 && j>m) { j=j-m; m=m/2; }
        j=j+m; }
    mmax=2;
    while (n>mmax) {
        istep=2*mmax; theta=TWOPI/(-isign*mmax); wr=1.0; wi=0.0;
        wpr=sin(0.5*theta); wpr=-2.0*wpr*wpr; wpi=sin(theta);
        for (m=1;m<=mmax;m=m+2) {
            for (i=m;i<=n;i=i+istep) {
                j=i+mmax; 
		tempr=(float)wr*data[j]-(float)wi*data[j+1]; 
                tempi=(float)wr*data[j+1]+(float)wi*data[j]; 
                data[j]=data[i]-tempr; data[j+1]=data[i+1]-tempi;
                data[i]=data[i]+tempr; data[i+1]=data[i+1]+tempi; }
            wtemp=wr; wr=wr*wpr-wi*wpi+wr; wi=wi*wpr+wtemp*wpi+wi; }
        mmax=istep; }
    return; }
