/*
************************************************************************
*
*   Simplex.c - Simplex method for linear programming
*
*   Copyright (c) 1994-95
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 95/12/30
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.Simplex.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <simplex.h>

#include <bool.h>

#include <stdlib.h>
#include <math.h>

#define EPS 1.0e-6

/* Simplex method for linear programming.
   Algorithm adapted from Numerical Recipes. */

/* adapt EPS to scale of input data */
static double Eps;

static void
simp1(double **a, int mm, int ll[], int nll, BOOL iabf,
    int *kp, double *bmax)
{
  int k;
  double test;

  *kp = ll[0];
  *bmax = a[mm + 1][ll[0] + 1];
  for (k = 1; k < nll; k++) {
    if (iabf)
      test = fabs(a[mm + 1][ll[k] + 1]) - fabs(*bmax);
    else
      test = a[mm + 1][ll[k] + 1] - *bmax;
    if (test > 0.0) {
      *bmax = a[mm + 1][ll[k] + 1];
      *kp = ll[k];
    }
  }
}

static void
simp2(double **a, int n, int l2[], int nl2, int *ip, int kp, double *q1)
{
  int k, ii, i;
  double qp, q0, q;

  *ip = -1;
  for (i = 0; i < nl2; i++)
    if (a[l2[i] + 1][kp + 1] < - Eps) {
      *q1 = - a[l2[i] + 1][0] / a[l2[i] + 1][kp + 1];
      *ip = l2[i];
      break;
    }

  i++;
  for (; i < nl2; i++) {
    ii = l2[i];
    if (a[ii + 1][kp + 1] >= - Eps)
      continue;
    
    q = - a[ii + 1][0] / a[ii + 1][kp + 1];
    if (q < *q1) {
      *ip = ii;
      *q1 = q;
    } else if (q == *q1) {
      for (k = 0; k < n; k++) {
	qp = - a[*ip + 1][k + 1] / a[*ip + 1][kp + 1];
	q0 = - a[ii + 1][k + 1] / a[ii + 1][kp + 1];
	if (q0 != qp)
	  break;
      }
      if (q0 < qp)
	*ip = ii;
    }
  }
}

static void
simp3(double **a, int i1, int k1, int ip, int kp)
{
  int kk, ii;
  double piv;

  piv = 1.0 / a[ip + 1][kp + 1];
  for (ii = 0; ii <= i1 + 1; ii++) {
    if (ii - 1 == ip)
      continue;
    
    a[ii][kp + 1] *= piv;
    for (kk = 0; kk <= k1 + 1; kk++) {
      if (kk - 1 == kp)
	continue;
      
      a[ii][kk] -= a[ip + 1][kk] * a[ii][kp + 1];
    }
  }

  for (kk = 0; kk <= k1 + 1; kk++) {
    if (kk - 1 == kp)
      continue;

    a[ip + 1][kk] *= - piv;
  }

  a[ip + 1][kp + 1] = piv;
}

SimplexRes
Simplex(double **a, int n, int m, int m1, int m2, int m3, double sol[])
{
  int i, ip, ir, is, k, kh, kp, m12, nl1, nl2;
  int *izrov, *iposv;
  int *l1, *l2, *l3;
  double maxa, q1, bmax;

  if (m != m1 + m2 + m3)
    return SR_ERROR;

  izrov = malloc(n * sizeof(*izrov));
  iposv = malloc(m * sizeof(*iposv));

  l1 = malloc((n + 1) * sizeof(*l1));
  l2 = malloc(m * sizeof(*l2));
  l3 = malloc(m * sizeof(*l3));

  nl1 = n;
  for (k = 0; k < n; k++) {
    l1[k] = k;
    izrov[k] = k;
  }

  nl2 = m;
  for (i = 0; i < m; i++) {
    if (a[i + 1][0] < 0.0)
      return SR_ERROR;
    
    l2[i] = i;
    iposv[i] = n + i;
  }

  maxa = 0.0;
  for (i = 0; i <= m; i++)
    for (k = 0; k <= n; k++)
      if (a[i][k] > maxa)
	maxa = a[i][k];
      else if (- a[i][k] > maxa)
	maxa = - a[i][k];
  Eps = EPS * maxa;

  for (i = 0; i < m2; i++)
    l3[i] = 0;
  
  ir = 0;
  if (m2 + m3 > 0) {
    ir = 1;
    for (k = 0; k < n + 1; k++) {
      q1 = 0.0;
      for (i = m1; i < m; i++)
	q1 += a[i + 1][k];
      a[m + 1][k] = - q1;
    }

    while (ir == 1) {
      simp1(a, m, l1, nl1, FALSE, &kp, &bmax);
      if (bmax <= Eps && a[m + 1][0] < - Eps) {
	free(izrov);
	free(iposv);
	free(l1);
	free(l2);
	free(l3);
	return SR_NO_SOL;
      }

      if (bmax <= Eps && a[m + 1][0] <= Eps) {
	m12 = m1 + m2 + 1;
	for (ip = m12 - 1; ip < m; ip++) {
	  if (iposv[ip] != ip + n)
	    continue;
	  
	  simp1(a, ip, l1, nl1, TRUE, &kp, &bmax);
	  if (bmax > 0.0)
	    goto one;
	}

	ir = 0;
	m12--;
	for (i = m1; i < m12; i++) {
	  if (l3[i - m1] != 1)
	    continue;
	  
	  for (k = 0; k < n + 1; k++)
	    a[i + 1][k] = - a[i + 1][k];
	}

	break;
      }

      simp2(a, n, l2, nl2, &ip, kp, &q1);
      if (ip == -1) {
	free(izrov);
	free(iposv);
	free(l1);
	free(l2);
	free(l3);
	return SR_NO_SOL;
      }

one:  simp3(a, m, n - 1, ip, kp);
      if (iposv[ip] >= n + m1 + m2) {
	for (k = 0; k < nl1; k++)
	  if (l1[k] == kp)
	    break;
	
	nl1--;
	for (is = k; is < nl1; is++)
	  l1[is] = l1[is + 1];
	
	a[m + 1][kp + 1] += 1.0;
	for (i = 0; i < m + 2; i++)
	  a[i][kp + 1] = - a[i][kp + 1];
      } else if (iposv[ip] >= n + m1) {
	kh = iposv[ip] - m1 - n;
	if (l3[kh] >= 0) {
	  l3[kh] = -1;
	  a[m + 1][kp + 1] += 1.0;
	  for (i = 0; i < m + 2; i++)
	    a[i][kp + 1] = - a[i][kp + 1];
	}
      }

      is = izrov[kp];
      izrov[kp] = iposv[ip];
      iposv[ip] = is;
    }
  }

  for (;;) {
    simp1(a, -1, l1, nl1, FALSE, &kp, &bmax);
    if (bmax <= 0.0) {
      for (k = 0; k < n; k++)
	sol[k] = 0.0;
      
      for (i = 0; i < m; i++)
	if (iposv[i] < n)
	  sol[iposv[i]] = a[i + 1][0];

      free(izrov);
      free(iposv);
      free(l1);
      free(l2);
      free(l3);
      return SR_SOLVED;
    }

    simp2(a, n, l2, nl2, &ip, kp, &q1);
    if (ip == -1) {
      free(izrov);
      free(iposv);
      free(l1);
      free(l2);
      free(l3);
      return SR_NO_BOUND;
    }

    simp3(a, m - 1, n - 1, ip, kp);
    is = izrov[kp];
    izrov[kp] = iposv[ip];
    iposv[ip] = is;
  }
}
