/*
************************************************************************
*
*   Powell.c - minimization by Powell's method
*
*   Copyright (c) 1994-95
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 95/12/30
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.Powell.c
*   SCCS identification       : 1.1
*
************************************************************************
*/

#include <powell.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <brent.h>

/* Powell's method for function minimization.
   Algorithm adapted from Numerical Recipes. */

#define TOL 2.0e-4
#define ITMAX 200

static PowellMinFunc MinFuncCom;
static float *PCom, *XitCom;
static int NCom;
static void *ClientDataCom;
static float *Xt;

static float
f1dim(float x, void *clientData)
{
  int j;

  for (j = 0; j < NCom; j++)
    Xt[j] = PCom[j] + x * XitCom[j];
  
  return MinFuncCom(Xt, NCom, ClientDataCom);
}

static float
linmin(void)
{
  int j;
  float xx, xmin, fx, fb, fa, bx, ax, fret;

  ax = 0.0;
  xx = 1.0;
  bx = 2.0;

  BrentBracketMin(f1dim, NULL, &ax, &xx, &bx, &fa, &fx, &fb);
  (void) BrentSolveMin(f1dim, NULL, ax, xx, bx, TOL, &xmin, &fret);

  for (j = 0; j < NCom; j++) {
    XitCom[j] *= xmin;
    PCom[j] += XitCom[j];
  }

  return fret;
}

BOOL
PowellMin(PowellMinFunc func, float p[], int n, void *clientData, float ftol)
{
  int iter, i, ibig, j;
  float fret, t, fptt, fp, del;
  float *pt, *ptt, *xi, *xit;

  pt = malloc(n * sizeof(*pt));
  ptt = malloc(n * sizeof(*ptt));
  xi = malloc(n * n * sizeof(*xi));
  xit = malloc(n * sizeof(*xit));

  MinFuncCom = func;
  PCom = p;
  NCom = n;
  XitCom = xit;
  ClientDataCom = clientData;
  Xt = malloc(n * sizeof(*Xt));

  for (i = 0; i < n; i++)
    for (j = 0; j < n; j++)
      if (i == j)
	xi[j * n + i] = 1.0;
      else
	xi[j * n + i] = 0.0;

  fret = func(p, n, clientData);
  for (j = 0; j < n; j++)
    pt[j] = p[j];

  for (iter = 0; ; iter++) {
    fp = fret;
    del = 0.0;
    for (i = 0; i < n; i++) {
      for (j = 0; j < n; j++)
	xit[j] = xi[j * n + i];
      fptt = fret;
      fret = linmin();
      if (fabs(fptt - fret) > del) {
	del = fabs(fptt - fret);
	ibig = i;
      }
    }

    if (2.0 * fabs(fp - fret) <= ftol * (fabs(fp) + fabs(fret))) {
      free(pt);
      free(ptt);
      free(xi);
      free(xit);
      free(Xt);
      return TRUE;
    }

    if (iter == ITMAX) {
      free(pt);
      free(ptt);
      free(xi);
      free(xit);
      free(Xt);
      return FALSE;
    }

    for (j = 0; j < n; j++) {
      ptt[j] = 2.0 * p[j] - pt[j];
      xit[j] = p[j] - pt[j];
      pt[j] = p[j];
    }

    fptt = func(ptt, n, clientData);
    if (fptt >= fp)
      continue;
    
    t = 2.0 * (fp - 2.0 * fret + fptt) *
	(fp - fret - del) * (fp - fret - del) -
	del * (fp - fptt) * (fp - fptt);
    if (t >= 0.0)
      continue;

    fret = linmin();
    for (j = 0; j < n; j++)
      xi[j * n + ibig] = xit[j];
  }
}
