/*
************************************************************************
*
*   Marquardt.c - non-linear least squares by Marquardt's method
*
*   Copyright (c) 1994-96
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/06/28
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.Marquardt.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <marquardt.h>

#include <stdlib.h>
#include <math.h>

#include <gauss_jordan.h>

#define MAX_STEP 50

/* Non-linear least squares by Marquardt's method.
   Algorithm adapted from Numerical Recipes. */

static void
mrqcof(FitFunc func, float *x, float *sig, int ndata,
    float *a, int ma,
    float **alpha, float *beta, float *chisq)
{
  int k, j, i;
  float ymod, wt, sig2i, dy, *dyda;

  dyda = malloc(ma * sizeof(*dyda));

  for (j = 0; j < ma; j++) {
    for (k = 0; k <= j; k++)
      alpha[j][k] = 0.0;
    
    beta[j] = 0.0;
  }
  *chisq = 0.0;

  for (i = 0; i < ndata; i++) {
    func(x, i, a, ma, &ymod, dyda);
    if (sig == NULL)
      sig2i = 1.0;
    else
      sig2i = 1.0 / (sig[i] * sig[i]);
    dy = - ymod;

    for (j = 0; j < ma; j++) {
      wt = dyda[j] * sig2i;
      for (k = 0; k <= j; k++)
	alpha[j][k] += wt * dyda[k];

      beta[j] += dy * wt;
    }

    *chisq += dy * dy * sig2i;
  }

  for (j = 1; j < ma; j++)
    for (k = 0; k < j; k++)
      alpha[k][j] = alpha[j][k];
  
  free(dyda);
}

BOOL
MarquardtStep(FitFunc func, float *x, float *sig, int ndata,
    float *a, int ma,
    float **covar, float **alpha, float *chisq, float *alamda)
{
  int k, j;
  static float *da, *atry, *beta, ochisq;

  if (*alamda < 0.0) {
    atry = malloc(ma * sizeof(*atry));
    da = malloc(ma * sizeof(*da));
    beta = malloc(ma * sizeof(*beta));

    *alamda = 0.001;
    mrqcof(func, x, sig, ndata, a, ma, alpha, beta, chisq);
    ochisq = *chisq;
  }

  for (j = 0; j < ma; j++) {
    for (k = 0; k < ma; k++)
      covar[j][k] = alpha[j][k];

    covar[j][j] *= 1.0 + *alamda;
    da[j] = beta[j];
  }

  if (! GaussJordan(covar, da, ma))
    return FALSE;

  if (*alamda == 0.0) {
    free(atry);
    free(da);
    free(beta);

    return TRUE;
  }

  for (j = 0; j < ma; j++)
    atry[j] = a[j] + da[j];

  mrqcof(func, x, sig, ndata, atry, ma, covar, da, chisq);

  if (*chisq < ochisq) {
    *alamda *= 0.1;
    ochisq = *chisq;

    for (j = 0; j < ma; j++) {
      for (k = 0; k < ma; k++)
	alpha[j][k] = covar[j][k];
      
      beta[j] = da[j];
      a[j] = atry[j];
    }
  } else {
    *alamda *= 10.0;
    *chisq = ochisq;
  }

  return TRUE;
}

BOOL
Marquardt(FitFunc func, float *x, float *sig, int ndata, float *a, int ma,
    float **covar, float *rmsd)
{
  BOOL localCovar;
  float *covarA, *alphaA, **alpha;
  float chisq, lastChisq, lamda;
  BOOL ok;
  int step, i;

  if (covar == NULL) {
    covarA = malloc(ma * ma * sizeof(*covarA));
    covar = malloc(ma * sizeof(*covar));
    for (i = 0; i < ma; i++)
      covar[i] = covarA + i * ma;
    localCovar = TRUE;
  } else {
    localCovar = FALSE;
  }

  alphaA = malloc(ma * ma * sizeof(*alphaA));
  alpha = malloc(ma * sizeof(*alpha));
  for (i = 0; i < ma; i++)
    alpha[i] = alphaA + i * ma;

  lamda = -1.0;

  ok = MarquardtStep(func, x, sig, ndata, a, ma,
      covar, alpha, &chisq, &lamda);

  for (step = 0; step < 100 && ok; step++) {
    lastChisq = chisq;

    ok = MarquardtStep(func, x, sig, ndata, a, ma,
	covar, alpha, &chisq, &lamda);

    if (chisq <= lastChisq && chisq > 0.999 * lastChisq)
      break;
  }

  lamda = 0.0;
  (void) MarquardtStep(func, x, sig, ndata, a, ma,
      covar, alpha, &chisq, &lamda);

  if (localCovar) {
    free(covarA);
    free(covar);
  }

  free(alphaA);
  free(alpha);

  *rmsd = sqrt(chisq / ndata);

  return ok && step < 100;
}
