/*
************************************************************************
*
*   LUDecomp.c - LU decomposition for solving set of linear equations
*
*   Copyright (c) 1994-95
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/01/05
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.LUDecomp.c
*   SCCS identification       : 1.1
*
************************************************************************
*/

#include <lu_decomp.h>

#include <stdlib.h>
#include <math.h>

/* LU decomposition for solving set of linear equations.
   Algorithm adapted from Numerical Recipes. */

#define TINY 1.0e-20

BOOL
LUDecomp(float **a, int n, int indx[])
{
  int i, imax, j, k;
  float big, dum, sum, temp;
  float *vv;

  vv = malloc(n * sizeof(*vv));

  for (i = 0; i < n; i++) {
    big = 0.0;
    for (j = 0; j < n; j++) {
      temp = fabs(a[i][j]);
      if (temp > big)
	big = temp;
    }

    if (big == 0.0) {
      free(vv);
      return FALSE;
    }

    vv[i] = 1.0 / big;
  }

  for (j = 0; j < n; j++) {
    for (i = 0; i < j; i++) {
      sum = a[i][j];
      for (k = 0; k < i; k++)
	sum -= a[i][k] * a[k][j];
      a[i][j] = sum;
    }

    big = 0.0;
    for (i = j; i < n; i++) {
      sum = a[i][j];
      for (k = 0; k < j; k++)
	sum -= a[i][k] * a[k][j];
      a[i][j] = sum;

      dum = vv[i] * fabs(sum);
      if (dum >= big) {
	big = dum;
	imax = i;
      }
    }

    if (j != imax) {
      for (k = 0; k < n; k++) {
	dum = a[imax][k];
	a[imax][k] = a[j][k];
	a[j][k] = dum;
      }

      vv[imax] = vv[j];
    }

    indx[j] = imax;

    if (a[j][j] == 0.0)
      a[j][j] = TINY;
    
    if (j != n - 1) {
      dum = 1.0 / a[j][j];
      for (i = j + 1; i < n; i++)
	a[i][j] *= dum;
    }
  }

  free(vv);

  return TRUE;
}

void
LUSolve(float **a, int n, int indx[], float b[])
{
  int i, ii = -1, ip, j;
  float sum;

  for (i = 0; i < n; i++) {
    ip = indx[i];
    sum = b[ip];
    b[ip] = b[i];

    if (ii >= 0)
      for (j = ii; j < i; j++)
	sum -= a[i][j] * b[j];
    else if (sum != 0.0)
      ii = i;
    
    b[i] = sum;
  }

  for (i = n - 1; i >= 0; i--) {
    sum = b[i];
    for (j = i + 1; j < n; j++)
      sum -= a[i][j] * b[j];
      b[i] = sum / a[i][i];
  }
}
