/*
************************************************************************
*
*   Interpol.c - function interpolation
*
*   Copyright (c) 1994
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 95/07/20
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.Interpol.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <interpol.h>

#include <stdlib.h>
#include <math.h>

#define MAX_DEGREE 20

/* Function interpolation algorithms, adapted from Numerical Recipes. */

#define TINY 1.0e-25

BOOL
InterpolPolynom(float *xa, float *ya, int n,
    float x, float *yP, float *dyP)
{
  int i, m, ns = 0;
  float den, dif, dift, ho, hp, w;
  float c[MAX_DEGREE], d[MAX_DEGREE];

  if (n > MAX_DEGREE)
    return FALSE;

  dif = fabs(x - xa[0]);
  for (i = 0; i < n; i++) {
    dift = fabs(x - xa[i]);
    if (dift < dif) {
      ns = i;
      dif = dift;
    }

    c[i] = ya[i];
    d[i] = ya[i];
  }

  *yP = ya[ns--];
  for (m = 0; m < n - 1; m++) {
    for (i = 0; i < n - m - 1; i++) {
      ho = xa[i] - x;
      hp = xa[i + m + 1] - x;
      w = c[i + 1] - d[i];
      den = ho - hp;

      if (den == 0.0)
	return FALSE;

      den = w / den;
      d[i] = hp * den;
      c[i] = ho * den;
    }

    *dyP = (2 * ns < n - m - 3 ? c[ns + 1] : d[ns--]);
    *yP += *dyP;
  }

  return TRUE;
}

BOOL
InterpolRational(float *xa, float *ya, int n,
    float x, float *yP, float *dyP)
{
  int i, m, ns = 0;
  float w, t, hh, h, dd;
  float c[MAX_DEGREE], d[MAX_DEGREE];

  if (n > MAX_DEGREE)
    return FALSE;

  hh = fabs(x - xa[0]);
  for (i = 0; i < n; i++) {
    h = fabs(x - xa[i]);
    if (h == 0.0) {
      *yP = ya[i];
      *dyP = 0.0;
      return TRUE;
    } else if (h < hh) {
      ns = i;
      hh = h;
    }

    c[i] = ya[i];
    d[i] = ya[i] + TINY;
  }

  *yP = ya[ns--];
  for (m = 0; m < n - 1; m++) {
    for (i = 0; i < n - m - 1; i++) {
      w = c[i + 1] - d[i];
      h = xa[i + m + 1] - x;
      t = (xa[i] - x) * d[i] / h;
      dd = t - c[i + 1];

      if (dd == 0.0)
	return FALSE;

      dd = w / dd;
      d[i] = c[i + 1] * dd;
      c[i] = t * dd;
    }

    *dyP = (2 * ns < n - m - 3 ? c[ns + 1] : d[ns--]);
    *yP += *dyP;
  }

  return TRUE;
}

BOOL
InterpolSplineCalc(float *xa, float *ya, int n, float *y2a)
{
  int i, k;
  float p, qn, sig, un, *u;

  u = malloc((n - 1) * sizeof(*u));

  y2a[0] = 0.0;
  u[0] = 0.0;

  for (i = 1; i < n - 1; i++) {
    sig = (xa[i] - xa[i - 1]) / (xa[i + 1] - xa[i - 1]);
    p = sig * y2a[i - 1] + 2.0;
    y2a[i] = (sig - 1.0) / p;
    u[i] = (ya[i+1] - ya[i]) / (xa[i + 1] - xa[i]) -
	(ya[i] - ya[i - 1]) / (xa[i] - xa[i - 1]);
    u[i] = (6.0 * u[i] / (xa[i + 1] - xa[i - 1]) - sig * u[i - 1]) / p;
  }

  qn = 0.0;
  un = 0.0;

  y2a[n - 1] = (un - qn * u[n - 2]) / (qn * y2a[n - 2] + 1.0);
  for (k = n - 2; k >= 0; k--)
    y2a[k] = y2a[k] * y2a[k + 1] + u[k];

  free(u);

  return TRUE;
}

BOOL
InterpolSplineVal(float *xa, float *ya, float *y2a, int n,
    float x, float *yP)
{
  int klo, khi, k;
  float h, b, a;

  klo = 0;
  khi = n - 1;
  while (khi - klo > 1) {
    k = (khi + klo) / 2;
    if (xa[k] > x)
      khi = k;
    else
      klo = k;
  }

  h = xa[khi] - xa[klo];
  if (h == 0.0)
    return FALSE;
  
  a = (xa[khi] - x) / h;
  b = (x - xa[klo]) / h;

  *yP = a * ya[klo] + b * ya[khi] +
      ((a * a * a - a) * y2a[klo] +
       (b * b * b - b) * y2a[khi]) * h * h / 6.0;

  return TRUE;
}
