/*
************************************************************************
*
*   ChebFit.c - discrete Chebyshev approximation
*
*   Copyright (c) 1994-96
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/11/25
*   Pathname of SCCS file     : /local/home/kor/molmol/tools/src/SCCS/s.ChebFit.c
*   SCCS identification       : 1.5
*
************************************************************************
*/

#include <cheb_fit.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* Discrete Chebyshev approximation.
   Converted to C from the FORTRAN function CHEB from netlib
   by Reto Koradi (kor@spectrospin.ch). */

/* BIG must be set equal to any very large double constant. */
#define BIG 1.0e70

/* TOL must be set to a small positive tolerance. Empirical evidence
   suggests tol=10**(-d+1) where d represents the number of decimal
   digits of accuracy available. */
#define TOL 1.0e-8

static void
doPivot(double *a[], int m, int n, int prow, int pcol, int nmr)
/* Pivot on a[prow][pcol]. */
{
  int np2;
  double pivot, tpivot, d;
  int i, j;

  np2 = n + 2;

  pivot = a[prow][pcol];
  for (j = 0; j < m; j++)
    a[prow][j] /= pivot;

  for (j = 0; j < m; j++) {
    if (j == pcol)
      continue;
    
    d = a[prow][j];
    for (i = nmr; i < np2; i++) {
      if (i == prow)
	continue;
      
      a[i][j] -= d * a[i][pcol];
    }
  }

  tpivot = -pivot;
  for (i = nmr; i < np2; i++)
    a[i][pcol] /= tpivot;

  a[prow][pcol] = 1.0 / pivot;

  d = a[prow][m];
  a[prow][m] = a[np2][pcol];
  a[np2][pcol] = d;
}

BOOL
ChebyshevFit(double *a[], double b[], int m, int n,
    double x[], double *resmaxP)
/*
* This subroutine uses a modification of the simplex method
* of linear programming to calculate a Chebyshev solution to
* an over-determined system of linear equations.
* Description of parameters.
* a      Two dimensional real array of size (n + 3, m + 1).
*        On entry, the transpose of the matrix of
*        coefficients of the over-determined system must
*        be stored in the first m columns and n rows of a.
*        These values are destroyed by the subroutine.
* b      One dimensional real array of size mdim. On entry,
*        b must contain the right-hand sides of the
*        equations in its first m locations. On exit, b
*        contains the residuals for the equations in its
*        first m locations (see description).
* m      Number of equations.
* n      Number of unknowns (n must not exceed m).
* x      One dimensional real array of size n. On exit,
*        this array contains a solution to the problem.
* resmax The largest residual in magnitude.
* Returns TRUE on success, FALSE on failure.
*/
{
  int mp1, mm1, np1, np2, np3;
  int rank, nmr, nmk, mode, prow, pcol, ocode;
  double d, dd, val;
  int step;
  int i, j, k;

  /* Initialization. */
  mp1 = m + 1;
  mm1 = m - 1;
  np1 = n + 1;
  np2 = n + 2;
  np3 = n + 3;

  rank = n;
  nmr = 0;

  for (j = 0; j < m; j++) {
    a[n][j] = 1.0;
    a[np1][j] = - b[j];
    a[np2][j] = n + j;
  }
  a[n][m] = -1.0;

  ocode = 1;

  for (i = 0; i < n; i++) {
    x[i] = 0.0;
    a[i][m] = i;
  }

  /* Level 1. */

  mode = 0;
  k = -1;
  for (;;) {
    if (mode == 0) {
      k++;
      nmk = n - k;

      for (j = k; j < m; j++)
	b[j] = 1.0;
    }

    /* Determine the vector to enter the basis. */
    d = - BIG;
    for (j = k; j < m; j++) {
      if (b[j] == 0.0)
	continue;
      dd = fabs(a[np1][j]);
      if (dd <= d)
	continue;
      pcol = j;
      d = dd;
    }

    /* Test for zero right-hand side. */
    if (k == 0 && d <= TOL) {
      *resmaxP = 0.0;
      mode = 2;
      goto output;
    }

    /* Determine the vector to leave the basis. */
    d = TOL;
    for (i = 0; i < nmk; i++) {
      dd = fabs(a[i][pcol]);
      if (dd <= d)
	continue;
      prow = i;
      d = dd;
    }

    if (d <= TOL) {
      /* Check for linear dependence in level 1. */
      b[pcol] = 0.0;
      if (mode == 1)
	continue;

      for (j = k; j < m; j++) {
	if (b[j] == 0.0)
	  continue;
	
	for (i = 0; i < nmk; i++)
	  if (fabs(a[i][j]) > TOL) {
	    mode = 1;
	    break;
	  }
	
	if (mode == 1)
	  break;
      }
      if (mode == 1)
	continue;

      rank = k;
      nmr = n - rank;
      ocode = 0;
      break;
    }

    doPivot(a, m, n, prow, pcol, nmr);

    if (pcol != k)
      /* Interchange columns in level 1. */
      for (i = 0; i < np3; i++) {
        d = a[i][pcol];
        a[i][pcol] = a[i][k];
        a[i][k] = d;
      }
    
    if (prow != nmk - 1)
      /* Interchange rows in level 1. */
      for (j = 0; j < mp1; j++) {
	d = a[prow][j];
	a[prow][j] = a[nmk - 1][j];
	a[nmk - 1][j] = d;
      }

    mode = 0;
    if (k >= n - 1)
      break;
  };

  if (rank == m)
    goto output;

  /* Level 2. */

  /* Determine the vector to enter the basis. */
  d = TOL;
  for (j = rank; j < m; j++) {
    dd = fabs(a[np1][j]);
    if (dd <= d)
      continue;
    pcol = j;
    d = dd;
  }

  /* Compare chebyshev error with TOL. */
  if (d <= TOL) {
    *resmaxP = 0.0;
    mode = 3;
    goto output;
  }

  if (a[np1][pcol] >= - TOL) {
    a[n][pcol] = 2.0 - a[n][pcol];
    for (i = nmr; i < np3; i++) {
      if (i == n)
	continue;
      a[i][pcol] = - a[i][pcol];
    }
  }

  /* Arrange for all entries in pivot column (except pivot) to be negative. */
  for (i = nmr; i < n; i++) {
    if (a[i][pcol] < TOL)
      continue;

    for (j = 0; j < m; j++) {
      a[n][j] = a[n][j] + 2.0 * a[i][j];
      a[i][j] = - a[i][j];
    }

    a[i][m] = - a[i][m];
  }

  prow = n;
  doPivot(a, m, n, prow, pcol, nmr);

  if (rank + 1 == m)
    goto output;

  mm1 = m - 1;

  if (pcol != m)
    /* interchange columns in level 2. */
    for (i = nmr; i < np3; i++) {
      d = a[i][pcol];
      a[i][pcol] = a[i][mm1];
      a[i][mm1] = d;
    }

  /* Level 3. */

  /* This loop normally terminates by a break, but there are rare cases
     where this doesn't work, probably due to numerical problems. */
  for (step = 0; step < n; step++) {
    /* Determine the vector to enter the basis. */
    d = - TOL;
    val = 2.0 * a[np1][mm1];
    for (j = rank; j < mm1; j++) {
      if (a[np1][j] >= d) {
	dd = val - a[np1][j];
	if (dd >= d)
	  continue;
	mode = 1;
	pcol = j;
	d = dd;
      } else {
	pcol = j;
	d = a[np1][j];
	mode = 0;
      }
    }

    if (d >= - TOL)
      goto output;

    dd = -d / a[np1][mm1];
    if (dd < 0.0) {
      mode = 4;
      goto output;
    }

    if (mode > 0) {
      for (i = nmr; i < np1; i++)
	a[i][pcol] = 2.0 * a[i][mm1] - a[i][pcol];

      a[np1][pcol] = d;
      a[np2][pcol] = - a[np2][pcol];
    }

    /* Determine the vector to leave the basis. */
    d = BIG;
    for (i = nmr; i < np1; i++) {
      if (a[i][pcol] <= TOL)
	continue;
      
      dd = a[i][mm1] / a[i][pcol];
      if (dd >= d)
	continue;

      prow = i;
      d = dd;
    }

    if (d >= BIG) {
      ocode = 2;
      break;
    }

    doPivot(a, m, n, prow, pcol, nmr);
  }

  /* Prepare output. */
output:
  for (j = 0; j < m; j++)
    b[j] = 0.0;

  if (mode == 2)
    goto end;

  for (j = 0; j < rank; j++) {
    k = a[np2][j];
    x[k] = a[np1][j];
  }

  if (mode == 3 || rank == m)
    goto end;

  for (i = nmr; i < np1; i++) {
    k = (int) (fabs(a[i][m]) - n);
    if (a[i][m] < 0.0)
      b[k] = - a[np1][mm1];
    else
      b[k] = a[np1][mm1];
  }

  for (j = rank; j < mm1; j++) {
    k = (int) (fabs(a[np2][j]) - n);
    if (a[np2][j] < 0.0)
      b[k] = - (a[np1][mm1] - a[np1][j]);
    else
      b[k] = a[np1][mm1] - a[np1][j];
  }

  /* Test for non-unique solution. */
  for (i = nmr; i < np1; i++)
    if (fabs(a[i][mm1]) <= TOL) {
      ocode = 0;
      break;
    }

end:
  if (mode != 2 && mode != 3)
    *resmaxP = a[np1][mm1];

  if (rank == m)
    *resmaxP = 0.0;

  if (mode == 4)
    *resmaxP -= d;

  return ocode != 2;
}
