/*
************************************************************************
*
*   Brent.c - minimum of function by Brent's method
*
*   Copyright (c) 1994-96
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/04/23
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.Brent.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <brent.h>

#include <stdio.h>
#include <math.h>

/* One-dimensional root finding and function minimation by
   Brent's method. Algorithms adapted from Numerical Recipes.
*/

#define ITMAX 100

#define NTRY 50
#define FACTOR 1.6
#define EPS 3.0e-8

#define GOLD 1.618034
#define GLIMIT 100.0
#define TINY 1.0e-20
#define CGOLD 0.3819660
#define ZEPS 1.0e-10

#define SIGN(a, b) ((b) > 0.0 ? fabs(a) : -fabs(a))
#define SHFT(a, b, c, d) (a) = (b); (b) = (c); (c) = (d)
#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MOV3(a,b,c, d,e,f) (a) = (d); (b) = (e); (c) = (f);

BOOL
BrentBracketRoot(RootFunc func, void *clientData,
    float *x1P, float *x2P)
/* Given a function <func> and an initial guessed range x1 to x2,
   the routine expands the range gemetrically until a root is
   bracketed by the returned values x1 and x2. Returns TRUE on success,
   FALSE on failure.
*/
{
  int j;
  float x1, x2;
  float f1, f2;

  x1 = *x1P;
  x2 = *x2P;

  if (x1 == x2)
    return FALSE;
  
  f1 = func(x1, clientData);
  f2 = func(x2, clientData);

  for (j = 0; j < NTRY; j++) {
    if (f1 * f2 < 0.0) {
      *x1P = x1;
      *x2P = x2;
      return TRUE;
    }

    if (fabs(f1) < fabs(f2)) {
      x1 += FACTOR * (x1 - x2);
      f1 = func(x1, clientData);
    } else {
      x2 += FACTOR * (x2 - x1);
      f2 = func(x2, clientData);
    }
  }

  return FALSE;
}

BOOL
BrentSolveRoot(RootFunc func, void *clientData,
    float x1, float x2, float tol,
    float *xrootP)
/* Using Brent's method, find the root of a function <func>
   known to lie between x1 and x2. The root will be refined
   until its accuracy is tol. Returns TRUE on success, FALSE
   on failure.
*/
{
  int iter;
  float a, b, c, d, e, min1, min2;
  float fa, fb, fc, p, q, r, s, tol1, xm;

  a = x1;
  b = x2;
  fa = func(a, clientData);
  fb = func(b, clientData);

  if (fa * fb > 0.0)
    return FALSE;
  
  fc = fb;
  for (iter = 0; iter < ITMAX; iter++) {
    if (fb * fc > 0.0) {
      c = a;
      fc = fa;
      d = b - a;
      e = d;
    }

    if (fabs(fc) < fabs(fb)) {
      a = b;
      b = c;
      c = a;
      fa = fb;
      fb = fc;
      fc = fa;
    }

    tol1 = 2.0 * EPS * fabs(b) + 0.5 * tol;
    xm = 0.5 * (c - b);
    if (fabs(xm) <= tol || fb == 0.0) {
      *xrootP = b;
      return TRUE;
    }

    if (fabs(e) >= tol1 && fabs(fa) > fabs(fb)) {
      s = fb / fa;
      if (a == c) {
	p = 2.0 * xm * s;
	q = 1.0 - s;
      } else {
	q = fa / fc;
	r = fb / fc;
	p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0));
	q = (q - 1.0) * (r - 1.0) * (s - 1.0);
      }

      if (p > 0.0)
	q = - q;
      p = fabs(p);
      min1 = 3.0 * xm * q - fabs(tol1 * q);
      min2 = fabs(e * q);

      if (2.0 * p < (min1 < min2 ? min1 : min2)) {
	e = d;
	d = p / q;
      } else {
	d = xm;
	e = d;
      }
    } else {
      d = xm;
      e = d;
    }

    a = b;
    fa = fb;
    if (fabs(d) > tol1)
      b += d;
    else
      b += (xm > 0.0 ? fabs(tol1) : - fabs(tol1));

    fb = func(b, clientData);
  }

  return FALSE;
}

void
BrentBracketMin(MinFunc func, void *clientData,
    float *axP, float *bxP, float *cxP,
    float *faP, float *fbP, float *fcP)
/* Given a function <func>, and given distinct initial points
   *axP and *bxP, this routine searches in the downhill direction
   (defined by the function as evaluated at the initial points)
   and returns new points ax, bx, cx which bracket a minimum of
   the function. Also returned are the function values at the
   three points, <fa>, <fb> and <fc> if faP, fbP and fcP are != NULL.
*/
{
  float ax, bx, cx, fa, fb, fc;
  float ulim, u, r, q, fu, dum;

  ax = *axP;
  bx = *bxP;
  fa = func(ax, clientData);
  fb = func(bx, clientData);

  if (fb > fa) {
    SHFT(dum, ax, bx, dum);
    SHFT(dum, fb, fa, dum);
  }

  cx = bx + GOLD * (bx - ax);
  fc = func(cx, clientData);

  while (fb > fc) {
    r = (bx - ax) * (fb - fc);
    q = (bx - cx) * (fb - fa);
    u = bx - ((bx - cx) * q - (bx - ax) * r) /
        (2.0 * SIGN(MAX(fabs(q - r), TINY), q - r));
    ulim = bx + GLIMIT * (cx - bx);

    if ((bx - u) * (u - cx) > 0.0) {
      fu = func(u, clientData);
      if (fu < fc) {
        *axP = bx;
        *bxP = u;
        *cxP = cx;
        if (faP != NULL)
          *faP = fb;
        if (fbP != NULL)
          *fbP = fu;
        if (fcP != NULL)
          *fcP = fc;
        return;
      } else if (fu > fb) {
        *axP = ax;
        *bxP = bx;
        *cxP = u;
        if (faP != NULL)
          *faP = fa;
        if (fbP != NULL)
          *fbP = fb;
        if (fcP != NULL)
          *fcP = fu;
        return;
      }
      u = cx + GOLD * (cx - bx);
      fu = func(u, clientData);
    } else if ((cx - u) * (u - ulim) > 0.0) {
      fu = func(u, clientData);
      if (fu < fc) {
        SHFT(bx, cx, u, cx + GOLD * (cx - bx));
        SHFT(fb, fc, fu, func(u, clientData));
      }
    } else if ((u - ulim) * (ulim - cx) >= 0.0) {
      u = ulim;
      fu = func(u, clientData);
    } else {
      u = cx + GOLD * (cx - bx);
      fu = func(u, clientData);
    }

    SHFT(ax, bx, cx, u);
    SHFT(fa, fb, fc, fu);
  }

  *axP = ax;
  *bxP = bx;
  *cxP = cx;
}

BOOL
BrentSolveMin(MinFunc func, void *clientData,
    float ax, float bx, float cx, float tol,
    float *xminP, float *yminP)
/* Given a function <func>, and given a bracketing triplet of
   abscissas ax, bx, cx (such that bx is between ax and cx,
   and f(bx) is less than both f(ax) and f(cx)), this routine
   isolates the minimum to a fractional precision of about <tol>
   using Brent's method. The abscissa of the minimum is returned
   as xmin, and the minimum function value as ymin.
   Returns TRUE on success, FALSE on failure.
*/
{
  int iter;
  float a, b, d, etemp, fu, fv, fw, fx, p, q, r;
  float tol1, tol2, u, v, w, x, xm;
  float e = 0.0;

  a = (ax < cx ? ax : cx);
  b = (ax > cx ? ax : cx);

  x = bx;
  w = x;
  v = x;
  fx = func(x, clientData);
  fw = fx;
  fv = fx;

  for (iter = 0; iter < ITMAX; iter++) {
    xm = 0.5 * (a + b);
    tol1 = tol * fabs(x) + ZEPS;
    tol2 = 2.0 * tol1;

    if (fabs(x - xm) <= tol2 - 0.5 * (b - a)) {
      *xminP = x;
      *yminP = fx;

      return TRUE;
    }

    if (fabs(e) > tol1) {
      r = (x - w) * (fx - fv);
      q = (x - v) * (fx - fw);
      p = (x - v) * q - (x - w) * r;
      q = 2.0 * (q - r);

      if (q > 0.0) 
        p = -p;
      else
        q = -q;

      etemp = e;
      e = d;

      if (fabs(p) >= fabs(0.5 * q * etemp) ||
          p <= q * (a - x) || p >= q * (b - x)) {
        e = (x >= xm ? a - x : b - x);
        d = CGOLD * e;
      } else {
        d = p / q;
        u = x + d;
        if (u - a < tol2 || b - u < tol2)
          d = SIGN(tol1, xm - x);
      }
    } else {
      e = (x >= xm ? a - x : b - x);
      d = CGOLD * e;
    }

    u = (fabs(d) >= tol1 ? x + d : x + SIGN(tol1, d));
    fu = func(u, clientData);

    if (fu <= fx) {
      if (u >= x)
        a = x;
      else
        b = x;

      SHFT(v, w, x, u);
      SHFT(fv, fw, fx, fu);
    } else {
      if (u < x)
        a = u;
      else
        b = u;

      if (fu <= fw || w == x) {
        v = w;
        w = u;
        fv = fw;
        fw = fu;
      } else if (fu <= fv || v == x || v == w) {
        v = u;
        fv = fu;
      }
    }
  }

  *xminP = x;
  *yminP = fx;

  return FALSE;
}

BOOL
BrentSolveDerMin(DerMinFunc func, void *clientData,
    float ax, float bx, float cx, float tol,
    float *xminP, float *yminP)
/* Given a function <func> that calculates function value and
   derivative, and given a bracketing triplet of abscissas
   ax, bx, cx (such that bx is between ax and cx, and f(bx) is less
   than both f(ax) and f(cx)), this routine isolates the minimum to
   a fractional precision of about <tol> using a modification of
   Brent's method that uses derivatives. The abscissa of the minimum
   is returned as xmin, and the minimum function value as ymin.
   Returns TRUE on success, FALSE on failure.
*/
{
  int iter, ok1, ok2;
  float a, b, d, d1, d2, du, dv, dw, dx;
  float fu, fv, fw, fx, olde, tol1, tol2, u, u1, u2, v, w, x, xm;
  float e = 0.0;

  a = (ax < cx ? ax : cx);
  b = (ax > cx ? ax : cx);

  x = bx;
  w = x;
  v = x;
  func(x, clientData, &fx, &dx);
  fw = fx;
  fv = fx;
  dw = dx;
  dv = dx;

  for (iter = 0; iter < ITMAX; iter++) {
    xm = 0.5 * (a + b);
    tol1 = tol * fabs(x) + ZEPS;
    tol2 = 2.0 * tol1;

    if (fabs(x - xm) <= (tol2 - 0.5 * (b - a))) {
      *xminP = x;
      *yminP = fx;

      return TRUE;
    }

    if (fabs(e) > tol1) {
      d1 = 2.0 * (b - a);
      d2 = d1;
      if (dw != dx)
	d1 = (w - x) * dx / (dx - dw);
      if (dv != dx)
	d2 = (v - x) * dx / (dx - dv);

      u1 = x + d1;
      u2 = x + d2;
      ok1 = (a - u1) * (u1 - b) > 0.0 && dx * d1 <= 0.0;
      ok2 = (a - u2) * (u2 - b) > 0.0 && dx * d2 <= 0.0;
      olde = e;
      e = d;

      if (ok1 || ok2) {
	if (ok1 && ok2)
	  d = (fabs(d1) < fabs(d2) ? d1 : d2);
	else if (ok1)
	  d = d1;
	else
	  d = d2;

	if (fabs(d) <= fabs(0.5 * olde)) {
	  u = x + d;
	  if (u - a < tol2 || b - u < tol2)
	    d = SIGN(tol1, xm - x);
	} else {
	  e = (dx >= 0.0 ? a - x : b - x);
	  d = 0.5 * e;
	}
      } else {
	e = (dx >= 0.0 ? a - x : b - x);
	d = 0.5 * e;
      }
    } else {
      e = (dx >= 0.0 ? a - x : b - x);
      d = 0.5 * e;
    }

    if (fabs(d) >= tol1) {
      u = x + d;
      func(u, clientData, &fu, &du);
    } else {
      u = x + SIGN(tol1, d);
      func(u, clientData, &fu, &du);
      if (fu > fx) {
	*xminP = x;
	*yminP = fx;

	return TRUE;
      }
    }

    if (fu <= fx) {
      if (u >= x)
	a = x;
      else
	b = x;

      MOV3(v, fv, dv, w, fw, dw)
      MOV3(w, fw, dw, x, fx, dx)
      MOV3(x, fx, dx, u, fu, du)
    } else {
      if (u < x)
	a = u;
      else
	b = u;

      if (fu <= fw || w == x) {
	MOV3(v, fv, dv, w, fw, dw)
        MOV3(w, fw, dw, u, fu, du)
      } else if (fu < fv || v == x || v == w) {
	MOV3(v, fv, dv, u, fu, du)
      }
    }
  }

  *xminP = x;
  *yminP = fx;

  return FALSE;
}
