/*
************************************************************************
*
*   BigInt.c - implementation of BIGINT type
*
*   Copyright (c) 1994
*
*   ETH Zuerich
*   Institut fuer Molekularbiologie und Biophysik
*   ETH-Hoenggerberg
*   CH-8093 Zuerich
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 95/07/20
*   Pathname of SCCS file     : /sgiext/molmol/tools/src/SCCS/s.BigInt.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <big_int.h>

#include <stdlib.h>

#define REALLOC(ptr, size) ptr = realloc(ptr, size);

#define CHARBITS 8                      /* bits in a char */
#define CHARMASK ((1 << CHARBITS) - 1)
#define CHARMAX  CHARMASK
#define CHARHIGH (1 << (CHARBITS - 1))

#define HEXBITS  4
#define HEXMASK  ((1 << HEXBITS) - 1)

struct big_int {
  unsigned memL;                        /* allocated length */
  unsigned curL;                        /* current length */
  unsigned char *d;                     /* pointer to contents */
};

static int
HexDigitToInt(char ch)
{
  if (ch >= '0' && ch <= '9')
    return ch - '0';
  if (ch >= 'a' && ch <= 'f')
    return ch - 'a' + 10;
  return ch - 'A' + 10;
}

static char
intToHexDigit(int i)
{
  if (i < 10)
    return '0' + i;
  return 'a' + (i - 10);
}

BIGINT
BiNew(void)
{                                       /* returns new BIGINT */
  BIGINT  b;

  b = (BIGINT) malloc(sizeof(struct big_int));

  b->d = (unsigned char *) malloc(1);
  b->memL = 1;
  b->curL = 1;
  b->d[0] = 0;

  return b;
}

void
BiFree(                                 /* free b */
    BIGINT b
)
{
  free(b->d);
  free(b);
}

void
BiIntAss(                               /* b = i */
    BIGINT b,
    int i
)
{
  if (b->memL < sizeof(int)) {
    b->memL = sizeof(int);

    REALLOC(b->d, b->memL);
  }
  b->curL = 0;
  while (i != 0) {
    b->d[b->curL++] = i & CHARMASK;
    i >>= CHARBITS;
  }

  if (b->curL == 0)
    b->curL = 1;
}

void
BiStrAss(                               /* b = s */
    BIGINT b,
    char *s
)
{
  unsigned len, len2;
  int     sInd, bInd;

  for (len = 0; s[len] != '\0'; len++);
  len2 = (len + 1) / 2;

  if (b->memL < len2) {
    b->memL = len2;
    REALLOC(b->d, b->memL);
  }
  sInd = len - 1;
  bInd = -1;
  while (sInd > 0) {
    bInd++;
    b->d[bInd] = (HexDigitToInt(s[sInd - 1]) << HEXBITS)
        + HexDigitToInt(s[sInd]);
    sInd -= 2;
  }

  if (sInd == 0) {
    bInd++;
    b->d[bInd] = HexDigitToInt(s[0]);
  }
  while (bInd >= 0 && b->d[bInd--] == 0);
  b->curL = bInd + 2;
}

void
BiAss(                                  /* b1 = b2 */
    BIGINT b1,
    BIGINT b2
)
{
  if (b1->memL < b2->curL) {
    b1->memL = b2->curL;
    REALLOC(b1->d, b1->memL);
  }
  for (b1->curL = 0; b1->curL < b2->curL; b1->curL++)
    b1->d[b1->curL] = b2->d[b1->curL];
}

void
BiAdd(                                  /* b1 = b1 + b2 */
    BIGINT b1,
    BIGINT b2
)
{
  unsigned maxL, minL, i, carry, digit;

  if (b1->curL > b2->curL) {
    maxL = b1->curL;
    minL = b2->curL;
  } else {
    maxL = b2->curL;
    minL = b1->curL;
  }

  if (b1->memL < maxL) {
    b1->memL = maxL + 1;
    REALLOC(b1->d, b1->memL);
  }
  carry = 0;
  for (i = 0; i < minL; i++) {
    digit = b1->d[i] + b2->d[i] + carry;
    if (carry = digit > CHARMAX)
      digit -= CHARMAX + 1;
    b1->d[i] = digit;
  }

  if (b2->curL > minL) {
    for (i = minL; i < maxL; i++) {
      digit = b2->d[i] + carry;
      if (carry = digit > CHARMAX)
        digit -= CHARMAX + 1;
      b1->d[i] = digit;
    }
    b1->curL = maxL;
  } else if (b1->curL > minL && carry) {
    for (i = minL; i < maxL && b1->d[i] == CHARMAX; i++)
      b1->d[i] = 0;
    carry = (i == maxL);
    if (!carry)
      b1->d[i]++;
  }
  if (carry) {
    if (b1->curL == b1->memL) {
      b1->memL = b1->curL + 1;
      REALLOC(b1->d, b1->memL);
    }
    b1->d[b1->curL] = 1;
    b1->curL++;
  }
}

void
BiSub(                                  /* b1 = b1 - b2 */
    BIGINT b1,
    BIGINT b2
)
{
  unsigned i, borrow;
  int     digit;

  if (b2->curL > b1->curL)
    return;                             /* error */

  borrow = 0;
  for (i = 0; i < b2->curL; i++) {
    digit = b1->d[i] - borrow - b2->d[i];
    if (borrow = digit < 0)
      digit += CHARMAX + 1;
    b1->d[i] = digit;
  }

  if (borrow) {
    for (i = b2->curL; i < b1->curL && b1->d[i] == 0; i++)
      b1->d[i] = CHARMAX;
    if (i < b1->curL)
      b1->d[i]--;
  }
  while (b1->curL > 0 && b1->d[--b1->curL] == 0);
  b1->curL++;
}

void
BiMul(                                  /* b1 = b1 * b2 */
    BIGINT b1,
    BIGINT b2
)
{
  unsigned maxL, i, k, p, d0, d1, j;
  unsigned char *res;

  maxL = b1->curL + b2->curL;
  res = (unsigned char *) malloc(maxL);
  for (i = 0; i < maxL; i++)
    res[i] = 0;

  for (i = 0; i < b1->curL; i++)
    for (k = 0; k < b2->curL; k++) {
      p = b1->d[i] * b2->d[k];
      d0 = res[i + k] + (p & CHARMASK);
      d1 = res[i + k + 1] + (p >> CHARBITS);
      if (d0 > CHARMAX) {
        d0 -= CHARMAX + 1;
        d1++;
      }
      if (d1 > CHARMAX) {
        d1 -= CHARMAX + 1;
        for (j = i + k + 2; res[j] == CHARMAX; j++)
          res[j] = 0;
        res[j]++;
      }
      res[i + k] = d0;
      res[i + k + 1] = d1;
    }

  b1->memL = maxL;
  free(b1->d);
  b1->d = res;
  b1->curL = maxL;
  while (b1->curL > 0 && b1->d[--b1->curL] == 0);
  b1->curL++;
}

void
BiDiv(                                  /* b1 = b1 / b2 */
    BIGINT b1,
    BIGINT b2
)
{
  BIGINT  b2s, b2m;
  unsigned resL, b1L, b2L, k, d, p, carry;
  unsigned char *res, b1Prev;
  int     i;

  b1L = b1->curL;
  b2L = b2->curL;

  if (b1L < b2L) {
    b1->curL = 1;
    b1->d[0] = 0;
    return;
  }
  b2s = BiNew();
  b2m = BiNew();

  b2s->memL = b2s->curL = b1L;
  REALLOC(b2s->d, b2s->memL);
  b2m->memL = b1L + 1;
  REALLOC(b2m->d, b2m->memL);
  for (i = 0; i < b1L - b2L; i++) {
    b2s->d[i] = 0;
    b2m->d[i] = 0;
  }

  resL = b1L - b2L + 1;
  res = (unsigned char *) malloc(resL);

  b1Prev = 0;
  for (i = b1L - b2L; i >= 0; i--) {
    for (k = 0; k < b2L; k++)
      b2s->d[k + i] = b2->d[k];
    d = ((CHARMAX + 1) * b1Prev + b1->d[b2L - 1 + i])
        / b2->d[b2L - 1];
    b2m->curL = b2L + i + 1;
    carry = 0;
    for (k = i; k < b2L + i; k++) {
      p = d * b2s->d[k] + carry;
      b2m->d[k] = p & CHARMASK;
      carry = p >> CHARBITS;
    }
    b2m->d[b2L + i] = carry;
    while (b2m->curL > 1 && b2m->d[b2m->curL - 1] == 0)
      b2m->curL--;
    while (BiLess(b1, b2m)) {
      d--;
      BiSub(b2m, b2s);
    }
    res[i] = d;
    BiSub(b1, b2m);
    b1Prev = b1->d[b2L - 1 + i];
    b2s->curL--;
  }

  b1->memL = b1->curL = resL;
  free(b1->d);
  b1->d = res;
  while (b1->curL > 0 && b1->d[--b1->curL] == 0);
  b1->curL++;

  BiFree(b2s);
  BiFree(b2m);
}

void
BiMod(                                  /* b1 = b1 % b2 */
    BIGINT b1,
    BIGINT b2
)
{
  BIGINT  b2s, b2m;
  unsigned b1L, b2L, k, d, p, carry;
  unsigned char b1Prev;
  int     i;

  b1L = b1->curL;
  b2L = b2->curL;

  if (b1L < b2L)
    return;

  b2s = BiNew();
  b2m = BiNew();

  b2s->memL = b2s->curL = b1L;
  REALLOC(b2s->d, b2s->memL);
  b2m->memL = b1L + 1;
  REALLOC(b2m->d, b2m->memL);
  for (i = 0; i < b1L - b2L; i++) {
    b2s->d[i] = 0;
    b2m->d[i] = 0;
  }

  b1Prev = 0;
  for (i = b1L - b2L; i >= 0; i--) {
    for (k = 0; k < b2L; k++)
      b2s->d[k + i] = b2->d[k];
    d = ((CHARMAX + 1) * b1Prev + b1->d[b2L - 1 + i])
        / b2->d[b2L - 1];
    b2m->curL = b2L + i + 1;
    carry = 0;
    for (k = i; k < b2L + i; k++) {
      p = d * b2s->d[k] + carry;
      b2m->d[k] = p & CHARMASK;
      carry = p >> CHARBITS;
    }
    b2m->d[b2L + i] = carry;
    while (b2m->curL > 1 && b2m->d[b2m->curL - 1] == 0)
      b2m->curL--;
    while (BiLess(b1, b2m))
      BiSub(b2m, b2s);
    BiSub(b1, b2m);
    b1Prev = b1->d[b2L - 1 + i];
    b2s->curL--;
  }

  BiFree(b2s);
  BiFree(b2m);
}

void
BiDiv2(                                 /* b = b / 2 */
    BIGINT b
)
{
  int     i, oldBit, newBit;

  newBit = 0;
  for (i = b->curL; i-- > 0;) {
    oldBit = newBit;
    newBit = b->d[i] % 2;
    b->d[i] >>= 1;
    if (oldBit)
      b->d[i] += CHARHIGH;
  }

  if (b->d[b->curL - 1] == 0)
    b->curL--;
}

void
BiInc(                                  /* b = b + 1 */
    BIGINT b
)
{
  unsigned i;

  for (i = 0; i < b->curL && b->d[i] == CHARMAX; i++)
    b->d[i] = 0;
  if (i < b->curL)
    b->d[i]++;
  else {
    if (b->curL == b->memL) {
      b->memL = b->curL + 1;
      REALLOC(b->d, b->memL);
    }
    b->d[b->curL] = 1;
    b->curL++;
  }
}

void
BiDec(                                  /* b = b - 1 */
    BIGINT b
)
{
  unsigned i;

  for (i = 0; i < b->curL && b->d[i] == 0; i++)
    b->d[i] = CHARMAX;
  if (i == b->curL)
    return;                             /* error: BiDec(0) */
  b->d[i]--;
  if (i == b->curL - 1 && i != 0 && b->d[i] == 0)
    b->curL--;
}

int
BiOdd(                                  /* b % 2 == 1 */
    BIGINT b
)
{
  return b->d[0] & 1;
}

int
BiZero(                                 /* b == 0 */
    BIGINT b
)
{
  return b->curL == 1 && b->d[0] == 0;
}

int
BiEqual(                                /* b1 == b2 */
    BIGINT b1,
    BIGINT b2
)
{
  unsigned i;

  if (b1->curL != b2->curL)
    return 0;

  for (i = 0; i < b1->curL; i++)
    if (b1->d[i] != b2->d[i])
      return 0;

  return 1;
}

int
BiLess(                                 /* b1 < b2 */
    BIGINT b1,
    BIGINT b2
)
{
  unsigned i;

  if (b1->curL < b2->curL)
    return 1;
  if (b1->curL > b2->curL)
    return 0;

  for (i = b1->curL; i-- > 0;) {
    if (b1->d[i] < b2->d[i])
      return 1;
    if (b1->d[i] > b2->d[i])
      return 0;
  }

  return 0;
}

char   *
BiStr(                                  /* convert b to string */
    BIGINT b
)
{
  char   *s;
  unsigned sInd, i;

  sInd = 2 * b->curL;
  s = malloc(sInd + 1);
  s[sInd] = '\0';

  for (i = 0; i < b->curL; i++) {
    s[--sInd] = intToHexDigit((int) b->d[i] & HEXMASK);
    s[--sInd] = intToHexDigit((int) b->d[i] >> HEXBITS);
  }

  return s;
}

void
BiGcd(                                  /* gcd(b1, b2) */
    BIGINT b1,
    BIGINT b2
)
{
  BIGINT  b2c;

  b2c = BiNew();
  BiAss(b2c, b2);

  while (!(BiZero(b1) || BiZero(b2c)))
    if (BiLess(b1, b2c))
      BiMod(b2c, b1);
    else
      BiMod(b1, b2c);

  if (BiZero(b1))
    BiAss(b1, b2c);

  BiFree(b2c);
}
