/*
************************************************************************
*
*   GraphDraw.c - graphical display, drawing
*
*   Copyright (c) 1994-96
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/12/03
*   Pathname of SCCS file     : /local/home/kor/autopsy/src/graph/SCCS/s.GraphDraw.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <graph_draw.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>

#ifdef PRINT_TIME
#include <time.h>
#endif

#include <mat_vec.h>
#include <linlist.h>
#include <sg.h>
#include <pu.h>
#include <io.h>
#include <os_sleep.h>
#include <par_names.h>
#include <par_hand.h>
#include <data_hand.h>
#include <prim_hand.h>
#include <prim_draw.h>
#include <attr_struc.h>
#include <attr_mng.h>

#define STRIP_SIZE 100

typedef struct {
  GraphDrawF drawF;
  void *clientData;
} OverlayData;

static float ViewP[3];
static float ZoomFact;
static int DrawDelay;
static AttrP LastAttrP;
static SgLightState LightState;

static BOOL RemoveCBAdded = FALSE;

static int RedrawBlockLevel = 0;
static BOOL RefreshPending = FALSE;
static BOOL RedrawPending = FALSE;

static LINLIST OverlayList = NULL;

static GraphDrawF AltDrawF = NULL;
static void *AltClientData;

void
GraphDrawInit(void)
{
  float backCol[3];
  int i;

  if (! ParDefined(PN_STEREO_MODE))
    ParSetIntVal(PN_STEREO_MODE, SM_OFF);

  if (ParDefined(PN_DRAW_DELAY)) {
    DrawDelay = ParGetIntVal(PN_DRAW_DELAY);
  } else {
    DrawDelay = 0;
    ParSetIntVal(PN_DRAW_DELAY, DrawDelay);
  }

  if (ParDefined(PN_DRAW_PREC)) {
    SgSetPrecision(ParGetIntVal(PN_DRAW_PREC));
  } else {
    ParSetIntVal(PN_DRAW_PREC, 3);
    SgSetPrecision(3);
  }

  if (ParDefined(PN_PROJECTION)) {
    SgSetProjection(ParGetIntVal(PN_PROJECTION));
  } else {
    ParSetIntVal(PN_PROJECTION, SG_PROJ_ORTHO);
    SgSetProjection(SG_PROJ_ORTHO);
  }

  if (ParDefined(PN_NEAR_PLANE)) {
/*PG    SgSetNearPlane(ParGetDoubleVal(PN_NEAR_PLANE)); */
    SgSetNearPlane(ParGetDoubleVal(PN_NEAR_PLANE), TRUE);
  } else {
    ParSetDoubleVal(PN_NEAR_PLANE, 1.0);
/*PG    SgSetNearPlane(1.0); */
    SgSetNearPlane(1.0, TRUE);
  }

  if (ParDefined(PN_FAR_PLANE)) {
/*PG    SgSetFarPlane(ParGetDoubleVal(PN_FAR_PLANE)); */
    SgSetFarPlane(ParGetDoubleVal(PN_FAR_PLANE), TRUE);
  } else {
    ParSetDoubleVal(PN_FAR_PLANE, 20.0);
/*PG    SgSetFarPlane(20.0); */
    SgSetFarPlane(20.0, TRUE);
  }

  if (ParDefined(PN_VIEW_ANGLE)) {
    SgSetViewAngle(ParGetDoubleVal(PN_VIEW_ANGLE));
  } else {
    ParSetDoubleVal(PN_VIEW_ANGLE, 0.7);
    SgSetViewAngle(0.7);
  }

  if (ParDefined(PN_BACK_COLOR)) {
    for (i = 0; i < 3; i++)
      backCol[i] = ParGetDoubleArrVal(PN_BACK_COLOR, i);
    SgSetBackgroundColor(backCol[0], backCol[1], backCol[2]);
  } else {
    for (i = 0; i < 3; i++)
      ParSetDoubleArrVal(PN_BACK_COLOR, i, 1.0);
    SgSetBackgroundColor(1.0, 1.0, 1.0);
  }

  if (ParDefined(PN_FOG)) {
    SgSetFogMode(ParGetIntVal(PN_FOG));
  } else {
    ParSetIntVal(PN_FOG, SG_FOG_MODE_OFF);
    SgSetFogMode(SG_FOG_MODE_OFF);
  }

  if (ParDefined(PN_FOG_DENSITY)) {
    SgSetFogPar(SG_FOG_DENSITY, ParGetDoubleVal(PN_FOG_DENSITY));
  } else {
    ParSetDoubleVal(PN_FOG_DENSITY, 0.1);
    SgSetFogPar(SG_FOG_DENSITY, 0.1);
  }

  if (ParDefined(PN_FOG_START)) {
    SgSetFogPar(SG_FOG_START, ParGetDoubleVal(PN_FOG_START));
  } else {
    ParSetDoubleVal(PN_FOG_START, 7.0);
    SgSetFogPar(SG_FOG_START, 7.0);
  }

  if (ParDefined(PN_FOG_END)) {
    SgSetFogPar(SG_FOG_END, ParGetDoubleVal(PN_FOG_END));
  } else {
    ParSetDoubleVal(PN_FOG_END, 13.0);
    SgSetFogPar(SG_FOG_END, 13.0);
  }

  if (! ParDefined(PN_LIGHT_POS)) {
    ParSetDoubleArrVal(PN_LIGHT_POS, 0, -5.0);
    ParSetDoubleArrVal(PN_LIGHT_POS, 1, 5.0);
    ParSetDoubleArrVal(PN_LIGHT_POS, 2, 10.0);
    /* LightPos is set in drawInit() */
  }

  if (! ParDefined(PN_LIGHT))
    ParSetIntVal(PN_LIGHT, SG_LIGHT_INFINITE);
    /* Light is set in drawInit() */

  if (ParDefined(PN_ZOOM_FACT)) {
    ZoomFact = ParGetDoubleVal(PN_ZOOM_FACT);
  } else {
    ZoomFact = 1.0;
    ParSetDoubleVal(PN_ZOOM_FACT, ZoomFact);
  }

  ViewP[0] = 0.0;
  ViewP[1] = 0.0;
  ViewP[2] = 10.0;
  SgSetViewPoint(ViewP);
}

void
GraphDelaySet(int delay)
{
  DrawDelay = delay;
}

void
GraphZoomSet(float zoomFact)
{
  ZoomFact = zoomFact;
}

void
GraphSetAttr(AttrP attrP)
{
  if (attrP == LastAttrP)
    return;

  if (attrP->shadeModel == -1) {
    SgSetLight(SG_LIGHT_OFF);
  } else {
    SgSetShadeModel(attrP->shadeModel);
    SgSetLight(LightState);
  }

  /* alpha must be set before color because some graphics devices
     prefer to update it together with the color */
  SgSetMatProp(SG_MAT_ALPHA, attrP->alpha);
  SgSetColor(attrP->colR, attrP->colG, attrP->colB);
  SgSetMatProp(SG_MAT_AMBIENT_FACTOR, attrP->ambCoeff);
  SgSetMatProp(SG_MAT_DIFFUSE_FACTOR, attrP->diffCoeff);
  SgSetMatProp(SG_MAT_SPECULAR_FACTOR, attrP->specCoeff);
  SgSetMatProp(SG_MAT_SHININESS, attrP->shininess);
  SgSetMatProp(SG_MAT_REFLECTION, attrP->reflect);
  SgSetMatProp(SG_MAT_REFRACTION_INDEX, attrP->refract);
  SgSetTexture(attrP->texture);
  SgSetMatProp(SG_MAT_TEXTURE_SCALE, attrP->textureScale);
  SgSetMatProp(SG_MAT_BUMP_DEPTH, attrP->bumpDepth);
  SgSetMatProp(SG_MAT_BUMP_SCALE, attrP->bumpScale);
  SgSetLineWidth(attrP->lineWidth);
  SgSetLineStyle(attrP->lineStyle);
  SgSetFontSize(attrP->fontSize);

  LastAttrP = attrP;
}

static void
drawPeak(DhPeakP peakP, void *clientData)
{
  BOOL *draw3DP = clientData;
  AttrP attrP;
  short oldModel;
  Vec3 x;
  Vec3 x1, x2, x3, x4;
  char *labelFormat;
  DSTR labelStr;
  AppPeakP appPeakP;
  char numStr[20];
  float f;
  int i;

  attrP = DhPeakGetAttr(peakP);
  if (attrP->peakStyle == PS_INVISIBLE && attrP->labelFormat[0] == '\0')
    return;

  if (! *draw3DP) {
    oldModel = attrP->shadeModel;
    attrP->shadeModel = -1;
    GraphSetAttr(attrP);
    attrP->shadeModel = oldModel;
  } else {
    GraphSetAttr(attrP);
  }

  SgSetPickObj("Peak", peakP);

  DhPeakGetPoint(peakP, x);

  if (*draw3DP) {
    if (attrP->peakStyle == PS_CIRCLE) {
      SgDrawSphere(x, attrP->radius);
    } else if (attrP->peakStyle == PS_CROSS) {
      x1[0] = x[0] - 1.000000 * attrP->radius;
      x1[1] = x[1] - 0.577350 * attrP->radius;
      x1[2] = x[2] - 0.408248 * attrP->radius;

      x2[0] = x[0] + 0.500000 * attrP->radius;
      x2[1] = x[1] - 0.577350 * attrP->radius;
      x2[2] = x[2] - 0.408248 * attrP->radius;

      x3[0] = x[0] + 0.000000 * attrP->radius;
      x3[1] = x[1] + 1.154700 * attrP->radius;
      x3[2] = x[2] - 0.408248 * attrP->radius;

      x4[0] = x[0] + 0.000000 * attrP->radius;
      x4[1] = x[1] + 0.000000 * attrP->radius;
      x4[2] = x[2] + 1.224745 * attrP->radius;

      SgDrawLine(x1, x2);
      SgDrawLine(x1, x3);
      SgDrawLine(x1, x4);
      SgDrawLine(x2, x3);
    }
  } else {
    if (attrP->peakStyle == PS_CIRCLE) {
      SgDrawDisc2D(x, attrP->radius);

      if (attrP->length > 0.0) {
	x1[0] = x[0] + attrP->length * sin(attrP->angle);
	x1[1] = x[1] + attrP->length * cos(attrP->angle);

	SgDrawLine2D(x, x1);
      }
    } else {
      x1[0] = x[0] - attrP->radius;
      x1[1] = x[1] - attrP->radius;

      x2[0] = x[0] - attrP->radius;
      x2[1] = x[1] + attrP->radius;

      x3[0] = x[0] + attrP->radius;
      x3[1] = x[1] - attrP->radius;

      x4[0] = x[0] + attrP->radius;
      x4[1] = x[1] + attrP->radius;

      SgDrawLine2D(x1, x4);
      SgDrawLine2D(x2, x3);
    }
  }

  if (attrP->labelFormat[0] != '\0') {
    labelFormat = attrP->labelFormat;
    labelStr = DStrNew();
    appPeakP = DhPeakGetApp(peakP);
    while (labelFormat[0] != '\0') {
      if (labelFormat[0] == 'N') {
        (void) sprintf(numStr, "%d", DhPeakGetNumber(peakP));
        DStrAppStr(labelStr, numStr);
      } else if (labelFormat[0] == 'A') {
        (void) sprintf(numStr, "%.3f", appPeakP->amplitude);
        DStrAppStr(labelStr, numStr);
      } else if (labelFormat[0] == 'S') {
	f = appPeakP->symmetryErrorA[0];
	for (i = 1; i < appPeakP->dimensionNo; i++)
	  if (appPeakP->symmetryErrorA[i] > f)
	    f = appPeakP->symmetryErrorA[i];
        (void) sprintf(numStr, "%.3f", f);
        DStrAppStr(labelStr, numStr);
      } else if (labelFormat[0] == 'U') {
        (void) sprintf(numStr, "%.3f", appPeakP->uniformError);
        DStrAppStr(labelStr, numStr);
      } else if (labelFormat[0] == 'Q') {
        (void) sprintf(numStr, "%.3f", appPeakP->quality);
        DStrAppStr(labelStr, numStr);
      } else {
        DStrAppChar(labelStr, labelFormat[0]);
      }
      labelFormat++;
    }

    if (*draw3DP)
      SgDrawAnnot(x, DStrToStr(labelStr));
    else
      SgDrawAnnot2D(x, DStrToStr(labelStr));

    DStrFree(labelStr);
  }
}

static void
drawContA(ContourLine *contA, int contNo)
{
  int i;

  for (i = 0; i < contNo; i++)
    SgDrawPolyline2D(contA[i].xA, contA[i].pointNo);
}

static void
drawSurfA(IsoSurface *surfA, int surfNo)
{
  int surfI, meshI, maxSize, i;
  Vec3 *xA, *nvA;
  IsoSurfaceMesh *meshP;

  maxSize = 0;
  for (surfI = 0; surfI < surfNo; surfI++)
    for (meshI = 0; meshI < surfA[surfI].meshNo; meshI++) {
      meshP = surfA[surfI].meshA + meshI;
      if (meshP->pointNo > maxSize)
	maxSize = meshP->pointNo;
    }

  xA = malloc(maxSize * sizeof(*xA));
  nvA = malloc(maxSize * sizeof(*nvA));

  SgStartSurface(SG_SURF_UNIFORM);

  for (surfI = 0; surfI < surfNo; surfI++)
    for (meshI = 0; meshI < surfA[surfI].meshNo; meshI++) {
      meshP = surfA[surfI].meshA + meshI;

      for (i = 0; i < meshP->pointNo; i++) {
	Vec3Copy(xA[i], surfA[surfI].xA[meshP->indA[i]]);
	Vec3Copy(nvA[i], surfA[surfI].nvA[meshP->indA[i]]);
      }

      SgDrawTriMesh(xA, nvA, meshP->pointNo);
    }

  SgEndSurface();

  free(xA);
  free(nvA);
}

static void
drawPrim(PrimObjP primP, void *clientData)
{
  if (PrimGetType(primP) == PT_TEXT ||
      PrimGetType(primP) == PT_DRAWOBJ)
    /* handled seperately in drawNoTrans */
    return;

  GraphSetAttr(PrimGetAttr(primP));
  SgSetPickObj("Prim", primP);

  PrimDraw(primP, ZoomFact);
}

static void
drawNoTrans(PrimObjP primP, void *clientData)
{
  GraphSetAttr(PrimGetAttr(primP));
  SgSetPickObj("Prim", primP);

  PrimDraw(primP, ZoomFact);
}

static void
drawSpecObj(void *clientData)
{
  DhSpecP specP = clientData;
  PropRefP refP = PropGetRef(PROP_DISPLAYED, FALSE);
  AttrP attrP;
  int dim;
  BOOL peak3D;
  int resol[3];
  Vec3 *pointP, *normP;
  ContourLine *contA;
  int contNo;
  IsoSurface *surfA;
  int surfNo;
  int i1Max, i1, startI, size;

  attrP = DhSpecGetAttr(specP);

  GraphSetAttr(attrP);

  SgSetPickObj("Spec", specP);

  peak3D = TRUE;
  dim = DhSpecGetDim(specP);
  if (dim == 2 && attrP->specStyle == SS_CONTOUR) {
    (void) DhSpecGetVal(specP, resol);
    DhSpecGetCont(specP, &contA, &contNo);
    drawContA(contA, contNo);
    peak3D = FALSE;
  } else if (dim == 3 && attrP->specStyle == SS_SURFACE) {
    (void) DhSpecGetVal(specP, resol);
    DhSpecGetSurf(specP, &surfA, &surfNo);
    drawSurfA(surfA, surfNo);
  } else if (dim == 2 && attrP->specStyle != SS_INVISIBLE) {
    (void) DhSpecGetVal(specP, resol);
    pointP = DhSpecGetPoints(specP);
    normP = DhSpecGetNorm(specP);

    if (attrP->specStyle == SS_GRID)
      i1Max = resol[1];
    else
      i1Max = resol[1] - 1;

    if (attrP->specStyle == SS_SURFACE)
      SgStartSurface(SG_SURF_UNIFORM);

    for (i1 = 0; i1 < i1Max; i1++) {
      startI = 0;

      while (startI < resol[0] - 1) {
	if (startI + STRIP_SIZE <= resol[0])
	  size = STRIP_SIZE;
	else
	  size = resol[0] - startI;
	
	if (attrP->specStyle == SS_GRID)
	  SgDrawPolyline(pointP + i1 * resol[0] + startI, size);
	else
	  SgDrawStrip(pointP + (i1 + 1) * resol[0] + startI,
	      pointP + i1 * resol[0] + startI,
	      normP + (i1 + 1) * resol[0] + startI,
	      normP + i1 * resol[0] + startI, size);

	startI += size - 1;
      }
    }

    if (attrP->specStyle == SS_SURFACE)
      SgEndSurface();
  }

  DhSpecApplyPeak(refP, specP, drawPeak, &peak3D);
  PrimSpecApply(PT_ALL, refP, specP, drawPrim, NULL);
}

static void
drawSpec(DhSpecP specP, void *clientData)
{
  SpecAttrP specAttrP;
  Mat4 rotMat;
  Vec3 rotPoint, transVect;

  specAttrP = DhSpecGetSpecAttr(specP);

  if (! specAttrP->objDefined)
    return;

  DhSpecGetRotMat(specP, rotMat);
  DhSpecGetRotPoint(specP, rotPoint);
  DhSpecGetTransVect(specP, transVect);

  SgPushMatrix();

  SgScale(ZoomFact, ZoomFact, ZoomFact);
  SgTranslate(transVect);
  SgMultMatrix(rotMat);
  Vec3Scale(rotPoint, -1.0);
  SgTranslate(rotPoint);

  SgDrawObj(specAttrP->objId);

  SgPopMatrix();
}

static void
drawScene(void *clientData)
{
  PropRefP refP = PropGetRef(PROP_DISPLAYED, FALSE);

  DhApplySpec(refP, drawSpec, NULL);
  PrimApply(PT_TEXT, refP, drawNoTrans, NULL);
  PrimApply(PT_DRAWOBJ, refP, drawNoTrans, NULL);
}

static void
drawOverlay(void *clientData)
{
  OverlayData *dataP;

  dataP = ListFirst(OverlayList);
  while (dataP != NULL) {
    dataP->drawF(dataP->clientData);
    dataP = ListNext(OverlayList, dataP);
  }
}

static void
drawInit(void)
{
  float lightPos[3];
  int i;

  LastAttrP = NULL;

  /* setup lighting before matrix is changed */
  SgSetLight(SG_LIGHT_OFF);

  for (i = 0; i < 3; i++)
    lightPos[i] = ParGetDoubleArrVal(PN_LIGHT_POS, i);
  SgSetLightPosition(lightPos);

  SgSetShadeModel(SHADE_DEFAULT);
  LightState = ParGetIntVal(PN_LIGHT);
  SgSetLight(LightState);
}

void
specRemove(DhSpecP specP, void *clientData)
{
  SpecAttrP specAttrP;

  specAttrP = DhSpecGetSpecAttr(specP);
  if (specAttrP->objDefined)
    SgDestroyObj(specAttrP->objId);
}

void
GraphSpecAdd(DhSpecP specP)
{
  SpecAttrP specAttrP;

  specAttrP = DhSpecGetSpecAttr(specP);
  if (specAttrP->objDefined)
    return;

  if (! RemoveCBAdded) {
    DhAddSpecInvalidCB(specRemove, NULL);
    RemoveCBAdded = TRUE;
  }

  drawInit();
  specAttrP->objId = SgCreateObj(drawSpecObj, specP);
  specAttrP->objDefined = TRUE;
  DhSpecSetSpecAttr(specP, specAttrP);
}

static void
specChanged(DhSpecP specP, void *clientData)
{
  SpecAttrP specAttrP;

  specAttrP = DhSpecGetSpecAttr(specP);
  drawInit();
  SgReplaceObj(specAttrP->objId, drawSpecObj, specP);
  DhSpecSetSpecAttr(specP, specAttrP);
}

void
GraphSpecChanged(char *prop)
{
  DhApplySpec(PropGetRef(prop, FALSE), specChanged, NULL);
}

static void
graphRefresh(void)
{
  if (! SgRefresh())
    GraphRedraw();
}

void
GraphRedrawEnable(BOOL onOff)
{
  if (onOff) {
    RedrawBlockLevel--;
    if (RedrawBlockLevel == 0) {
      if (RedrawPending) {
        GraphRedraw();
        RefreshPending = FALSE;
        RedrawPending = FALSE;
      } else if (RefreshPending) {
        graphRefresh();
        RefreshPending = FALSE;
      }
    }
  } else {
    RedrawBlockLevel++;
  }
}

void
GraphRedrawNeeded(void)
{
  if (RedrawBlockLevel > 0)
    RedrawPending = TRUE;
  else
    GraphRedraw();
}

void
GraphRefreshNeeded(void)
{
  if (RedrawBlockLevel > 0)
    RefreshPending = TRUE;
  else
    graphRefresh();
}

void
drawFrame(GraphDrawF drawF, void *clientData)
{
#ifdef PRINT_TIME
  clock_t t0 = clock();
#endif
  float d;
  float x, y, w, h;

  d = ViewP[2] * 0.05;
  SgSetEye(SG_EYE_LEFT);

  switch (ParGetIntVal(PN_STEREO_MODE)) {
    case SM_OFF:
      ViewP[0] = 0.0;
      SgSetViewPoint(ViewP);
      SgUpdateView();
      drawF(clientData);
      break;
    case SM_LEFT:
      ViewP[0] = - d;
      SgSetViewPoint(ViewP);
      SgUpdateView();
      drawF(clientData);
      break;
    case SM_RIGHT:
      ViewP[0] = d;
      SgSetViewPoint(ViewP);
      SgUpdateView();
      drawF(clientData);
      break;
    case SM_SIDE_BY_SIDE:
      SgGetViewport(&x, &y, &w, &h);

      ViewP[0] = - d;
      SgSetViewPoint(ViewP);
      SgSetViewport(x, y, 0.5 * w, h);
      SgUpdateView();
      drawF(clientData);

      SgSetEye(SG_EYE_RIGHT);
      ViewP[0] = d;
      SgSetViewPoint(ViewP);
      SgSetViewport(x + 0.5 * w, y, 0.5 * w, h);
      SgUpdateView();
      drawF(clientData);

      SgSetViewport(x, y, w, h);

      break;
    case SM_CROSS_EYE:
      SgGetViewport(&x, &y, &w, &h);

      ViewP[0] = d;
      SgSetViewPoint(ViewP);
      SgSetViewport(x, y, 0.5 * w, h);
      SgUpdateView();
      drawF(clientData);

      SgSetEye(SG_EYE_RIGHT);
      ViewP[0] = - d;
      SgSetViewPoint(ViewP);
      SgSetViewport(x + 0.5 * w, y, 0.5 * w, h);
      SgUpdateView();
      drawF(clientData);

      SgSetViewport(x, y, w, h);

      break;
    case SM_HARDWARE:
      if (SgGetStereo()) {
        ViewP[0] = -d;
        SgSetViewPoint(ViewP);
        SgUpdateView();
        drawF(clientData);

        SgSetEye(SG_EYE_RIGHT);
        ViewP[0] = d;
        SgSetViewPoint(ViewP);
        SgUpdateView();
        drawF(clientData);
      } else {
        /* hardware stereo not supported */
        ViewP[0] = 0.0;
        SgSetViewPoint(ViewP);
        SgUpdateView();
        drawF(clientData);
      }
      break;
  }

#ifdef PRINT_TIME
  (void) printf("draw time: %d\n", clock() - t0);
#endif
}

void
GraphDraw(void)
/* Draw scene without calling IOEndFrame(), used for plotting
   and picking because they don't have a full IO device, only
   the Sg device. */
{
  drawInit();
  SgClear();

  if (AltDrawF != NULL)
    AltDrawF(AltClientData);
  else
    drawFrame(drawScene, NULL);

  SgEndFrame();
}

void
GraphRedraw(void)
/* Used for screen redraw, calls IOEndFrame. */
{
  drawInit();
  SgClear();

  if (AltDrawF != NULL) {
    AltDrawF(AltClientData);
  } else {
    drawFrame(drawScene, NULL);
    drawFrame(drawOverlay, NULL);
  }

  SgEndFrame();
  IOEndFrame();

  if (DrawDelay > 0)
    OsSleep(DrawDelay);
}

void
GraphDrawOverlay(GraphDrawF drawF, void *clientData)
{
  drawFrame(drawF, clientData);
}

void
GraphAddOverlay(GraphDrawF drawF, void *clientData)
{
  OverlayData data;

  if (OverlayList == NULL)
    OverlayList = ListOpen(sizeof(OverlayData));

  data.drawF = drawF;
  data.clientData = clientData;

  (void) ListInsertLast(OverlayList, &data);
}

void
GraphShowAlt(GraphDrawF drawF, void *clientData)
{
  AltDrawF = drawF;
  AltClientData = clientData;

  GraphRedrawNeeded();
}
