/*
************************************************************************
*
*   DataHand.c - data structure handling
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/21
*   Pathname of SCCS file     : /sgiext/autopsy/src/data/SCCS/s.DataHand.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <data_hand.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>

#include <app/spec_segm.h>
#include <specio/bruker.h>
#include <specio/easy.h>
#include <file_name.h>
#include <prop_tab.h>
#include <attr_struc.h>
#include <attr_mng.h>
#include "data_struc.h"

typedef struct {
  DhSpecInvalidCB invalidCB;
  void *clientData;
} InvalidSpecData;

static LINLIST SpecList = NULL;
static LINLIST InvalidSpecList = NULL;

void
DhAddSpecInvalidCB(DhSpecInvalidCB invalidCB, void *clientData)
{
  InvalidSpecData entry;

  entry.invalidCB = invalidCB;
  entry.clientData = clientData;

  if (InvalidSpecList == NULL)
    InvalidSpecList = ListOpen(sizeof(InvalidSpecData));

  (void) ListInsertLast(InvalidSpecList, &entry);
}

void
DhRemoveSpecInvalidCB(DhSpecInvalidCB invalidCB, void *clientData)
{
  InvalidSpecData *entryP;

  entryP = ListFirst(InvalidSpecList);
  while (entryP != NULL) {
    if (entryP->invalidCB == invalidCB && entryP->clientData == clientData) {
      ListRemove(InvalidSpecList, entryP);
      return;
    }
    entryP = ListNext(InvalidSpecList, entryP);
  }
}

void
callSpecInvalid(DhSpecP specP)
{
  InvalidSpecData *entryP, *nextEntryP;

  entryP = ListFirst(InvalidSpecList);
  while (entryP != NULL) {
    /* entryP could be removed from callback */
    nextEntryP = ListNext(InvalidSpecList, entryP);
    entryP->invalidCB(specP, entryP->clientData);
    entryP = nextEntryP;
  }
}

void
DhDestroyAll(void)
{
  ListClose(SpecList);
  SpecList = NULL;
}

static void
freeCont(DhSpecP specP)
{
  int i;

  if (specP->contA == NULL)
    return;

  for (i = 0; i < specP->contNo; i++)
    free(specP->contA[i].xA);

  free(specP->contA);
  specP->contA = NULL;
  specP->contNo = 0;
}

static void
freeSurf(DhSpecP specP)
{
  IsoSurface *surfP;
  int i, k;

  if (specP->surfA == NULL)
    return;

  for (i = 0; i < specP->surfNo; i++) {
    surfP = specP->surfA + i;

    free(surfP->xA);
    free(surfP->nvA);

    for (k = 0; k < surfP->meshNo; k++)
      free(surfP->meshA[k].indA);
  }

  free(specP->surfA);
  specP->surfA = NULL;
  specP->surfNo = 0;
}

static void
freePoints(DhSpecP specP)
{
  if (specP->points != NULL) {
    free(specP->points);
    specP->points = NULL;
  }

  if (specP->norm != NULL) {
    free(specP->norm);
    specP->norm = NULL;
  }

  freeCont(specP);
  freeSurf(specP);
}

static void
freeVal(DhSpecP specP)
{
  if (specP->type != ST_REGION && specP->val != NULL) {
    free(specP->val);
    specP->val = NULL;
  }

  freePoints(specP);
}

static void
freeSpec(void *p, void *clientData)
{
  DhSpecP specP = p;
  int i;

  callSpecInvalid(specP);

  freeVal(specP);

  if (specP->type == ST_NOISE) {
    AppNoiseDestroy(specP->u.noise.appNoiseP);
    specP->u.noise.specP->u.file.noiseP = NULL;
  } else if (specP->type == ST_REGION) {
    AppRegionRemove(specP->u.reg.appRegP);
  } else {
    DStrFree(specP->u.file.fileName);
    if (specP->type == ST_BRUKER)
      SpecIOBrukerClose(specP->u.file.appSpecP);
    else
      SpecIOEasyClose(specP->u.file.appSpecP);
    if (specP->u.file.noiseP != NULL)
      ListRemove(SpecList, specP->u.file.noiseP);
  }

  ListClose(specP->peakL);
  PropFreeTab(specP->propTab);
  AttrReturn(specP->attrP);
  SpecAttrReturn(specP->specAttrP);
}

static int
getSpecNum(void)
{
  DhSpecP prevSpecP;

  if (SpecList == NULL) {
    SpecList = ListOpen(sizeof(struct DhSpecS));
    ListAddDestroyCB(SpecList, freeSpec, NULL, NULL);
    return 1;
  }

  if (ListSize(SpecList) == 0)
    return 1;

  prevSpecP = ListLast(SpecList);
  return prevSpecP->num + 1;
}

static void
freePeak(void *p, void *clientData)
{
  DhPeakP peakP = p;

  AttrReturn(peakP->attrP);
  PropFreeTab(peakP->propTab);
}

static void
initSpec(DhSpecP specP, BOOL visible)
{
  struct AttrS attr;

  specP->scale = 1.0;
  specP->level = 0.000001;
  specP->val = NULL;
  specP->points = NULL;
  specP->norm = NULL;
  specP->contA = NULL;
  specP->contNo = 0;
  specP->surfA = NULL;
  specP->surfNo = 0;

  specP->name = DStrNew();
  specP->peakL = ListOpen(sizeof(struct DhPeakS));
  ListAddDestroyCB(specP->peakL, freePeak, NULL, NULL);

  Mat4Ident(specP->rotMat);
  Vec3Zero(specP->rotPoint);
  Vec3Zero(specP->transVect);
  specP->propTab = PropNewTab(TRUE);
  specP->attrP = AttrGetInit();
  specP->specAttrP = SpecAttrGet();

  if (visible && specP->dim == 2) {
    AttrCopy(&attr, specP->attrP);
    AttrReturn(specP->attrP);
    attr.specStyle = SS_CONTOUR;
    specP->attrP = AttrGet(&attr);
  }
}

DhSpecP
DhSpecNew(DSTR fileName, SpecType type)
{
  struct DhSpecS specS;
  int i;

  specS.type = type;

  if (type == ST_BRUKER)
    specS.u.file.appSpecP = SpecIOBrukerOpen(DStrToStr(fileName));
  else
    specS.u.file.appSpecP = SpecIOEasyOpen(DStrToStr(fileName));

  if (specS.u.file.appSpecP == NULL)
    return NULL;

  AppGetSpectrumInfo(specS.u.file.appSpecP, &specS.dim, specS.sizes);

  for (i = 0; i < specS.dim; i++) {
    specS.range[i][0] = 0;
    specS.range[i][1] = specS.sizes[i] - 1;
    specS.resol[i] = 64;
  }

  specS.u.file.fileName = DStrNew();
  DStrAssignDStr(specS.u.file.fileName, fileName);
  specS.u.file.noiseP = NULL;
  specS.num = getSpecNum();

  initSpec(&specS, TRUE);

  return ListInsertLast(SpecList, &specS);
}

DhSpecP
DhSpecNewNoise(DhSpecP specP)
{
  struct DhSpecS specS;
  int i;

  if (specP->type == ST_NOISE || specP->type == ST_REGION)
    return NULL;
  
  if (specP->u.file.noiseP != NULL)
    return specP->u.file.noiseP;

  specS.type = ST_NOISE;
  specS.dim = specP->dim;

  for (i = 0; i < specS.dim; i++) {
    specS.sizes[i] = specP->sizes[i];
    specS.range[i][0] = specP->range[i][0];
    specS.range[i][1] = specP->range[i][1];
    specS.resol[i] = specP->resol[i];
  }

  specS.num = getSpecNum();
  initSpec(&specS, FALSE);

  Mat4Copy(specS.rotMat, specP->rotMat);
  Vec3Copy(specS.rotPoint, specP->rotPoint);
  Vec3Copy(specS.transVect, specP->transVect);

  specS.u.noise.specP = specP;
  specS.u.noise.appNoiseP = AppNoisePrepare(
      specP->u.file.appSpecP, specP->range);

  specP->u.file.noiseP = ListInsertLast(SpecList, &specS);

  return specP->u.file.noiseP;
}

DhSpecP
DhRegNew(DhSpecP specP, AppRegionP appRegP)
{
  AppRegionDataP appRegDataP = (AppRegionDataP) appRegP;
  struct DhSpecS specS;
  int i;

  specS.type = ST_REGION;
  specS.dim = specP->dim;

  for (i = 0; i < specS.dim; i++) {
    specS.range[i][0] = appRegDataP->rangeA[i][0];
    specS.range[i][1] = appRegDataP->rangeA[i][1];
    specS.sizes[i] = specS.range[i][1] - specS.range[i][0] + 1;
    specS.resol[i] = specS.sizes[i];
  }

  specS.num = getSpecNum();
  initSpec(&specS, TRUE);

  specS.val = appRegDataP->dataA;

  Mat4Copy(specS.rotMat, specP->rotMat);
  Vec3Copy(specS.rotPoint, specP->rotPoint);
  Vec3Copy(specS.transVect, specP->transVect);

  specS.u.reg.specP = specP;
  specS.u.reg.appRegP = appRegP;

  (void) ListInsertLast(SpecList, &specS);
}

void
DhSpecDestroy(DhSpecP specP)
{
  DhSpecP nextSpecP;
  int ind;

  nextSpecP = ListNext(SpecList, specP);
  ind = specP->num;

  ListRemove(SpecList, specP);

  specP = nextSpecP;
  while (specP != NULL) {
    specP->num = ind;
    specP = ListNext(SpecList, specP);
    ind++;
  }
}

DhPeakP
DhPeakNew(DhSpecP specP, AppPeakP appPeakP)
{
  struct DhPeakS peakS;
  DhPeakP peakP, prevPeakP;

  if (specP->type == ST_REGION)
    specP = specP->u.reg.specP;

  peakS.appPeakP = appPeakP;

  peakS.specP = specP;
  prevPeakP = ListLast(specP->peakL);
  if (prevPeakP == NULL)
    peakS.num = 1;
  else
    peakS.num = prevPeakP->num + 1;
  
  peakP = ListInsertLast(specP->peakL, &peakS);

  peakP->propTab = PropNewTab(FALSE);
  peakP->attrP = AttrGetInit();

  return peakP;
}

void
DhPeakDestroy(DhPeakP peakP)
{
  LINLIST peakL;
  int ind;

  peakL = peakP->specP->peakL;
  ListRemove(peakL, peakP);

  peakP = ListFirst(peakL);
  ind = 1;
  while (peakP != NULL) {
    peakP->num = ind;
    peakP = ListNext(peakL, peakP);
    ind++;
  }
}

int
DhGetSpecNo(void)
{
  return ListSize(SpecList);
}

void
DhApplySpec(PropRefP refP, DhSpecApplyFunc applyF, void *clientData)
{
  DhSpecP specP, nextSpecP;

  specP = ListFirst(SpecList);
  while (specP != NULL) {
    nextSpecP = ListNext(SpecList, specP);
    if (specP->propTab[refP->index] & refP->mask)
      applyF(specP, clientData);
    specP = nextSpecP;
  }
}

void
DhApplyPeak(PropRefP refP, DhPeakApplyFunc applyF, void *clientData)
{
  DhSpecP specP, nextSpecP;

  specP = ListFirst(SpecList);
  while (specP != NULL) {
    if (specP->propTab[refP->index] & refP->mask)
      DhSpecApplyPeak(refP, specP, applyF, clientData);
    specP = ListNext(SpecList, specP);
  }
}

void
DhSpecApplyPeak(PropRefP refP, DhSpecP specP,
    DhPeakApplyFunc applyF, void *clientData)
{
  DhPeakP peakP, nextPeakP;

  peakP = ListFirst(specP->peakL);
  while (peakP != NULL) {
    nextPeakP = ListNext(specP->peakL, peakP);
    if (peakP->propTab[refP->index] & refP->mask)
      applyF(peakP, clientData);
    peakP = nextPeakP;
  }
}

DhPeakP
DhPeakFind(DhSpecP specP, int num)
{
  DhPeakP peakP;

  peakP = ListFirst(specP->peakL);
  while (peakP != NULL) {
    if (peakP->num == num)
      return peakP;
    peakP = ListNext(specP->peakL, peakP);
  }

  return NULL;
}

AppSpectrumP
DhSpecGetApp(DhSpecP specP)
{
  if (specP->type != ST_BRUKER && specP->type != ST_EASY)
    return NULL;

  return specP->u.file.appSpecP;
}

AppNoiseP
DhSpecGetNoise(DhSpecP specP)
{
  if (specP->type != ST_BRUKER && specP->type != ST_EASY)
    return NULL;

  if (specP->u.file.noiseP == NULL)
    return NULL;

  return specP->u.file.noiseP->u.noise.appNoiseP;
}

AppRegionP
DhRegGetApp(DhSpecP regP)
{
  if (regP->type != ST_REGION)
    return NULL;

  return regP->u.reg.appRegP;
}

AppPeakP
DhPeakGetApp(DhPeakP peakP)
{
  return peakP->appPeakP;
}

DhSpecP
DhSpecFindName(DSTR name)
{
  DhSpecP specP;

  specP = ListFirst(SpecList);
  while (specP != NULL) {
    if (DStrCmp(specP->name, name) == 0)
      return specP;
    specP = ListNext(SpecList, specP);
  }

  return NULL;
}

DhSpecP
DhSpecFindNumber(int num)
{
  DhSpecP specP;

  specP = ListFirst(SpecList);
  while (specP != NULL) {
    if (specP->num == num)
      return specP;
    specP = ListNext(SpecList, specP);
  }

  return NULL;
}

DhSpecP
DhRegGetSpec(DhSpecP specP)
{
  if (specP->type != ST_REGION)
    return NULL;

  return specP->u.reg.specP;
}

DhSpecP
DhPeakGetReg(DhPeakP peakP)
{
  return peakP->specP;
}

DhSpecP
DhPeakGetSpec(DhPeakP peakP)
{
  DhSpecP specP;

  specP = peakP->specP;
  if (specP->type == ST_REGION)
    specP = specP->u.reg.specP;

  return specP;
}

SpecType
DhSpecGetType(DhSpecP specP)
{
  return specP->type;
}

int
DhSpecGetDim(DhSpecP specP)
{
  return specP->dim;
}

void
DhSpecGetSizes(DhSpecP specP, int sizes[])
{
  int i;

  for (i = 0; i < specP->dim; i++)
    sizes[i] = specP->sizes[i];
}

void
DhSpecSetRegion(DhSpecP specP, int range[][2])
{
  DhSpecP regP;
  int i, k, t;

  if (specP->type == ST_REGION)
    return;

  freeVal(specP);

  for (i = 0; i < specP->dim; i++) {
    if (range[i][0] > range[i][1]) {
      t = range[i][0];
      range[i][0] = range[i][1];
      range[i][1] = t;
    }

    for (k = 0; k < 2; k++) {
      if (range[i][k] < 0)
	range[i][k] = 0;
      else if (range[i][k] >= specP->sizes[i])
	range[i][k] = specP->sizes[i] - 1;

      specP->range[i][k] = range[i][k];
    }
  }

  regP = ListFirst(SpecList);
  while (regP != NULL) {
    if (regP->type == ST_REGION && regP->u.reg.specP == specP) {
      freeCont(regP);
      freeSurf(regP);
    }
    regP = ListNext(SpecList, regP);
  }
}

void
DhSpecGetRegion(DhSpecP specP, int range[][2])
{
  int i, k;

  for (i = 0; i < specP->dim; i++)
    for (k = 0; k < 2; k++)
      range[i][k] = specP->range[i][k];
}

void
DhSpecSetResol(DhSpecP specP, int *resol)
{
  int i;

  if (specP->type == ST_REGION)
    return;

  freeVal(specP);

  for (i = 0; i < specP->dim; i++)
    specP->resol[i] = resol[i];
}

void
DhSpecSetScale(DhSpecP specP, float scale)
{
  freePoints(specP);

  specP->scale = scale;
}

static void
getResol(DhSpecP specP, int *resol)
{
  int i;

  for (i = 0; i < specP->dim; i++) {
    resol[i] = specP->range[i][1] - specP->range[i][0] + 1;
    if (resol[i] > specP->resol[i])
      resol[i] = specP->resol[i];
  }
}

float *
DhSpecGetVal(DhSpecP specP, int *resol)
{
  int totSize;
  int coord[APP_MAX_DIM];
  float val, d0, d1, s0, s1, zFact;
  int i0, i1, ind, i, k;
  float z, zLeft, zRight, zBottom, zTop;
  Vec3 v, vLeft, vRight, vBottom, vTop;

  getResol(specP, resol);
  totSize = 1;
  for (i = 0; i < specP->dim; i++)
    totSize *= resol[i];

  if (specP->val == NULL) {
    specP->val = malloc(totSize * sizeof(*specP->val));

    if (specP->type == ST_NOISE) {
      for (i = 0; i < totSize; i++)
	specP->val[i] = - MAXFLOAT;

      zFact = 1.0 / specP->u.noise.specP->u.file.maxVal;

      for (i = 0; i < specP->dim; i++)
	coord[i] = specP->range[i][0];
      
      for (;;) {
	val = zFact * AppNoiseGetMagnitude(specP->u.noise.appNoiseP, coord);

	i1 = 0;
	for (k = specP->dim - 1; k >= 0; k--)
	  i1 = i1 * resol[k] + (coord[k] - specP->range[k][0]) *
	      resol[k] / (specP->range[k][1] - specP->range[k][0] + 1);

	if (val > specP->val[i1])
	  specP->val[i1] = val;

	for (i = 0; i < specP->dim; i++)
	  if (coord[i] == specP->range[i][1]) {
	    coord[i] = specP->range[i][0];
	  } else {
	    coord[i]++;
	    break;
	  }
	
	if (i == specP->dim)
	  break;
      }
    } else {
      if (specP->type == ST_BRUKER)
	SpecIOBrukerGetCompressed(specP->u.file.appSpecP,
	    specP->range, resol, specP->val);
      else
	SpecIOEasyGetCompressed(specP->u.file.appSpecP,
	    specP->range, resol, specP->val);
      
      specP->u.file.maxVal = 0.0;
      for (i = 0; i < totSize; i++)
	if (specP->val[i] > specP->u.file.maxVal)
	  specP->u.file.maxVal = specP->val[i];
	else if (- specP->val[i] > specP->u.file.maxVal)
	  specP->u.file.maxVal = - specP->val[i];

      zFact = 1.0 / specP->u.file.maxVal;
      for (i = 0; i < totSize; i++)
	specP->val[i] *= zFact;
    }
  }

  if (specP->points == NULL && specP->dim == 2) {
    specP->points = malloc(totSize * sizeof(*specP->points));

    if (specP->type == ST_REGION) {
      s0 = 2.0 / (specP->u.reg.specP->range[0][1] -
	  specP->u.reg.specP->range[0][0] + 1);
      s1 = 2.0 / (specP->u.reg.specP->range[1][1] -
	  specP->u.reg.specP->range[1][0] + 1);
      d0 = -1.0 + 2.0 *
	  (specP->range[0][0] - specP->u.reg.specP->range[0][0] + 0.5) /
	  (specP->u.reg.specP->range[0][1] -
	   specP->u.reg.specP->range[0][0] + 1);
      d1 = -1.0 + 2.0 *
	  (specP->range[1][0] - specP->u.reg.specP->range[1][0] + 0.5) /
	  (specP->u.reg.specP->range[1][1] -
	   specP->u.reg.specP->range[1][0] + 1);
      zFact = specP->scale / specP->u.reg.specP->u.file.maxVal;
    } else {
      s0 = 2.0 / resol[0];
      s1 = 2.0 / resol[1];
      d0 = -1.0 + 0.5 * s0;
      d1 = -1.0 + 0.5 * s1;
      zFact = specP->scale;
    }

    for (i0 = 0; i0 < resol[0]; i0++)
      for (i1 = 0; i1 < resol[1]; i1++) {
	v[0] = d0 + i0 * s0;
	v[1] = d1 + i1 * s1;

	ind = i1 * resol[0] + i0;
	if (specP->type == ST_REGION) {
	  if (((AppRegionDataP) specP->u.reg.appRegP)->validA[ind])
	    v[2] = zFact * specP->val[ind] + 0.01;
	  else
	    v[2] = -0.1;
	} else {
	  v[2] = zFact * specP->val[ind];
	}

	Vec3Copy(specP->points[i1 * resol[0] + i0], v);
      }
  }

  if (specP->norm == NULL && specP->dim == 2) {
    specP->norm = malloc(totSize * sizeof(*specP->norm));
    s0 = 2.0 / resol[0];
    s1 = 2.0 / resol[1];

    for (i0 = 0; i0 < resol[0]; i0++)
      for (i1 = 0; i1 < resol[1]; i1++) {
	z = zFact * specP->val[i1 * resol[0] + i0];

	if (i0 == 0)
	  zLeft = 0.0;
	else
	  zLeft = zFact * specP->val[i1 * resol[0] + i0 - 1];

	if (i0 == resol[0] - 1)
	  zRight = 0.0;
	else
	  zRight = zFact * specP->val[i1 * resol[0] + i0 + 1];

	if (i1 == 0)
	  zBottom = 0.0;
	else
	  zBottom = zFact * specP->val[(i1 - 1) * resol[0] + i0];

	if (i1 == resol[1] - 1)
	  zTop = 0.0;
	else
	  zTop = zFact * specP->val[(i1 + 1) * resol[0] + i0];

	vLeft[0] = zLeft - z;
	vLeft[1] = 0.0;
	vLeft[2] = s0;
	Vec3Norm(vLeft);

	vRight[0] = z - zRight;
	vRight[1] = 0.0;
	vRight[2] = s0;
	Vec3Norm(vRight);

	vBottom[0] = 0.0;
	vBottom[1] = zBottom - z;
	vBottom[2] = s1;
	Vec3Norm(vBottom);

	vTop[0] = 0.0;
	vTop[1] = z - zTop;
	vTop[2] = s1;
	Vec3Norm(vTop);

	Vec3Copy(v, vLeft);
	Vec3Add(v, vRight);
	Vec3Add(v, vBottom);
	Vec3Add(v, vTop);
	Vec3Norm(v);

	Vec3Copy(specP->norm[i1 * resol[0] + i0], v);
      }
  }

  return specP->val;
}

Vec3 *
DhSpecGetPoints(DhSpecP specP)
/* must only be called after DhSpecGetVal() */
{
  return specP->points;
}

Vec3 *
DhSpecGetNorm(DhSpecP specP)
/* must only be called after DhSpecGetVal() */
{
  return specP->norm;
}

void
DhSpecSetLevel(DhSpecP specP, float level)
{
  specP->level = level;
  freeCont(specP);
  freeSurf(specP);
}

void
DhSpecGetCont(DhSpecP specP, ContourLine **contAP, int *contNoP)
{
  int resol[APP_MAX_DIM];
  float xMin, yMin, xRange, yRange, level;
  AppRegionDataP regDataP;
  float *val;
  ContourLine *contA;
  int contNo, i;

  if (specP->contA == NULL && specP->dim == 2) {
    getResol(specP, resol);

    if (specP->type == ST_REGION) {
      xMin = -1.0 + 2.0 *
	  (specP->range[0][0] - specP->u.reg.specP->range[0][0] + 0.5) /
	  (specP->u.reg.specP->range[0][1] -
	   specP->u.reg.specP->range[0][0] + 1);
      yMin = -1.0 + 2.0 *
	  (specP->range[1][0] - specP->u.reg.specP->range[1][0] + 0.5) /
	  (specP->u.reg.specP->range[1][1] -
	   specP->u.reg.specP->range[1][0] + 1);
      xRange = 2.0 * (specP->range[0][1] - specP->range[0][0]) /
	  (specP->u.reg.specP->range[0][1] -
	   specP->u.reg.specP->range[0][0] + 1);
      yRange = 2.0 * (specP->range[1][1] - specP->range[1][0]) /
	  (specP->u.reg.specP->range[1][1] -
	   specP->u.reg.specP->range[1][0] + 1);
      level = specP->level * specP->u.reg.specP->u.file.maxVal;

      regDataP = (AppRegionDataP) specP->u.reg.appRegP;
      if (level < regDataP->baseLevel)
	level = regDataP->baseLevel;

      AppRegionPushState(specP->u.reg.appRegP, APP_PUSH_DATA);
      val = regDataP->dataA;
      for (i = 0; i < specP->sizes[0] * specP->sizes[1]; i++)
	if (val[i] > level && ! regDataP->validA[i])
	  val[i] = 0.9 * level;
    } else {
      xMin = -1.0 + 1.0 / resol[0];
      yMin = -1.0 + 1.0 / resol[1];
      xRange = 2.0 - 2.0 / resol[0];
      yRange = 2.0 - 2.0 / resol[1];
      level = specP->level;
      val = specP->val;
    }

    for (;;) {
      if (resol[0] == 1 || resol[1] == 1)
	contNo = 0;
      else
	ContourLineCalc(val, resol[0], resol[1],
	    xMin, yMin, xRange, yRange, level,
	    &contA, &contNo);

      if (contNo == 0)
	break;

      if (specP->contNo == 0)
	specP->contA = malloc(contNo * sizeof(*specP->contA));
      else
	specP->contA = realloc(specP->contA,
	    (specP->contNo + contNo) * sizeof(*specP->contA));

      for (i = 0; i < contNo; i++)
	specP->contA[specP->contNo + i] = contA[i];

      free(contA);

      specP->contNo += contNo;
      level *= 1.4;
    }

    if (specP->type == ST_REGION)
      AppRegionPopState(specP->u.reg.appRegP);
  }

  *contAP = specP->contA;
  *contNoP = specP->contNo;
}

void
DhSpecGetSurf(DhSpecP specP, IsoSurface **surfAP, int *surfNoP)
/* must only be called after DhSpecGetVal() */
{
  int resol[APP_MAX_DIM];
  float xMin, yMin, zMin, xRange, yRange, zRange, level;
  AppRegionDataP regDataP;
  float *val;
  int i;

  if (specP->surfA == NULL && specP->dim == 3) {
    getResol(specP, resol);

    if (specP->type == ST_REGION) {
      xMin = -1.0 + 2.0 *
	  (specP->range[0][0] - specP->u.reg.specP->range[0][0] + 0.5) /
	  (specP->u.reg.specP->range[0][1] -
	   specP->u.reg.specP->range[0][0] + 1);
      yMin = -1.0 + 2.0 *
	  (specP->range[1][0] - specP->u.reg.specP->range[1][0] + 0.5) /
	  (specP->u.reg.specP->range[1][1] -
	   specP->u.reg.specP->range[1][0] + 1);
      zMin = -1.0 + 2.0 *
	  (specP->range[2][0] - specP->u.reg.specP->range[2][0] + 0.5) /
	  (specP->u.reg.specP->range[2][1] -
	   specP->u.reg.specP->range[2][0] + 1);
      xRange = 2.0 * (specP->range[0][1] - specP->range[0][0]) /
	  (specP->u.reg.specP->range[0][1] -
	   specP->u.reg.specP->range[0][0] + 1);
      yRange = 2.0 * (specP->range[1][1] - specP->range[1][0]) /
	  (specP->u.reg.specP->range[1][1] -
	   specP->u.reg.specP->range[1][0] + 1);
      zRange = 2.0 * (specP->range[2][1] - specP->range[2][0]) /
	  (specP->u.reg.specP->range[2][1] -
	   specP->u.reg.specP->range[2][0] + 1);
      level = specP->level * specP->u.reg.specP->u.file.maxVal;

      regDataP = (AppRegionDataP) specP->u.reg.appRegP;
      if (level < regDataP->baseLevel)
	level = regDataP->baseLevel;
      
      AppRegionPushState(specP->u.reg.appRegP, APP_PUSH_DATA);
      val = regDataP->dataA;
      for (i = 0; i < specP->sizes[0] * specP->sizes[1] * specP->sizes[2]; i++)
	if (val[i] > level && ! regDataP->validA[i])
	  val[i] = 0.9 * level;
    } else {
      xMin = -1.0 + 1.0 / resol[0];
      yMin = -1.0 + 1.0 / resol[1];
      zMin = -1.0 + 1.0 / resol[2];
      xRange = 2.0 - 2.0 / resol[0];
      yRange = 2.0 - 2.0 / resol[1];
      zRange = 2.0 - 2.0 / resol[2];
      level = specP->level;
      val = specP->val;
    }

    IsoSurfaceCalc(val, resol[0], resol[1], resol[2],
	xMin, yMin, zMin, xRange, yRange, zRange,
	level, TRUE,
	&specP->surfA, &specP->surfNo);

    if (specP->type == ST_REGION)
      AppRegionPopState(specP->u.reg.appRegP);
  }

  *surfAP = specP->surfA;
  *surfNoP = specP->surfNo;
}

void
DhPeakGetPoint(DhPeakP peakP, Vec3 coord)
{
  DhSpecP specP;
  float *shifts;

  specP = DhPeakGetSpec(peakP);
  shifts = peakP->appPeakP->positionA;

  coord[0] = - 1.0 + 2.0 * (shifts[0] - specP->range[0][0] + 0.5) /
      (specP->range[0][1] - specP->range[0][0] + 1);
  coord[1] = - 1.0 + 2.0 * (shifts[1] - specP->range[1][0] + 0.5) /
      (specP->range[1][1] - specP->range[1][0] + 1);
  
  if (specP->dim == 2)
    coord[2] = specP->scale * peakP->appPeakP->amplitude;
  else
    coord[2] = - 1.0 + 2.0 * (shifts[2] - specP->range[2][0] + 0.5) /
	(specP->range[2][1] - specP->range[2][0] + 1);
}

void
DhSpecSetProp(PropRefP refP, DhSpecP specP, BOOL onOff)
{
  specP->propTab = PropChange(specP->propTab, refP, onOff);
}

void
DhPeakSetProp(PropRefP refP, DhPeakP peakP, BOOL onOff)
{
  DhSpecP specP = peakP->specP;

  peakP->propTab = PropChange(peakP->propTab, refP, onOff);
  if (onOff)
    specP->propTab = PropChange(specP->propTab, refP, onOff);
}

BOOL
DhSpecGetProp(PropRefP refP, DhSpecP specP)
{
  return (specP->propTab[refP->index] & refP->mask) != 0;
}

BOOL
DhPeakGetProp(PropRefP refP, DhPeakP peakP)
{
  return (peakP->propTab[refP->index] & refP->mask) != 0;
}

unsigned *
DhSpecGetPropTab(DhSpecP specP)
{
  return specP->propTab;
}

unsigned *
DhPeakGetPropTab(DhPeakP peakP)
{
  return peakP->propTab;
}

void
DhSpecSetName(DhSpecP specP, DSTR name)
{
  DStrAssignDStr(specP->name, name);
}

void
DhSpecSetRotMat(DhSpecP specP, Mat4 mat)
{
  Mat4Copy(specP->rotMat, mat);
}

void
DhSpecSetRotPoint(DhSpecP specP, Vec3 p)
{
  Vec3Copy(specP->rotPoint, p);
}

void
DhSpecSetTransVect(DhSpecP specP, Vec3 v)
{
  Vec3Copy(specP->transVect, v);
}

void
DhSpecSetAttr(DhSpecP specP, AttrP attrP)
{
  specP->attrP = attrP;
}

void
DhSpecSetSpecAttr(DhSpecP specP, SpecAttrP attrP)
{
  specP->specAttrP = attrP;
}

void
DhPeakSetAttr(DhPeakP peakP, AttrP attrP)
{
  peakP->attrP = attrP;
}

DSTR
DhSpecGetName(DhSpecP specP)
{
  return specP->name;
}

int
DhSpecGetNumber(DhSpecP specP)
{
  return specP->num;
}

void
DhSpecGetRotMat(DhSpecP specP, Mat4 mat)
{
  Mat4Copy(mat, specP->rotMat);
}

void
DhSpecGetRotPoint(DhSpecP specP, Vec3 p)
{
  Vec3Copy(p, specP->rotPoint);
}

void
DhSpecGetTransVect(DhSpecP specP, Vec3 v)
{
  Vec3Copy(v, specP->transVect);
}

AttrP
DhSpecGetAttr(DhSpecP specP)
{
  return specP->attrP;
}

SpecAttrP
DhSpecGetSpecAttr(DhSpecP specP)
{
  return specP->specAttrP;
}

int
DhPeakGetNumber(DhPeakP peakP)
{
  return peakP->num;
}

AttrP
DhPeakGetAttr(DhPeakP peakP)
{
  return peakP->attrP;
}

void
DhTransfVec(Vec3 x, DhSpecP specP)
{
  Vec4 w;

  Vec3Sub(x, specP->rotPoint);
  Vec3To4(w, x);
  Mat4VecMult(w, specP->rotMat);
  Vec4To3(x, w);
  Vec3Add(x, specP->transVect);
}
