/*
************************************************************************
*
*   ExDefProp.c - DefProp* commands
*
*   Copyright (c) 1994
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/13
*   Pathname of SCCS file     : /sgiext/autopsy/src/cmdprop/SCCS/s.ExDefProp.c
*   SCCS identification       : 1.1
*
************************************************************************
*/

#include <cmd_prop.h>

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <pu.h>
#include <arg.h>
#include <expr.h>
#include <data_hand.h>
#include <prim_hand.h>
#include <graph_draw.h>

typedef struct {
  ExprP exprP;
  BOOL val;
  PropRefP refP;
  int no;
} EvalData;

static DataEntityType
getEntType(char *name)
{
  if (strcmp(name, "Spec") == 0)
    return DE_SPEC;
  else if (strcmp(name, "Peak") == 0)
    return DE_PEAK;
  else
    return DE_PRIM;
}

static char *
getEntName(DataEntityType entType)
{
  switch(entType) {
    case DE_SPEC:
      return "spec";
    case DE_PEAK:
      return "peak";
    default:
      return "prim";
  }
}

static ExprP
getExpr(DataEntityType entType, char *str, ExprResType *resTypeP)
{
  ExprP exprP;
  DSTR fullStr, partStr;
  BOOL prevShort, inStr;

  fullStr = DStrNew();
  partStr = DStrNew();

  prevShort = FALSE;
  inStr = FALSE;

  while (*str != '\0') {
    if (*str == '#' && ! inStr) {
      str++;
      DStrAppChar(partStr, '(');

      for (;;) {
	if (isdigit(*str)) {
	  DStrAppStr(partStr, "num = ");
	  while (strchr(", )", *str) == NULL) {
	    if (*str == '-')
	      DStrAppStr(partStr, "..");
	    else
	      DStrAppChar(partStr, *str);
	    str++;
	  }
	} else {
	  DStrAppStr(partStr, "name = ");
	  DStrAppChar(partStr, '"');
	  while (strchr(", )", *str) == NULL) {
	    DStrAppChar(partStr, *str);
	    str++;
	  }
	  DStrAppChar(partStr, '"');
	}

	if (*str == ',') {
	  DStrAppStr(partStr, " || ");
	  str++;
	} else {
	  break;
	}
      }

      DStrAppChar(partStr, ')');
    }

    if (DStrLen(partStr) > 0) {
      if (prevShort)
	DStrAppStr(fullStr, " || ");
      else
	DStrAppStr(fullStr, " (");

      DStrAppDStr(fullStr, partStr);
      DStrAssignStr(partStr, "");

      prevShort = TRUE;

      while (*str == ' ')
	str++;
    } else {
      if (prevShort)
	DStrAppStr(fullStr, ") ");

      if (*str == '"')
	inStr = ! inStr;

      DStrAppChar(fullStr, *str);
      str++;

      prevShort = FALSE;
    }
  }

  if (prevShort)
    DStrAppStr(fullStr, ")");

  exprP = ExprCompile(entType, DStrToStr(fullStr), resTypeP);

  DStrFree(fullStr);
  DStrFree(partStr);

  return exprP;
}

static void
countNo(EvalData *evalP, BOOL val)
{
  if (val)
    evalP->no++;
}

static void
evalSpec(DhSpecP specP, void *clientData)
{
  EvalData *evalP = clientData;
  ExprRes exprRes;

  if (evalP->exprP == NULL) {
    DhSpecSetProp(evalP->refP, specP, evalP->val);
    countNo(evalP, evalP->val);
  } else {
    ExprEval(specP, evalP->exprP, &exprRes);
    DhSpecSetProp(evalP->refP, specP, exprRes.u.boolVal);
    countNo(evalP, exprRes.u.boolVal);
  }
}

static void
evalPeak(DhPeakP peakP, void *clientData)
{
  EvalData *evalP = clientData;
  ExprRes exprRes;

  if (evalP->exprP == NULL) {
    DhPeakSetProp(evalP->refP, peakP, evalP->val);
    countNo(evalP, evalP->val);
  } else {
    ExprEval(peakP, evalP->exprP, &exprRes);
    DhPeakSetProp(evalP->refP, peakP, exprRes.u.boolVal);
    countNo(evalP, exprRes.u.boolVal);
  }
}

static void
evalPrim(PrimObjP primP, void *clientData)
{
  EvalData *evalP = clientData;
  ExprRes exprRes;

  if (evalP->exprP == NULL) {
    PrimSetProp(evalP->refP, primP, evalP->val);
    countNo(evalP, evalP->val);
  } else {
    ExprEval(primP, evalP->exprP, &exprRes);
    PrimSetProp(evalP->refP, primP, exprRes.u.boolVal);
    countNo(evalP, exprRes.u.boolVal);
  }
}

static void
setProp(DataEntityType entType, char *prop, EvalData *evalP)
{
  PropRefP allRefP;
  char numBuf[20];
  DSTR statStr;

  allRefP = PropGetRef(PROP_ALL, FALSE);

  evalP->refP = PropGetRef(prop, TRUE);
  evalP->no = 0;

  switch (entType) {
    case DE_SPEC:
      DhApplySpec(allRefP, evalSpec, evalP);
      break;
    case DE_PEAK:
      DhApplyPeak(allRefP, evalPeak, evalP);
      break;
    case DE_PRIM:
      PrimApply(PT_ALL, allRefP, evalPrim, evalP);
      break;
  }

  statStr = DStrNew();
  (void) sprintf(numBuf, "%d", evalP->no);
  DStrAppStr(statStr, numBuf);
  DStrAppChar(statStr, ' ');
  DStrAppStr(statStr, getEntName(entType));
  if (evalP->no != 1)
    DStrAppChar(statStr, 's');
  DStrAppChar(statStr, ' ');
  DStrAppStr(statStr, prop);
  PuSetTextField(PU_TF_STATUS, DStrToStr(statStr));
  DStrFree(statStr);
}

#define ARG_NUM 2

ErrCode
ExDefProp(char *cmd)
{
  ArgDescr arg[ARG_NUM];
  ErrCode errCode;
  DataEntityType entType;
  char *exprStr;
  EvalData evalS;
  ExprResType resType;

  arg[0].type = AT_STR;
  arg[1].type = AT_STR;

  ArgInit(arg, ARG_NUM);

  arg[0].prompt = "Property";
  arg[1].prompt = "Definition";

  errCode = ArgGet(arg, ARG_NUM);
  if (errCode != EC_OK) {
    ArgCleanup(arg, ARG_NUM);
    return errCode;
  }

  entType = getEntType(cmd + 7);
  exprStr = DStrToStr(arg[1].v.strVal);

  if (exprStr[0] == '\0') {
    evalS.exprP = NULL;
    evalS.val = TRUE;
  } else if (strcmp(exprStr, "0") == 0) {
    evalS.exprP = NULL;
    evalS.val = FALSE;
  } else {
    evalS.exprP = getExpr(entType, exprStr, &resType);
    if (evalS.exprP == NULL) {
      CipSetError(ExprGetErrorMsg());
      ArgCleanup(arg, ARG_NUM);
      return EC_ERROR;
    }
    if (resType != ER_BOOL) {
      CipSetError("expression must be boolean");
      ArgCleanup(arg, ARG_NUM);
      ExprFree(evalS.exprP);
      return EC_ERROR;
    }
  }

  setProp(entType, DStrToStr(arg[0].v.strVal), &evalS);

  if (strcmp(DStrToStr(arg[0].v.strVal), PROP_DISPLAYED) == 0) {
    GraphSpecChanged(PROP_DISPLAYED);
    GraphRedrawNeeded();
  }

  ArgCleanup(arg, ARG_NUM);
  if (evalS.exprP != NULL)
    ExprFree(evalS.exprP);

  return EC_OK;
}

#undef ARG_NUM
#define ARG_NUM 1

ErrCode
ExUndefProp(char *cmd)
{
  ArgDescr arg[ARG_NUM];
  ErrCode errCode;
  PropRefP refP;
  EvalData evalS;

  arg[0].type = AT_STR;

  ArgInit(arg, ARG_NUM);

  arg[0].prompt = "Property";

  errCode = ArgGet(arg, ARG_NUM);
  if (errCode != EC_OK) {
    ArgCleanup(arg, ARG_NUM);
    return errCode;
  }

  refP = PropGetRef(PROP_ALL, FALSE);
  ArgCleanup(arg, ARG_NUM);
  if (refP == NULL)
    return EC_OK;

  evalS.exprP = NULL;
  evalS.val = FALSE;
  evalS.refP = PropGetRef(DStrToStr(arg[0].v.strVal), TRUE);

  DhApplySpec(refP, evalSpec, &evalS);
  PrimApply(PT_ALL, refP, evalPrim, &evalS);

  return EC_OK;
}

#undef ARG_NUM
#define ARG_NUM 1

ErrCode
ExSelect(char *cmd)
{
  ArgDescr arg[ARG_NUM];
  ErrCode errCode;
  DataEntityType entType;
  char *exprStr;
  EvalData evalS;
  ExprResType resType;

  arg[0].type = AT_STR;

  ArgInit(arg, ARG_NUM);

  arg[0].prompt = "Expression";

  errCode = ArgGet(arg, ARG_NUM);
  if (errCode != EC_OK) {
    ArgCleanup(arg, ARG_NUM);
    return errCode;
  }

  entType = getEntType(cmd + 6);
  exprStr = DStrToStr(arg[0].v.strVal);

  if (exprStr[0] == '\0') {
    evalS.exprP = NULL;
    evalS.val = TRUE;
  } else if (strcmp(exprStr, "0") == 0) {
    evalS.exprP = NULL;
    evalS.val = FALSE;
  } else {
    evalS.exprP = getExpr(entType, exprStr, &resType);
    if (evalS.exprP == NULL) {
      CipSetError(ExprGetErrorMsg());
      ArgCleanup(arg, ARG_NUM);
      return EC_ERROR;
    }
    if (resType != ER_BOOL) {
      CipSetError("expression must be boolean");
      ArgCleanup(arg, ARG_NUM);
      ExprFree(evalS.exprP);
      return EC_ERROR;
    }
  }

  setProp(entType, PROP_SELECTED, &evalS);
  ArgCleanup(arg, ARG_NUM);
  if (evalS.exprP != NULL)
    ExprFree(evalS.exprP);

  return EC_OK;
}
