/*
************************************************************************
*
*   ExWritePeak.c - WritePeak command
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/13
*   Pathname of SCCS file     : /sgiext/autopsy/src/cmdio/SCCS/s.ExWritePeak.c
*   SCCS identification       : 1.1
*
************************************************************************
*/

#include <cmd_io.h>

#include <stdio.h>

#include <app/spec_access.h>

#include <hashtab.h>
#include <g_file.h>
#include <prog_vers.h>
#include <curr_dir.h>
#include <arg.h>
#include <data_hand.h>

typedef struct {
  GFile gf;
  AppSpectrumP appSpecP;
  int dim;
  int peakNum;
  BOOL ok;
  HASHTABLE corrTab;
} WriteData;

typedef struct {
  AppShapeP appShapeP;
  LINLIST numList;
} CorrData;

static unsigned
hashFunc(void *p, unsigned size)
{
  CorrData *corrP = p;

  return (unsigned) corrP->appShapeP % size;
}

static int
compFunc(void *p1, void *p2)
{
  CorrData *corr1P = p1;
  CorrData *corr2P = p2;

  if (corr1P->appShapeP == corr2P->appShapeP)
    return 0;
  else
    return 1;
}

#define CHECK_RES(s) if (s != GF_RES_OK) {dataP->ok = FALSE; return;}

static void
writePeak(DhPeakP peakP, void *clientData)
{
  WriteData *dataP = clientData;
  AppPeakP appPeakP;
  float shiftA[APP_MAX_DIM];
  DSTR line;
  char buf[50];
  int color;
  float integr, err;
  CorrData corr, *corrP;
  int dom;

  appPeakP = DhPeakGetApp(peakP);

  AppConvertSpectrumUnit(dataP->appSpecP, appPeakP->positionA, "point",
      shiftA, "ppm");

  line = DStrNew();

  (void) sprintf(buf, "%6d", dataP->peakNum);
  DStrAppStr(line, buf);

  for (dom = dataP->dim - 1; dom >= 0; dom--) {
    (void) sprintf(buf, " %7.3f", shiftA[dom]);
    DStrAppStr(line, buf);
  }

  if (appPeakP->quality > 0.9)
    color = 1;
  else if (appPeakP->quality > 0.7)
    color = 2;
  else if (appPeakP->quality > 0.5)
    color = 3;
  else if (appPeakP->quality > 0.3)
    color = 4;
  else if (appPeakP->quality > 0.1)
    color = 5;
  else
    color = 6;

  AppPeakGetIntegral(appPeakP, &integr, &err);

  (void) sprintf(buf, " %1d %-9s %10.3e %9.2e %c 0",
      color, "N", integr, 100.0 * err / integr, 'a');
  DStrAppStr(line, buf);

  for (dom = dataP->dim - 1; dom >= 0; dom--) {
    (void) sprintf(buf, " %5d", 0);
    DStrAppStr(line, buf);
  }

  (void) sprintf(buf, " #QU %5.3f", appPeakP->quality);
  DStrAppStr(line, buf);

  CHECK_RES(GFileWriteStr(dataP->gf, DStrToStr(line)));
  CHECK_RES(GFileWriteNL(dataP->gf));

  DStrFree(line);

  for (dom = 0; dom < dataP->dim; dom++) {
    corr.appShapeP = appPeakP->shapePA[dom];
    corrP = HashSearch(dataP->corrTab, &corr);
    if (corrP == NULL) {
      corr.numList = ListOpen(sizeof(int));
      corrP = HashInsert(dataP->corrTab, &corr, FALSE);
    }
    (void) ListInsertLast(corrP->numList, &dataP->peakNum);
  }

  dataP->peakNum++;
}

static void
writeCorr(void *p, void *clientData)
{
  CorrData *corrP = p;
  WriteData *dataP = clientData;
  DSTR line;
  char buf[50];
  int *numP, idx;
  float c1, c2, c;

  if (ListSize(corrP->numList) < 2)
    return;

  line = DStrNew();

  (void) sprintf(buf, "%1d", dataP->dim - corrP->appShapeP->domain);
  DStrAppStr(line, buf);

  c1 = corrP->appShapeP->clusterDiff;
  c2 = corrP->appShapeP->otherDiff;
  (void) sprintf(buf, "%6.3f", c2 / (c1 + c2));
  DStrAppStr(line, buf);

  numP = ListFirst(corrP->numList);
  idx = 0;
  while (numP != NULL) {
    (void) sprintf(buf, "%7d", *numP);
    if (idx > 0 && idx % 9 == 0) {
      CHECK_RES(GFileWriteStr(dataP->gf, DStrToStr(line)));
      CHECK_RES(GFileWriteNL(dataP->gf));
      DStrAssignStr(line, "       ");
    }
    DStrAppStr(line, buf);
    numP = ListNext(corrP->numList, numP);
    idx++;
  }

  CHECK_RES(GFileWriteStr(dataP->gf, DStrToStr(line)));
  CHECK_RES(GFileWriteNL(dataP->gf));

  DStrFree(line);
}

static void
freeCorr(void *p, void *clientData)
{
  CorrData *corrP = p;

  ListClose(corrP->numList);
}

static void
writeSpecPeak(DhSpecP specP, void *clientData)
{
  WriteData *dataP = clientData;
  char buf[50];

  if (DhSpecGetType(specP) == ST_REGION)
    return;

  dataP->appSpecP = DhSpecGetApp(specP);
  dataP->dim = DhSpecGetDim(specP);

  (void) sprintf(buf, "# Number of dimensions %d", dataP->dim);
  CHECK_RES(GFileWriteStr(dataP->gf, buf));
  CHECK_RES(GFileWriteNL(dataP->gf));
  (void) sprintf(buf, "#FORMAT xeasy%dD_QU", dataP->dim);
  CHECK_RES(GFileWriteStr(dataP->gf, buf));
  CHECK_RES(GFileWriteNL(dataP->gf));

  dataP->corrTab = HashOpen(997, sizeof(CorrData), hashFunc, compFunc);
  HashAddDestroyCB(dataP->corrTab, freeCorr, NULL, NULL);

  DhSpecApplyPeak(PropGetRef(PROP_SELECTED, FALSE), specP, writePeak, dataP);

  CHECK_RES(GFileWriteStr(dataP->gf, "#CORRELATIONS"));
  CHECK_RES(GFileWriteNL(dataP->gf));
  HashApply(dataP->corrTab, writeCorr, dataP);

  HashClose(dataP->corrTab);
}

ErrCode
ExWritePeak(char *cmd)
{
  GFile gf;
  DSTR fileName;
  ErrCode errCode;
  WriteData writeData;

  fileName = DStrNew();
  errCode = ArgGetFilename(fileName, CurrDirGet(), "*.peaks");
  if (errCode != EC_OK) {
    DStrFree(fileName);
    return errCode;
  }

  gf = GFileOpenNew(DStrToStr(fileName), GF_FORMAT_ASCII);
  DStrFree(fileName);
  if (gf == NULL)
    return EC_ERROR;

  writeData.gf = gf;
  writeData.ok = TRUE;
  writeData.peakNum = 1;

  DhApplySpec(PropGetRef(PROP_SELECTED, FALSE), writeSpecPeak, &writeData);

  GFileClose(gf);

  if (! writeData.ok)
    return EC_ERROR;

  return EC_OK;
}
