#include <stdio.h>
#include <stdlib.h>

#include <app/spec_access.h>
#include <app/spec_noise.h>
#include <app/spec_segm.h>
#include <app/shape_list.h>
#include <app/elim.h>
#include <app/peak.h>
#include <app/deconv.h>
#include <app/peak_symm.h>
#ifdef BRUKER
#include <specio/bruker.h>
#else
#include <specio/easy.h>
#endif

typedef struct {
  float minLevel;
  int minSizeA[2];
  float maxSplitA[2];
  float splitDiff;
  float maxErrFact;
} ElimData;

typedef struct {
  float minLevel;
  float minMatch;
} DeconvData;

static void
elimReg(AppRegionP regP, void *clientData)
{
  ElimData *elimP = clientData;

  AppRegionPushState(regP, APP_PUSH_DATA | APP_PUSH_MAX_LIST);
  AppElimAll(regP, elimP->minLevel, elimP->minSizeA,
      elimP->maxSplitA, elimP->splitDiff, elimP->maxErrFact);
  AppRegionPopState(regP);
}

static void
deconvReg(AppRegionP regP, void *clientData)
{
  DeconvData *deconvP = clientData;

  AppDeconvol(regP, deconvP->minLevel, deconvP->minMatch);
}

static void
reportPeak(AppPeakP peakP, void *clientData)
{
  int dom;

  for (dom = 0; dom < peakP->dimensionNo; dom++)
    (void) printf("%7.2f ", peakP->positionA[dom]);
  (void) printf(": %5.3f\n", peakP->quality);
}

static void
peakPick(AppSpectrumP specP, int xMin, int xMax, int yMin, int yMax,
    AppNoiseP noiseP, float noiseLevel, int xMinSize, int yMinSize,
    float minElimLevel, float xMaxSplit, float yMaxSplit, float splitDiff,
    float maxErrFact,
    float minShapeDiff, float maxShapeDiff,
    float minDeconvLevel, float minMatch,
    float symmDist)
{
  int dim, sizeA[2], minSizeA[2];
  BOOL exclBorder, exclDiag;
  AppDataRange rangeA[2];
  AppMaskP maskP, exclP;
  float noiseFact;
  ElimData elim;
  DeconvData deconv;

  AppGetSpectrumInfo(specP, &dim, sizeA);
  if (dim != 2)
    return;

  exclBorder = TRUE;
  exclDiag = TRUE;

  minSizeA[0] = xMinSize;
  minSizeA[1] = yMinSize;

  rangeA[0][0] = xMin;
  rangeA[0][1] = xMax;
  rangeA[1][0] = yMin;
  rangeA[1][1] = yMax;

  maskP = AppMaskPrepare(specP);
  AppMaskBox(maskP, rangeA);

  noiseFact = 1.0;

  for (;;) {
    exclP = AppMaskPrepare(specP);

    if (exclBorder)
      AppMaskBorder(exclP, maskP, noiseP, noiseFact * noiseLevel, 1);
    if (exclDiag)
      AppMaskDiagonal(exclP, maskP, noiseP, noiseFact * noiseLevel,
	  "ppm", 0, 1);

    (void) AppSegment(specP, maskP, exclP,
        noiseP, noiseFact * noiseLevel, noiseFact * minDeconvLevel, minSizeA);

    AppMaskDestroy(maskP);
    maskP = exclP;

    if (AppMaskIsEmpty(maskP))
      break;

    noiseFact *= 1.4;
  }

  AppMaskDestroy(maskP);

  elim.minLevel = minElimLevel;
  elim.minSizeA[0] = xMinSize;
  elim.minSizeA[1] = yMinSize;
  elim.maxSplitA[0] = xMaxSplit;
  elim.maxSplitA[1] = yMaxSplit;
  elim.splitDiff = splitDiff;
  elim.maxErrFact = maxErrFact;

  AppRegionApply(specP, elimReg, &elim);

  AppClusterShapes(specP, 0, 1, minShapeDiff, maxShapeDiff);
  AppClusterShapes(specP, 1, 1, minShapeDiff, maxShapeDiff);

  deconv.minLevel = minDeconvLevel;
  deconv.minMatch = minMatch;

  AppRegionApply(specP, deconvReg, &deconv);

  if (symmDist > 0.0) {
    AppPeakSetSymmetric(specP, "ppm", 0, 1);
    AppPeakSymmetrize(specP, "ppm", 0, 1, symmDist, noiseP, noiseLevel);
  }

  AppPeakApply(specP, reportPeak, NULL);
}

int
main(int argc, char *argv[])
{
  AppSpectrumP specP;
  AppDataRange rangeA[2];
  AppNoiseP noiseP;
  float avg, magn;
  int dim, sizeA[2];

#ifdef BRUKER
  specP = SpecIOBrukerOpen("/u/data/shit/nmr/shit/1/shit/1/2rr");
#else
  specP = SpecIOEasyOpen("/home/troll/kor/noesy/ER2-NH.3D.param");
#endif
  if (specP == NULL) {
    (void) fprintf(stderr, "could not open spectrum\n");
    return 1;
  }

  AppGetSpectrumInfo(specP, &dim, sizeA);
  rangeA[0][0] = 0;
  rangeA[0][1] = sizeA[0] - 1;
  rangeA[1][0] = 0;
  rangeA[1][1] = sizeA[1] - 1;
  noiseP = AppNoisePrepare(specP, rangeA);

  peakPick(specP, 700, 895, 695, 800, noiseP, 1.0, 6, 3,
      2.5, 5.0, 1.0, 0.02, 0.5,
      0.2, 1.0,
      1.5, 0.5,
      0.01);

  AppNoiseDestroy(noiseP);

#ifdef BRUKER
  SpecIOBrukerClose(specP);
#else
  SpecIOEasyClose(specP);
#endif

  return 0;
}
