/*
************************************************************************
*
*   Easy.c - I/O of spectra in EASY format
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/09/13
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/specio/SCCS/s.Easy.c
*   SCCS identification       : 1.2
*
************************************************************************
*/

#include <specio/easy.h>

#include <stdio.h>
#include <string.h>

#include <dstr.h>
#include <file_name.h>
#include <linlist.h>
#include <ndio/ndio.h>

typedef struct {
  NDIOHANDLE io;
  int dim;
  AppSpectrumP specP;
} SpecDescr;

static LINLIST SpecList = NULL;

static void
readEasy(void *clientData, AppDataRange rangeA[], AppDataP dataA)
{
  SpecDescr *descrP = clientData;
  NDIOrange ndioRangeA[APP_MAX_DIM];
  float *dataBuf = dataA, *dataP;
  int size, i;

  size = 1;
  for (i = 0; i < descrP->dim; i++) {
    ndioRangeA[i][0] = rangeA[i][0];
    ndioRangeA[i][1] = rangeA[i][1];
    size *= rangeA[i][1] - rangeA[i][0] + 1;
  }

  dataP = NdioGet(descrP->io, ndioRangeA, NdioREAD);
  for (i = 0; i < size; i++)
    dataBuf[i] = dataP[i];
  NdioPut(descrP->io, dataP, NdioREAD);
}

AppSpectrumP
SpecIOEasyOpen(char *parFileName)
{
  SpecDescr descr, *descrP;
  FILE *fp;
  char parName[50];
  float parVal;
  NdioFileType type;
  int sizes[APP_MAX_DIM], xdims[APP_MAX_DIM];
  DSTR fileName, baseName;
  int sizeA[APP_MAX_DIM];
  float offsA[APP_MAX_DIM], swA[APP_MAX_DIM];
  int ret, dom, i;

  if (SpecList == NULL) {
    SpecList = ListOpen(sizeof(SpecDescr));
    NdioInit();
  }
  
  fp = fopen(parFileName, "r");
  if (fp == NULL)
    return NULL;

  descr.dim = 0;
  type = NdioEASY16;
  while (! feof(fp)) {
    ret = fscanf(fp, "%[^.]%*[.] %f", parName, &parVal);
    (void) fscanf(fp, "%*[^\n]");
    (void) fscanf(fp, "\n");
    if (ret != 2)
      continue;

    if (strncmp(parName, "Number of dimensions", 20) == 0) {
      descr.dim = (int) parVal;
    } else if (strncmp(parName, "16 or 8 bit file type", 21) == 0) {
      if (parVal == 16.0)
        type = NdioEASY16;
      else
        type = NdioEASY8;
    } else if (strncmp(parName, "Size of spectrum in w", 21) == 0) {
      dom = parName[21] - '0';
      if (dom <= descr.dim)
        sizes[descr.dim - dom] = (int) parVal;
    } else if (strncmp(parName, "Submatrix size in w", 19) == 0) {
      dom = parName[19] - '0';
      if (dom <= descr.dim)
        xdims[descr.dim - dom] = (int) parVal;
    } else if (strncmp(parName, "Maximum chemical shift in w", 27) == 0) {
      dom = parName[27] - '0';
      if (dom <= descr.dim)
        offsA[descr.dim - dom] = parVal;
    } else if (strncmp(parName, "Spectral sweep width in w", 25) == 0) {
      dom = parName[25] - '0';
      if (dom <= descr.dim)
        swA[descr.dim - dom] = parVal;
    }
  }

  (void) fclose(fp);

  if (descr.dim == 0)
    return NULL;

  for (dom = 0; dom < descr.dim; dom++)
    offsA[dom] -= 0.5 * swA[dom] / sizes[dom];

  fileName = DStrNew();
  DStrAssignStr(fileName, parFileName);
  FileNamePath(fileName);

  baseName = DStrNew();
  DStrAssignStr(baseName, parFileName);
  FileNameBase(baseName);

  DStrAppDStr(fileName, baseName);
  DStrFree(baseName);
  if (type == NdioEASY16)
    DStrAppStr(fileName, ".16");
  else
    DStrAppStr(fileName, ".8");

  descr.io = NdioRawOpen(DStrToStr(fileName), type, descr.dim,
      sizes, xdims, NdioREAD, TRUE);
  DStrFree(fileName);
  if (descr.io == NULL)
    return NULL;

  for (i = 0; i < descr.dim; i++)
    sizeA[i] = sizes[i];

  descrP = ListInsertLast(SpecList, &descr);

  descrP->specP = AppOpenSpectrum(descr.dim, sizeA, offsA, swA, ADT_FLT32,
      readEasy, descrP);

  return descrP->specP;
}

void
SpecIOEasyGetCompressed(AppSpectrumP specP,
    AppDataRange rangeA[], int resolA[], AppDataP dataA)
{
  SpecDescr *descrP;
  NDIOrange ndioRangeA[APP_MAX_DIM];
  int ndioResolA[APP_MAX_DIM];
  int i;

  descrP = ListFirst(SpecList);
  while (descrP->specP != specP)
    descrP = ListNext(SpecList, descrP);

  for (i = 0; i < descrP->dim; i++) {
    ndioRangeA[i][0] = rangeA[i][0];
    ndioRangeA[i][1] = rangeA[i][1];
    ndioResolA[i] = resolA[i];
  }

  (void) NdioGetCompressed(descrP->io, ndioRangeA, dataA, ndioResolA);
}

void
SpecIOEasyClose(AppSpectrumP specP)
{
  SpecDescr *descrP;

  AppCloseSpectrum(specP);

  descrP = ListFirst(SpecList);
  while (descrP != NULL) {
    if (descrP->specP == specP) {
      NdioClose(descrP->io);
      ListRemove(SpecList, descrP);
      break;
    }
    descrP = ListNext(SpecList, descrP);
  }
}
