/*
************************************************************************
*
*   Bruker.c - I/O of spectra in BRUKER format
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/15
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/specio/SCCS/s.Bruker.c
*   SCCS identification       : 1.3
*
************************************************************************
*/

#include <specio/bruker.h>

#include <stdio.h>
#include <string.h>

#include <dstr.h>
#include <file_name.h>
#include <linlist.h>
#include <ndio/ndio.h>

typedef struct {
  NDIOHANDLE io;
  int dim;
  AppSpectrumP specP;
} SpecDescr;

static LINLIST SpecList = NULL;

static void
readBruker(void *clientData, AppDataRange rangeA[], AppDataP dataA)
{
  SpecDescr *descrP = clientData;
  NDIOrange ndioRangeA[APP_MAX_DIM];
  float *dataBuf = dataA, *dataP;
  int size, i;

  size = 1;
  for (i = 0; i < descrP->dim; i++) {
    ndioRangeA[i][0] = rangeA[i][0];
    ndioRangeA[i][1] = rangeA[i][1];
    size *= rangeA[i][1] - rangeA[i][0] + 1;
  }

  dataP = NdioGet(descrP->io, ndioRangeA, NdioREAD);
  for (i = 0; i < size; i++)
    dataBuf[i] = dataP[i];
  NdioPut(descrP->io, dataP, NdioREAD);
}

AppSpectrumP
SpecIOBrukerOpen(char *fileName)
{
  SpecDescr descr, *descrP;
  DSTR pathName, parFileName;
  FILE *fp;
  char parName[50];
  int ret;
  float parVal;
  int sizes[APP_MAX_DIM], xdims[APP_MAX_DIM];
  BOOL bigEndian;
  int sizeA[APP_MAX_DIM];
  float sfA[APP_MAX_DIM], offsA[APP_MAX_DIM], swA[APP_MAX_DIM];
  int i;

  if (SpecList == NULL) {
    SpecList = ListOpen(sizeof(SpecDescr));
    NdioInit();
  }
  
  pathName = DStrNew();
  DStrAssignStr(pathName, fileName);
  FileNamePath(pathName);

  parFileName = DStrNew();

  descr.dim = 0;
  for (i = 0; i < APP_MAX_DIM; i++) {
    DStrAssignDStr(parFileName, pathName);
    if (i == 0) {
      DStrAppStr(parFileName, "procs");
    } else {
      DStrAppStr(parFileName, "proc");
      DStrAppChar(parFileName, '1' + i);
      DStrAppChar(parFileName, 's');
    }

    fp = fopen(DStrToStr(parFileName), "r");
    if (fp == NULL)
      break;

    descr.dim++;

    sizes[i] = 0;
    xdims[i] = 0;

    while (! feof(fp)) {
      ret = fscanf(fp, "##$%[0-9A-Za-z_]=%f", parName, &parVal);
      (void) fscanf(fp, "%*[^\n]");
      (void) fscanf(fp, "\n");

      if (ret != 2)
        continue;

      if (i == 0 && strcmp(parName, "BYTORDP") == 0)
        bigEndian = (int) parVal;
      else if (strcmp(parName, "SI") == 0)
        sizes[i] = (int) parVal;
      else if (strcmp(parName, "XDIM") == 0)
        xdims[i] = (int) parVal;
      else if (strcmp(parName, "SF") == 0)
        sfA[i] = parVal;
      else if (strcmp(parName, "OFFSET") == 0)
        offsA[i] = parVal;
      else if (strcmp(parName, "SW") == 0)
        swA[i] = parVal;
      else if (strcmp(parName, "SW_p") == 0)
        swA[i] = parVal / sfA[i];
    }

    (void) fclose (fp);
  }

  DStrFree(pathName);
  DStrFree(parFileName);

  if (descr.dim == 0)
    return NULL;

  for (i = 0; i < descr.dim; i++)
    if (sizes[i] <= 0 || xdims[i] <= 0)
      return NULL;

  descr.io = NdioRawOpen(fileName, NdioINT, descr.dim,
      sizes, xdims, NdioREAD, bigEndian);
  if (descr.io == NULL)
    return NULL;

  for (i = 0; i < descr.dim; i++)
    sizeA[i] = sizes[i];

  descrP = ListInsertLast(SpecList, &descr);

  descrP->specP = AppOpenSpectrum(descr.dim, sizeA, offsA, swA, ADT_FLT32,
      readBruker, descrP);

  return descrP->specP;
}

void
SpecIOBrukerGetCompressed(AppSpectrumP specP,
    AppDataRange rangeA[], int resolA[], AppDataP dataA)
{
  SpecDescr *descrP;
  NDIOrange ndioRangeA[APP_MAX_DIM];
  int ndioResolA[APP_MAX_DIM];
  int i;

  descrP = ListFirst(SpecList);
  while (descrP->specP != specP)
    descrP = ListNext(SpecList, descrP);

  for (i = 0; i < descrP->dim; i++) {
    ndioRangeA[i][0] = rangeA[i][0];
    ndioRangeA[i][1] = rangeA[i][1];
    ndioResolA[i] = resolA[i];
  }

  (void) NdioGetCompressed(descrP->io, ndioRangeA, dataA, ndioResolA);
}

void
SpecIOBrukerClose(AppSpectrumP specP)
{
  SpecDescr *descrP;

  AppCloseSpectrum(specP);

  descrP = ListFirst(SpecList);
  while (descrP != NULL) {
    if (descrP->specP == specP) {
      NdioClose(descrP->io);
      ListRemove(SpecList, descrP);
      break;
    }
    descrP = ListNext(SpecList, descrP);
  }
}
