/*
************************************************************************
*
*   Ndio.c - N dimensional data i/o module
*
*   Copyright (c) 1991-96
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification   : 96/10/15
*   Pathname of SCCS file       : /sgiext/autopsy/app/src/ndio/SCCS/s.Ndio.c
*   SCCS identification         : 1.2
*
*   Modifications :
*   <NAME>  <DATE>  <COMMENTS>
*
************************************************************************
*/

#include <ndio/ndio.h>

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include <ndio/ndio_types.h>
#include <ndio/ndio_error.h>
#include "byte_swap.h"
#include "block_tab.h"

#undef ABS
#define ABS(x) ((x) > 0.0f ? (x) : (-(x)))

#define HEADER_SIZE 512

#define H_POS_TOTAL_SIZE 2
#define H_POS_TOTAL_SMX_SIZE 3
#define H_POS_DIM 4
#define H_POS_SIZES 5

typedef struct {
  CARD16  par;                          /* NDIO_PAR_* */
  CARD16  len;                          /* length in 4-byte words */
} NdioParDescr;

/***********************************************************************
        TYPES
***********************************************************************/

typedef struct {
  NdioFileMode mode;
  NDIOrange *coordV;
  NdioData data[1];                     /* malloced more as needed */
} DATA_HDR;


struct NdioStruct {
  char   *fileName;                     /* name of file */
  NdioFileMode mode;
  BOOL    byteSwap;                     /* byte swapping necessary */
  int     dim;                          /* number of dimensions */
  int    *fileSizes;                    /* size of file */
  int    *smxSizes;                     /* smx sizes */
  int     headerSize;                   /* size of header (in bytes) */
  CARD32 *header;                       /* contents of header */
  BLOCKTABHANDLE blockTab;              /* Pointer to BlockTabStruct */

  int    *subSizes;              /* holds intermediate smx sub sizes */
  int    *smxNumbers;            /* holds number of smx */
  int    *datSizes;              /* holds intermediate data sub sizes */
  int    *bStart;
  int    *bEnd;
  int    *bCurr;
  int    *dStart;
  int    *dNo;
  int    *dCount;
  int    *sPtrDec;
  int    *dPtrDec;
  int    *comprSizes;
  int    *comprOrgSize;
  int    *comprResSize;
  int    *comprStartEps;
  int    *comprEps;
};



/***********************************************************************
	STATICS
***********************************************************************/

/*----------------------------------------------------------------------
calculate next curr[] n-tuple withing loop[] range, for example:

loop[0] = { 0, 3 } =>  0 1 2 3 0 1 2 3 0 1 2 3 0 1 ...
loop[1] = { 3, 3 } =>  3 3 3 3 3 3 3 3 3 3 3 3 3 3 ...
loop[2] = { 0, 2 } =>  0 0 0 0 1 1 1 1 2 2 2 2 0 0 ...
loop[3] = { 0, 6 } =>  0 0 0 0 0 0 0 0 0 0 0 0 1 1 ...
----------------------------------------------------------------------*/


/**********************************************************************/
static void
getSubSizes(NDIOHANDLE io)
/*----------------------------------------------------------------------
prepare sub dimension sizes for line, plane, cube etc. of an smx block

This routine calculates the following constant products

  zdim*ydim*xdim , ydim*xdim , xdim , 1

for fast n - dimension array access e.g.

smx[u][z][y][x] == *(smx + u*zdim*ydim*xdim +z*ydim*xdim + y*xdim + x)
----------------------------------------------------------------------*/
{
  int     i;

  io->subSizes[0] = 1;

  for (i = 1; i < io->dim; i++)
    io->subSizes[i] = io->subSizes[i - 1] * io->smxSizes[i - 1];
}


/**********************************************************************/
static void
initBlockLoopRange(
    NDIOHANDLE io,
    NDIOrange coordV[]   /* specify x,y,.. dimension to read */
)
/*----------------------------------------------------------------------
Initialize the block loop arrays (bStart, bEnd, bCurr) with the range
for each dimension, to be able to index into the data file to retrieve a
submatrix block.
----------------------------------------------------------------------*/
{
  int     i;

  for (i = 0; i < io->dim; i++) {
    io->bStart[i] = coordV[i][0] / io->smxSizes[i];
    io->bEnd[i] = coordV[i][1] / io->smxSizes[i];
    io->bCurr[i] = io->bStart[i];
  }
}


/**********************************************************************/
static int
initDataLoopRange(
    NDIOHANDLE io,
    NDIOrange coordV[]   /* specify (x,y,..) dimension to read */
)
/*----------------------------------------------------------------------
Initialize the data loop arrays (dStart, dNo) with the range for
each dimension to be able to index into a submatrix block to retrieve
a data point.
Also initialize the dCount vector with the start values.
Returns the start offset for the data.
----------------------------------------------------------------------*/
{
  int     i;
  int     startCoord;
  int     dataPos = 0;

  for (i = 0; i < io->dim; i++) {
    startCoord = io->bCurr[i] * io->smxSizes[i];
    if (coordV[i][0] > startCoord)
      io->dStart[i] = coordV[i][0] - startCoord;
    else
      io->dStart[i] = 0;
    if (coordV[i][1] < startCoord + io->smxSizes[i] - 1)
      io->dNo[i] = coordV[i][1] - startCoord - io->dStart[i];
    else
      io->dNo[i] = io->smxSizes[i] - 1 - io->dStart[i];
    io->dCount[i] = 0;
    if (startCoord > coordV[i][0])
      dataPos += (startCoord - coordV[i][0]) * io->datSizes[i];
  }

  return dataPos;
}


/**********************************************************************/
static int
getTotalAndDatSizes(NDIOHANDLE io, NDIOrange coordV[])
/*----------------------------------------------------------------------
calculate total and subdimension sizes in advance for faster access
in innermost loop. Put the sub sizes of the output data block into
io->datSizes.
----------------------------------------------------------------------*/
{
  int     i, dim;

  dim = io->dim;

  io->datSizes[0] = 1;

  for (i = 0; i < dim - 1; i++)
    io->datSizes[i + 1] =
        (coordV[i][1] - coordV[i][0] + 1) * io->datSizes[i];

  return io->datSizes[dim - 1] * (coordV[dim - 1][1] - coordV[dim - 1][0] + 1);
}


/**********************************************************************/
static int
getBlockPos(NDIOHANDLE io)
/*----------------------------------------------------------------------
Returns the position of the current block in the file.
----------------------------------------------------------------------*/
{
  int     i;
  int     pos = 0, prod = 1;

  for (i = 0; i < io->dim; i++) {
    pos += io->bCurr[i] * prod;
    prod *= io->smxNumbers[i];
  }

  return pos;
}


/***********************************************************************
        GLOBAL FUNCTIONS
***********************************************************************/


/**********************************************************************/
void
NdioSetMemLimit(
    int memLimit                 /* specify memory limit */
)
/*----------------------------------------------------------------------
Set size of buffer for I/O.
----------------------------------------------------------------------*/
{
  BlockSetMemLimit(memLimit);
}


/**********************************************************************/
void
NdioInit(void)
/*----------------------------------------------------------------------
Initialize Ndio module.
----------------------------------------------------------------------*/
{
  BlockInit();
}


/**********************************************************************/
static int
calcSize(
    int dim,                  /* specify number of elements in vector */
    int sizes[]               /* specify vector with sizes */
)
{
  int     size = 1;
  int     i;

  for (i = 0; i < dim; i++)
    size *= sizes[i];

  return size;
}


/**********************************************************************/
static  NDIOHANDLE
ndioOpen(
    char *fileName,      /* Pointer file name of data file */
    NdioFileType type,   /* type of data in file */
    int dim,             /* dimension of file */
    int fileSizes[],     /* size of file (number of data points) */
    int smxSizes[],      /* size of submatrix (number of data points) */
    NdioFileMode mode,                  /* read, write or read_write */
    int headerSize,                     /* size of header (in bytes) */
    CARD32 * header,                    /* pointer to header or NULL */
    BOOL byteSwap                       /* byte swapping necessary */
)
/*----------------------------------------------------------------------
Open the file. fileSizes, smxSizes and header must be dynamically
allocated (header can also be NULL for raw files).
If the open fails, all memory (including fileSizes, smxSizes and header)
is freed and the function returns NULL.
----------------------------------------------------------------------*/
{
  int     i;
  int     fileSize, blockSize;
  BLOCKTABHANDLE blockTab;              /* Pointer to BlockTabStruct */
  NDIOHANDLE io;

  fileSize = calcSize(dim, fileSizes);
  blockSize = calcSize(dim, smxSizes);

  blockTab = BlockOpen(fileName, type, headerSize, fileSize,
      blockSize, mode, byteSwap);

  if (blockTab == NULL) {
    free(fileSizes);
    free(smxSizes);
    if (header != NULL)
      free(header);
    return NULL;                 /* error message issued by BlockOpen */
  }
  io = malloc(sizeof(struct NdioStruct));

  io->blockTab = blockTab;
  io->dim = dim;
  io->fileSizes = fileSizes;
  io->smxSizes = smxSizes;
  io->header = header;
  io->headerSize = headerSize;
  io->mode = mode;
  io->byteSwap = byteSwap;
  io->subSizes = malloc(dim * sizeof(int));
  io->smxNumbers = malloc(dim * sizeof(int));
  io->datSizes = malloc(dim * sizeof(int));
  io->bStart = malloc(dim * sizeof(int));
  io->bEnd = malloc(dim * sizeof(int));
  io->bCurr = malloc(dim * sizeof(int));
  io->dStart = malloc(dim * sizeof(int));
  io->dNo = malloc(dim * sizeof(int));
  io->dCount = malloc(dim * sizeof(int));
  io->sPtrDec = malloc(dim * sizeof(int));
  io->dPtrDec = malloc(dim * sizeof(int));
  io->comprSizes = malloc(dim * sizeof(int));
  io->comprOrgSize = malloc(dim * sizeof(int));
  io->comprResSize = malloc(dim * sizeof(int));
  io->comprStartEps = malloc(dim * sizeof(int));
  io->comprEps = malloc(dim * sizeof(int));

  io->fileName = malloc(strlen(fileName) + 1);
  (void) strcpy(io->fileName, fileName);

  /* calculate intermediate sizes for fast smx access */
  getSubSizes(io);

  for (i = 0; i < dim; i++) {
    io->smxNumbers[i] = fileSizes[i] / smxSizes[i];
    if (io->smxNumbers[i] * smxSizes[i] < fileSizes[i]) {
      /* Incomplete submatrices are sometimes used in EASY files,
	 the files contain the full submatrices, filled with zeros. */
      io->smxNumbers[i]++;
      fileSizes[i] = io->smxNumbers[i] * smxSizes[i];
    }
  }

  return io;
}


/**********************************************************************/
static int *
copySizes(int dim, int sizes[])
{
  int    *sizesCopy;
  int     i;

  sizesCopy = malloc(dim * sizeof(int));
  for (i = 0; i < dim; i++)
    sizesCopy[i] = sizes[i];

  return sizesCopy;
}


/**********************************************************************/
static  BOOL
isBigEndian(void)
{
#ifdef BIG_ENDIAN
  return TRUE;
#else
  return FALSE;
#endif
}


/**********************************************************************/
static  BOOL
createEmptyFile(char *fileName, int size)
{
  int     fd;
  char    ch;

  (void) unlink(fileName);
  fd = open(fileName, O_WRONLY | O_CREAT, 0777);
  if (fd < 0) {
    NdioError("creation of file %s failed: %s",
        fileName, NdioErrorGetSys());
    return FALSE;
  }
  /* force new file to correct length */
  (void) lseek(fd, (off_t) size - 1, 0);
  (void) write(fd, &ch, 1);

  (void) close(fd);

  return TRUE;
}


/**********************************************************************/
NDIOHANDLE
NdioRawOpen(
    char *fileName,      /* file name of data file */
    NdioFileType type,   /* type of data in file */
    int dim,             /* dimension of file */
    int fileSizes[],     /* size of file (number of data points) */
    int smxSizes[],      /* size of submatrix (number of data points) */
    NdioFileMode mode,                  /* read, write or read_write */
    BOOL bigEndian                      /* TRUE if file is big-endian */
)
/*----------------------------------------------------------------------
Open an Ndio file without header.
----------------------------------------------------------------------*/
{
  return ndioOpen(fileName, type, dim,
      copySizes(dim, fileSizes), copySizes(dim, smxSizes),
      mode, 0, (CARD32 *) NULL, (BOOL) (isBigEndian() != bigEndian));
}


/**********************************************************************/
NDIOHANDLE
NdioRawCreate(
    char *fileName,      /* file name of data file */
    int dim,             /* dimension of file */
    int fileSizes[],     /* size of file (number of data points) */
    int smxSizes[],      /* size of submatrix (number of data points) */
    NdioFileMode mode,                  /* read, write or read_write */
    BOOL bigEndian                      /* TRUE if file is big-endian */
)
/*----------------------------------------------------------------------
Create an Ndio file without header and small-endian byte order.
----------------------------------------------------------------------*/
{
  if (mode == NdioREAD) {
    NdioError("attempt to create file %s with read-only mode",
        fileName);
    return NULL;
  }
  if (!createEmptyFile(fileName,
          DATA_SIZE * calcSize(dim, fileSizes)))
    return NULL;
  return NdioRawOpen(fileName, NdioFLOAT, dim,
      fileSizes, smxSizes, mode, bigEndian);
}


/**********************************************************************/
NDIOHANDLE
NdioCreate(
    char *fileName,      /* file name of data file */
    int dim,             /* dimension of file */
    int fileSizes[],     /* size of file (number of data points) */
    int smxSizes[],      /* size of submatrix (number of data points) */
    NdioFileMode mode    /* read, write or read_write */
)
/*----------------------------------------------------------------------
Create an Ndio file with header.
----------------------------------------------------------------------*/
{
  int     totalSize;
  CARD32 *header;
  int     i;

  if (mode == NdioREAD) {
    NdioError("attempt to create file %s with read-only mode",
        fileName);
    return NULL;
  }
  totalSize = calcSize(dim, fileSizes);
  if (!createEmptyFile(fileName,
          HEADER_SIZE + DATA_SIZE * totalSize))
    return NULL;

  header = malloc(HEADER_SIZE);

  header[0] = NDIO_MAGIC;
  header[1] = HEADER_SIZE;
  header[H_POS_TOTAL_SIZE] = totalSize;
  header[H_POS_TOTAL_SMX_SIZE] = calcSize(dim, smxSizes);
  header[H_POS_DIM] = dim;

  for (i = 0; i < dim; i++) {
    header[H_POS_SIZES + i] = fileSizes[i];
    header[H_POS_SIZES + dim + i] = smxSizes[i];
  }

  return ndioOpen(fileName, NdioFLOAT, dim,
      copySizes(dim, fileSizes), copySizes(dim, smxSizes),
      mode, HEADER_SIZE, header, FALSE);
}


/**********************************************************************/
NDIOHANDLE
NdioOpen(
    char *fileName,      /* file name of data file */
    NdioFileMode mode    /* read, write or read_write */
)
/*----------------------------------------------------------------------
Open an Ndio file with header.
----------------------------------------------------------------------*/
{
  int     fd;
  int     ret;
  CARD32  magicAndSize[2], *header;
  int     headerSize;
  BOOL    byteSwap;
  int     dim, i;
  int    *fileSizes, *smxSizes;

  fd = open(fileName, O_RDONLY);
  if (fd < 0) {
    NdioError("can't open file %s: %s", fileName, NdioErrorGetSys());
    return NULL;
  }
  /* read magic number and size of header */
  ret = read(fd, (char *) magicAndSize, sizeof(magicAndSize));
  if (ret != sizeof(magicAndSize)) {
    NdioError("can't read from file %s: %s",
        fileName, NdioErrorGetSys());
    (void) close(fd);
    return NULL;
  }
  if (magicAndSize[0] == NDIO_MAGIC) {
    /* ok, no byte swapping necessary */
    byteSwap = FALSE;
  } else {
    /* try with byte swapping */
    Reverse4ByteOrder(2, (char *) magicAndSize);
    if (magicAndSize[0] == NDIO_MAGIC) {
      /* ok, need byte swapping */
      byteSwap = TRUE;
    } else {
      NdioError("file %s has wrong magic number", fileName);
      (void) close(fd);
      return NULL;
    }
  }

  headerSize = magicAndSize[1];
  header = malloc(headerSize);
  (void) lseek(fd, (off_t) 0, 0);
  ret = read(fd, (char *) header, headerSize);
  (void) close(fd);
  if (ret != headerSize) {
    NdioError("can't read header from file %s: %s",
        fileName, NdioErrorGetSys());
    return NULL;
  }
  dim = header[H_POS_DIM];
  fileSizes = malloc(dim * sizeof(int));
  smxSizes = malloc(dim * sizeof(int));
  for (i = 0; i < dim; i++) {
    fileSizes[i] = header[H_POS_SIZES + i];
    smxSizes[i] = header[H_POS_SIZES + dim + i];
  }

  return ndioOpen(fileName, NdioFLOAT, dim,
      fileSizes, smxSizes, mode, headerSize, header, byteSwap);
}


/**********************************************************************/
NdioDataPtr
NdioGet(
    NDIOHANDLE io,
    NDIOrange coordV[],  /* specify coordinate ranges (x,y,z...) to read
                          * */
    NdioFileMode mode
)
/*----------------------------------------------------------------------
Read a part of the (N-dimensional) data set specified by io.

WARNING:
!! The caller may free the returned data ptr only using NdioPut() !!
----------------------------------------------------------------------*/
{
  DATA_HDR *dhdr;
  int     dim = io->dim;
  int     i, pos;
  BlockPtr blockP;       /* pointer to block */
  int     dataSize;
  int     dStartOff;
  NdioDataPtr sPtr, dPtr;
  int     blockPos;

  /* get total and intermediate sizes for smx and data */
  dataSize = getTotalAndDatSizes(io, coordV);

  if ((dhdr = malloc(sizeof(DATA_HDR) + (dataSize - 1) * DATA_SIZE)) == NULL)
    return NULL;

  dhdr->coordV = malloc(dim * 2 * sizeof(int));
  if (dhdr->coordV == NULL) {
    NdioError("memory allocation failed in NdioGet");
    return NULL;
  }
  for (i = 0; i < dim; i++) {
    dhdr->coordV[i][0] = coordV[i][0];
    dhdr->coordV[i][1] = coordV[i][1];
  }
  dhdr->mode = mode;

  if (mode == NdioWRITE)
    return dhdr->data;

  /* prepare array of block-index, describing the start and end index
   * of each dimension to read (1=x, 2=y, 3=z, ...) */
  initBlockLoopRange(io, coordV);

  for (;;) {             /* for all blocks */

    /* read one submatrix block */
    blockPos = getBlockPos(io);
    blockP = BlockGet(io->blockTab, blockPos, mode);

    dStartOff = initDataLoopRange(io, coordV);

    sPtr = blockP;
    for (i = 0; i < dim; i++) {
      sPtr += io->dStart[i] * io->subSizes[i];
      io->sPtrDec[i] = io->dNo[i] * io->subSizes[i];
      io->dPtrDec[i] = io->dNo[i] * io->datSizes[i];
    }

    dPtr = dhdr->data + dStartOff;

    for (;;) {           /* for all datapoints in one block */

      *dPtr = *sPtr;

      pos = 0;
      while (pos < dim && io->dCount[pos] == io->dNo[pos]) {
        io->dCount[pos] = 0;
        sPtr -= io->sPtrDec[pos];
        dPtr -= io->dPtrDec[pos];
        pos++;
      }
      if (pos == dim)
        break;

      io->dCount[pos]++;
      sPtr += io->subSizes[pos];
      dPtr += io->datSizes[pos];

    }  /* for all datapoints */

    BlockPut(io->blockTab, blockPos, NdioREAD);

    /* calculate next bCurr[], give next block */
    pos = 0;
    while (pos < dim && io->bCurr[pos] == io->bEnd[pos]) {
      io->bCurr[pos] = io->bStart[pos];
      pos++;
    }
    if (pos == dim)
      break;

    io->bCurr[pos]++;

  }  /* for all blocks */

  return dhdr->data;
}


/**********************************************************************/
void
NdioPut(
    NDIOHANDLE io,
    NdioDataPtr data,          /* specify pointer returned by NdioGet */
    NdioFileMode mode
)
/*----------------------------------------------------------------------
Write back a sub-dimension from the (N-dimensional) data set specified
by io. Frees all associated memory ( also data ).
mode must be compatible with the mode used in NdioGet:
    NdioGet        | NdioPut
    ---------------+------------------------------------
    NdioREAD       | NdioREAD
    NdioWRITE      | NdioWRITE
    NdioREAD_WRITE | NdioREAD, NdioWRITE, NdioREAD_WRITE
----------------------------------------------------------------------*/
{
  int     i, pos;
  BlockPtr blockP;
  int     offset = offsetof(DATA_HDR, data[0]);
  DATA_HDR *dhdr = (DATA_HDR *) (((char *) data) - offset);
  int     dim = io->dim;
  int     dStartOff;
  NdioDataPtr sPtr, dPtr;
  int     blockPos;

#if DEBUG
  if (dhdr->mode != NdioREAD_WRITE && dhdr->mode != mode)
    (void) fprintf(stderr,
        "NdioGet() vs. NdioPut(): error: mode mismatch\n");
#endif

  if (mode == NdioREAD) {      /* read only */
    free(dhdr->coordV);
    free(dhdr);
    return;
  }
  /* prepare array of block-index, describing the start and and index
   * of each dimension to read (1=x, 2=y, 3=z, ....0) */
  initBlockLoopRange(io, dhdr->coordV);

  for (;;) {                   /* for all blocks */

    /* read one submatrix block */
    blockPos = getBlockPos(io);
    blockP = BlockGet(io->blockTab, blockPos, mode);

    dStartOff = initDataLoopRange(io, dhdr->coordV);

    sPtr = blockP;
    for (i = 0; i < dim; i++) {
      sPtr += io->dStart[i] * io->subSizes[i];
      io->sPtrDec[i] = io->dNo[i] * io->subSizes[i];
      io->dPtrDec[i] = io->dNo[i] * io->datSizes[i];
    }

    dPtr = dhdr->data + dStartOff;

    for (;;) {                 /* for all datapoints in one block */

      *sPtr = *dPtr;

      pos = 0;
      while (pos < dim && io->dCount[pos] == io->dNo[pos]) {
        io->dCount[pos] = 0;
        sPtr -= io->sPtrDec[pos];
        dPtr -= io->dPtrDec[pos];
        pos++;
      }
      if (pos == dim)
        break;

      io->dCount[pos]++;
      sPtr += io->subSizes[pos];
      dPtr += io->datSizes[pos];

    }  /* for all datapoints */

    BlockPut(io->blockTab, blockPos, mode);

    /* calculate next bCurr[], give next block */
    pos = 0;
    while (pos < dim && io->bCurr[pos] == io->bEnd[pos]) {
      io->bCurr[pos] = io->bStart[pos];
      pos++;
    }
    if (pos == dim)
      break;

    io->bCurr[pos]++;

  }  /* for all blocks */

  free(dhdr->coordV);
  free(dhdr);
}


/**********************************************************************/
NdioDataPtr
NdioGetCompressed(
    NDIOHANDLE io,
    NDIOrange coordV[],                     /* specify coordinate ranges
                                             * (x,y,z...) to read */
    NdioDataPtr dataP,                  /* pointer to destination */
    int maxV[]                          /* vector with maximal sizes */
)
/*----------------------------------------------------------------------
Read a part of the (N-dimensional) data set specified by io.
Compress if necessary.
----------------------------------------------------------------------*/
{
  int     dim = io->dim;
  int     i, pos;
  BlockPtr blockP;                      /* pointer to block */
  int     resSize;
  NdioDataPtr sPtr, dPtr;
  int     blockPos;
  int     startCoord, epsSum;

  /* get intermediate sizes for smx and data */
  (void) getTotalAndDatSizes(io, coordV);

  /* prepare array of block-index, describing the start and end index
   * of each dimension to read (1=x, 2=y, 3=z, ...) */
  initBlockLoopRange(io, coordV);

  io->comprSizes[0] = 1;
  for (i = 0; i < dim - 1; i++)
    io->comprSizes[i + 1] = maxV[i] * io->comprSizes[i];

  resSize = maxV[dim - 1] * io->comprSizes[dim - 1];

  for (i = 0; i < dim; i++) {
    io->comprOrgSize[i] = coordV[i][1] - coordV[i][0] + 1;
    if (maxV[i] > io->comprOrgSize[i])
      io->comprResSize[i] = io->comprOrgSize[i];
    else
      io->comprResSize[i] = maxV[i];
  }

  (void) memset((char *) dataP, 0, (int) (resSize * DATA_SIZE));

  for (;;) {                            /* for all blocks */

    /* read one submatrix block */
    blockPos = getBlockPos(io);
    blockP = BlockGet(io->blockTab, blockPos, NdioREAD);

    (void) initDataLoopRange(io, coordV);

    dPtr = dataP;
    sPtr = blockP;
    for (i = 0; i < dim; i++) {
      sPtr += io->dStart[i] * io->subSizes[i];
      io->sPtrDec[i] = io->dNo[i] * io->subSizes[i];
      startCoord = io->bCurr[i] * io->smxSizes[i];
      if (startCoord > coordV[i][0]) {
        epsSum = (1 + 2 * (startCoord - coordV[i][0])) * io->comprResSize[i];
        dPtr += epsSum / (2 * io->comprOrgSize[i]) * io->comprSizes[i];
        io->comprStartEps[i] = epsSum % (2 * io->comprOrgSize[i]);
      } else {
        io->comprStartEps[i] = io->comprResSize[i];
      }
      io->dPtrDec[i] = (io->comprStartEps[i] +
          2 * io->dNo[i] * io->comprResSize[i]) /
          (2 * io->comprOrgSize[i]) * io->comprSizes[i];
      io->comprEps[i] = io->comprStartEps[i];
    }

    for (;;) {                     /* for all datapoints in one block */

      if (ABS(*sPtr) > ABS(*dPtr))
        *dPtr = *sPtr;

      pos = 0;
      while (pos < dim && io->dCount[pos] == io->dNo[pos]) {
        io->dCount[pos] = 0;
        sPtr -= io->sPtrDec[pos];
        dPtr -= io->dPtrDec[pos];
        io->comprEps[pos] = io->comprStartEps[pos];
        pos++;
      }
      if (pos == dim)
        break;

      io->dCount[pos]++;
      sPtr += io->subSizes[pos];
      io->comprEps[pos] += 2 * io->comprResSize[pos];
      if (io->comprEps[pos] >= 2 * io->comprOrgSize[pos]) {
        io->comprEps[pos] -= 2 * io->comprOrgSize[pos];
        dPtr += io->comprSizes[pos];
      }
    }  /* for all datapoints */

    BlockPut(io->blockTab, blockPos, NdioREAD);

    /* calculate next bCurr[], give next block */
    pos = 0;
    while (pos < dim && io->bCurr[pos] == io->bEnd[pos]) {
      io->bCurr[pos] = io->bStart[pos];
      pos++;
    }
    if (pos == dim)
      break;

    io->bCurr[pos]++;

  }  /* for all blocks */

  return dataP;
}


/**********************************************************************/
void *
NdioGetPar(
    NDIOHANDLE io,
    void *buf,        /* pointer to variable of parameter type */
    int parId,        /* one of NDIO_PAR_* */
    int arrInd        /* specify array index, 0 for normal parameters */
)
/*----------------------------------------------------------------------
Get a paramer out of the header of an Ndio file.
Returns buf on success, NULL otherwise.
----------------------------------------------------------------------*/
{
  int     dim;

  if (io->header == NULL) {
    NdioError("file has not header for NdioGetPar");
    return NULL;
  }
  switch (parId) {
  case NDIO_PAR_BYTE_SWAP:
    *(INT32 *) buf = io->byteSwap;
    break;
  case NDIO_PAR_HEADER_SIZE:
    *(INT32 *) buf = io->header[1];
    break;
  case NDIO_PAR_DIM:
    *(INT32 *) buf = io->header[H_POS_DIM];
    break;
  case NDIO_PAR_SIZE:
    dim = io->header[H_POS_DIM];
    if (arrInd >= dim) {
      NdioError("array index out of bounds for SIZE in NdioGetPar");
      return NULL;
    }
    *(INT32 *) buf = io->header[H_POS_SIZES + arrInd];
    break;
  case NDIO_PAR_TOTAL_SIZE:
    *(INT32 *) buf = io->header[H_POS_TOTAL_SIZE];
    break;
  case NDIO_PAR_SMX_SIZE:
    dim = io->header[H_POS_DIM];
    if (arrInd >= dim) {
      NdioError("array index out of bounds for SMX_SIZE in NdioGetPar");
      return NULL;
    }
    *(INT32 *) buf = io->header[H_POS_SIZES + dim + arrInd];
    break;
  case NDIO_PAR_TOTAL_SMX_SIZE:
    *(INT32 *) buf = io->header[H_POS_TOTAL_SMX_SIZE];
    break;
  default:
    NdioError(
        "reading of optional parameters not implemented in NdioGetPar");
    return NULL;
  }

  return buf;
}


/**********************************************************************/
void *
NdioSetPar(
    NDIOHANDLE io,
    void *buf,        /* pointer to variable of parameter type */
    int parId,        /* one of NDIO_PAR_* */
    int arrInd        /* specify array index, 0 for normal parameters */
)
/*----------------------------------------------------------------------
Set a paramer in the header of an Ndio file.
Returns buf on success, NULL otherwise.
----------------------------------------------------------------------*/
{
  NdioError("NdioSetPar not implemented");
  return NULL;
}


/**********************************************************************/
void
NdioClose(NDIOHANDLE io)
/*----------------------------------------------------------------------
Close file, give storage free.
----------------------------------------------------------------------*/

{
  int     fd;

  BlockClose(io->blockTab);

  if (io->mode != NdioREAD && io->header != NULL) {
    fd = open(io->fileName, O_WRONLY);
    if (fd < 0) {
      NdioError("cannot open data file %s for writing header",
          io->fileName);
    } else {
      (void) write(fd, (char *) io->header, io->headerSize);
      (void) close(fd);
    }
  }
  free(io->header);
  free(io->smxSizes);
  free(io->fileName);
  free(io->fileSizes);
  free(io->subSizes);
  free(io->smxNumbers);
  free(io->datSizes);
  free(io->bStart);
  free(io->bEnd);
  free(io->bCurr);
  free(io->dStart);
  free(io->dNo);
  free(io->dCount);
  free(io->sPtrDec);
  free(io->dPtrDec);
  free(io->comprSizes);
  free(io->comprOrgSize);
  free(io->comprResSize);
  free(io->comprStartEps);
  free(io->comprEps);
  free(io);
}


/**********************************************************************/
void
NdioCleanup(void)
/*----------------------------------------------------------------------
----------------------------------------------------------------------*/
{
  BlockCleanup();
}
