/*
************************************************************************
*
*   Shape.c - handling of line shapes
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/13
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.Shape.c
*   SCCS identification       : 1.4
*
************************************************************************
*/

#include <app/shape.h>
#include "shape.h"

#include <stdlib.h>
#include <values.h>
#include <math.h>

#include <marquardt.h>
#include <app/base_types.h>

#define MIN_WIDTH 0.5

float
ShapeNorm(AppShapeP shapeP)
{
  float a;
  int i;

  a = - MAXFLOAT;
  for (i = 0; i < shapeP->valueNo; i++)
    if (shapeP->valueA[i] > a)
      a = shapeP->valueA[i];

  for (i = 0; i < shapeP->valueNo; i++) {
    shapeP->valueA[i] /= a;
    shapeP->errorA[i] /= a;
  }

  return a;
}

void
ShapeLimitError(AppShapeP shapeP)
{
  float maxFact;
  int i;

  /* this value makes sure that the error obtained when
     multiplying shapes is not larger than the value itself */
  maxFact = 0.9 / sqrt((float) shapeP->dimensionNo);

  for (i = 0; i < shapeP->valueNo; i++)
    if (shapeP->errorA[i] > maxFact * shapeP->valueA[i])
      shapeP->errorA[i] = maxFact * shapeP->valueA[i];
}

static void
estimatePar(AppShapeP shapeP)
{
  float max, sum, wSum, cent, width, w, x, d, err;
  int i;

  max = 0.0;
  for (i = 0; i < shapeP->valueNo; i++)
    if (shapeP->valueA[i] > max)
      max = shapeP->valueA[i];
  shapeP->amplitude = max;

  sum = 0.0;
  wSum = 0.0;
  for (i = 0; i < shapeP->valueNo; i++) {
    sum += i * shapeP->valueA[i];
    wSum += shapeP->valueA[i];
  }
  cent = sum / wSum;

  sum = 0.0;
  wSum = 0.0;
  for (i = 0; i < shapeP->valueNo; i++) {
    if (shapeP->valueA[i] >= 1.0 || shapeP->valueA[i] <= 0.0)
      continue;

    x = 2.0 * (i - cent);
    /* largest weight for values near 0.5 */
    if (shapeP->valueA[i] > 0.5)
      w = 2.0 * (1.0 - shapeP->valueA[i]);
    else
      w = 2.0 * shapeP->valueA[i];
    sum += w * sqrt(- M_LN2 * x * x / log(shapeP->valueA[i]));
    wSum += w;
  }
  if (wSum == 0.0) {
    width = MIN_WIDTH;
  } else {
    width = sum / wSum;
    if (width < MIN_WIDTH)
      width = MIN_WIDTH;
  }

  err = 0.0;
  for (i = 0; i < shapeP->valueNo; i++) {
    x = 2.0 * (i - cent) / width;
    d = shapeP->valueA[i] - exp(- M_LN2 * x * x);
    if (d > err)
      err = d;
    else if (- d > err)
      err = - d;
  }

  shapeP->centerPosition = shapeP->startIndex + cent;
  shapeP->centerError = 0.5 * width;
  shapeP->width = width;
  shapeP->lorentzFraction = 0.0;
  shapeP->gaussFraction = 1.0;
}

static void
fitLorGau(float *valA, int xI, float *sol, int paramNo, float *rP, float dr[])
{
  float x, a, f, w, t;
  float z, lor, gau, s;

  x = (float) xI;

  a = sol[0];
  f = sol[1];
  w = sol[2];
  t = sol[3];

  if (w == 0.0)
    w = 1.0e-4;

  z = 2.0 * (f - x) / w;
  lor = 1.0 / (1.0 + z * z);
  gau = exp(- M_LN2 * z * z);
  s = t * lor + (1.0 - t) * gau;

  dr[0] = s;
  dr[1] = - 4.0 * a * z * (t * lor * lor + M_LN2 * (1.0 - t) * gau) / w;
  dr[2] = 2.0 * a * z * z * (t * lor * lor + M_LN2 * (1.0 - t) * gau) / w;
  dr[3] = a * (lor - gau);

  *rP = a * s - valA[xI];
}

static void
fitLor(float *valA, int xI, float *sol, int paramNo, float *rP, float dr[])
{
  float x, a, f, w;
  float z, s;

  x = (float) xI;

  a = sol[0];
  f = sol[1];
  w = sol[2];

  if (w == 0.0)
    w = 1.0e-4;

  z = 2.0 * (f - x) / w;
  s = 1.0 / (1.0 + z * z);

  dr[0] = s;
  dr[1] = - 4.0 * a * z * s * s / w;
  dr[2] = 2.0 * a * z * z * s * s / w;

  *rP = a * s - valA[xI];
}

static void
fitGau(float *valA, int xI, float *sol, int paramNo, float *rP, float dr[])
{
  float x, a, f, w;
  float z, s;

  x = (float) xI;

  a = sol[0];
  f = sol[1];
  w = sol[2];

  if (w == 0.0)
    w = 1.0e-4;

  z = 2.0 * (f - x) / w;
  s = exp(- M_LN2 * z * z);

  dr[0] = s;
  dr[1] = - 4.0 * M_LN2 * a * z * s / w;
  dr[2] = 2.0 * M_LN2 * a * z * z * s / w;

  *rP = a * s - valA[xI];
}

static float
evalLorGau(AppShapeP shapeP, float x)
{
  float z, lor, gau;

  z = 2.0 * (shapeP->centerPosition - shapeP->startIndex - x) / shapeP->width;
  lor = 1.0 / (1.0 + z * z);
  gau = exp(- M_LN2 * z * z);
  
  return shapeP->amplitude *
      (shapeP->lorentzFraction * lor + shapeP->gaussFraction * gau);
}

void
AppShapeCalcPar(AppShapeP shapeP)
{
  float sol[4], rmsd;
  float *covar[4], covarA[4][4];
  BOOL fitOk;
  int i;

  estimatePar(shapeP);

  if (shapeP->valueNo < 3)
    return;

  for (i = 0; i < 4; i++)
    covar[i] = covarA[i];

  sol[0] = shapeP->amplitude;
  sol[1] = shapeP->centerPosition - shapeP->startIndex;
  sol[2] = shapeP->width;

  if (shapeP->valueNo > 3) {
  /* try a mixed Gauss/Lorentz */
    sol[3] = 0.5;  /* Lorentzian fraction */
    fitOk = Marquardt(fitLorGau,
	shapeP->valueA, shapeP->errorA, shapeP->valueNo,
	sol, 4, covar, &rmsd);
  } else {
    fitOk = Marquardt(fitGau,
	shapeP->valueA, shapeP->errorA, shapeP->valueNo,
	sol, 3, covar, &rmsd);
    sol[3] = 0.0;
  }

  if (sol[3] > 1.0) {
    fitOk = Marquardt(fitLor,
	shapeP->valueA, shapeP->errorA, shapeP->valueNo,
	sol, 3, covar, &rmsd);
    sol[3] = 1.0;
  } else if (sol[3] < 0.0) {
    fitOk = Marquardt(fitGau,
	shapeP->valueA, shapeP->errorA, shapeP->valueNo,
	sol, 3, covar, &rmsd);
    sol[3] = 0.0;
  }

  if (sol[2] < 0.0)
    sol[2] = - sol[2];

  /* test width to avoid creating huge shape when it is unreasonable */
  if (fitOk && sol[2] < 2.0 * shapeP->valueNo) {
    shapeP->amplitude = sol[0];
    shapeP->centerPosition = shapeP->startIndex + sol[1];
    shapeP->centerError = sqrt(covar[1][1]);
    shapeP->width = sol[2];
    shapeP->gaussFraction = 1.0 - sol[3];
    shapeP->lorentzFraction = sol[3];
  }
}

void
AppShapeCompleteLevel(AppShapeP shapeP, float level)
{
  AppShapeComplete(shapeP, - MAXINT, MAXINT, level);
}

void
AppShapeComplete(AppShapeP shapeP,
    int leftIndex, int rightIndex, float level)
{
  int leftInc, rightInc;
  int newValNo;
  float *newValA, *newErrA;
  int newI, i;

  leftInc = 0;
  while (shapeP->startIndex - leftInc > leftIndex &&
      evalLorGau(shapeP, - leftInc) >= level)
    leftInc++;

  rightInc = 0;
  while (shapeP->startIndex + shapeP->valueNo - 1 + rightInc < rightIndex &&
      evalLorGau(shapeP, shapeP->valueNo - 1 + rightInc) >= level)
    rightInc++;

  if (leftInc + rightInc == 0)
    return;

  newValNo = shapeP->valueNo + leftInc + rightInc;
  newValA = malloc(newValNo * sizeof(*shapeP->valueA));
  newErrA = malloc(newValNo * sizeof(*shapeP->errorA));

  for (i = 0; i < leftInc; i++) {
    newValA[i] = evalLorGau(shapeP, i - leftInc);
    newErrA[i] = shapeP->errorA[0];
  }

  for (i = 0; i < shapeP->valueNo; i++) {
    newValA[leftInc + i] = shapeP->valueA[i];
    newErrA[leftInc + i] = shapeP->errorA[i];
  }

  for (i = 0; i < rightInc; i++) {
    newI = leftInc + shapeP->valueNo + i;
    newValA[newI] = evalLorGau(shapeP, shapeP->valueNo + i);
    newErrA[newI] = shapeP->errorA[shapeP->valueNo - 1];
  }

  free(shapeP->valueA);
  free(shapeP->errorA);
  shapeP->startIndex -= leftInc;
  shapeP->valueNo = newValNo;
  shapeP->origLeft = leftInc;
  shapeP->valueA = newValA;
  shapeP->errorA = newErrA;
  (void) ShapeNorm(shapeP);
}

void
AppShapeOriginal(AppShapeP shapeP)
{
  int i;

  if (shapeP->valueNo == shapeP->origValueNo)
    return;

  if (shapeP->origLeft > 0)
    for (i = 0; i < shapeP->origValueNo; i++) {
      shapeP->valueA[i] = shapeP->valueA[shapeP->origLeft + i];
      shapeP->errorA[i] = shapeP->errorA[shapeP->origLeft + i];
    }
  
  shapeP->startIndex += shapeP->origLeft;
  shapeP->valueNo = shapeP->origValueNo;
  shapeP->origLeft = 0;
  shapeP->valueA = realloc(shapeP->valueA,
      shapeP->valueNo * sizeof(*shapeP->valueA));
  shapeP->errorA = realloc(shapeP->errorA,
      shapeP->valueNo * sizeof(*shapeP->errorA));
}

float
AppShapeCompare(AppShapeP shape1P, AppShapeP shape2P)
{
  int left1, left2, right1, right2, left, right;
  float v1, v2, vDiff, err1, err2, rmsd;
  int i;

  if (shape1P->domain != shape2P->domain)
    return 100.0;
  
  left1 = shape1P->startIndex;
  left2 = shape2P->startIndex;
  right1 = shape1P->startIndex + shape1P->valueNo - 1;
  right2 = shape2P->startIndex + shape2P->valueNo - 1;

  if (left1 > right2 || left2 > right1)
    return 100.0;

  if (left1 < left2)
    left = left1;
  else
    left = left2;

  if (right1 > right2)
    right = right1;
  else
    right = right2;

  AppShapeComplete(shape1P, left, right, 0.0);
  AppShapeComplete(shape2P, left, right, 0.0);

  rmsd = 0.0;
  for (i = left; i <= right; i++) {
    v1 = shape1P->valueA[i - left];
    v2 = shape2P->valueA[i - left];
    vDiff = v1 - v2;
    
    err1 = shape1P->errorA[i - left];
    err2 = shape2P->errorA[i - left];

    rmsd += vDiff * vDiff / (err1 * err1 + err2 * err2);
  }

  rmsd = sqrt(rmsd / (right - left + 1));

  AppShapeOriginal(shape1P);
  AppShapeOriginal(shape2P);

  vDiff = shape1P->centerPosition - shape2P->centerPosition;
  err1 = shape1P->centerError;
  err2 = shape2P->centerError;

  return 0.5 * rmsd + 0.01 * vDiff * vDiff / (err1 * err1 + err2 * err2);
}

void
AppShapeCombine(AppShapeP shape1P, AppShapeP shape2P)
{
  int left1, left2, right1, right2, left, right, origRight;
  float *newValA, *newErrA, weight1, weight2;
  float v1, v2, err1, err2, diff;
  AppShapeP **sPPP;
  int i;

  if (shape1P->domain != shape2P->domain)
    return;
  
  left1 = shape1P->startIndex;
  left2 = shape2P->startIndex;
  right1 = shape1P->startIndex + shape1P->valueNo - 1;
  right2 = shape2P->startIndex + shape2P->valueNo - 1;

  if (left1 > right2 || left2 > right1)
    return;  /* shapes must overlap */

  if (left1 < left2)
    left = left1;
  else
    left = left2;

  if (right1 > right2)
    right = right1;
  else
    right = right2;

  AppShapeComplete(shape1P, left, right, 0.0);
  AppShapeComplete(shape2P, left, right, 0.0);

  newValA = malloc((right - left + 1) * sizeof(*newValA));
  newErrA = malloc((right - left + 1) * sizeof(*newErrA));

  for (i = left; i <= right; i++) {
    err1 = shape1P->errorA[i - left];
    err2 = shape2P->errorA[i - left];

    if (err1 < 1.0e-10) {
      if (err2 < 1.0e-10) {
	weight1 = 0.5;
	weight2 = 0.5;
      } else {
	weight1 = 1.0;
	weight2 = 0.0;
      }
    } else if (err2 < 1.0e-10) {
      weight1 = 0.0;
      weight2 = 1.0;
    } else {
      weight1 = 1.0 / (err1 * err1);
      weight2 = 1.0 / (err2 * err2);
    }

    v1 = shape1P->valueA[i - left];
    v2 = shape2P->valueA[i - left];

    diff = (v1 - v2) * (v1 - v2) / (err1 * err1 + err2 * err2);
    if (diff > 1.0) {
      /* give a higher weight to smaller value if difference is large */
      if (v1 < v2)
	weight1 *= diff;
      else
	weight2 *= diff;
    }

    newValA[i - left] = (weight1 * v1 + weight2 * v2) / (weight1 + weight2);
    err1 *= weight1;
    err2 *= weight2;
    newErrA[i - left] = sqrt(err1 * err1 + err2 * err2) / (weight1 + weight2);
  }

  free(shape1P->valueA);
  free(shape1P->errorA);

  shape1P->startIndex = left;
  shape1P->valueNo = right - left + 1;
  shape1P->origLeft = 0;
  shape1P->origValueNo = shape1P->valueNo;
  shape1P->valueA = newValA;
  shape1P->errorA = newErrA;

  (void) ShapeNorm(shape1P);
  AppShapeCalcPar(shape1P);
  AppShapeOriginal(shape2P);

  if (shape1P->pointerList == NULL)
    shape1P->pointerList = ListOpen(sizeof(AppShapeP *));

  sPPP = ListFirst(shape2P->pointerList);
  while (sPPP != NULL) {
    (void) ListInsertLast(shape1P->pointerList, sPPP);
    **sPPP = shape1P;
    sPPP = ListNext(shape2P->pointerList, sPPP);
  }
}

AppShapeP
AppShapeCopy(AppShapeP shapeP)
{
  AppShapeP copyP;
  AppShapeP **sPPP;
  int i;

  copyP = malloc(sizeof(*copyP));

  *copyP = *shapeP;
  copyP->valueA = malloc(copyP->valueNo * sizeof(*copyP->valueA));
  copyP->errorA = malloc(copyP->valueNo * sizeof(*copyP->errorA));
  for (i = 0; i < copyP->valueNo; i++) {
    copyP->valueA[i] = shapeP->valueA[i];
    copyP->errorA[i] = shapeP->errorA[i];
  }

  if (shapeP->pointerList != NULL) {
    copyP->pointerList = ListOpen(sizeof(AppShapeP *));
    sPPP = ListFirst(shapeP->pointerList);
    while (sPPP != NULL) {
      (void) ListInsertLast(copyP->pointerList, sPPP);
      sPPP = ListNext(shapeP->pointerList, sPPP);
    }
  }

  return copyP;
}

void
AppShapeDestroy(AppShapeP shapeP)
{
  if (shapeP->valueNo > 0) {
    free(shapeP->valueA);
    free(shapeP->errorA);
  }

  if (shapeP->pointerList != NULL)
    ListClose(shapeP->pointerList);
}
