/*
************************************************************************
*
*   RegSymm.c - calculate symmetry of spectrum region
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/04
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.RegSymm.c
*   SCCS identification       : 1.5
*
************************************************************************
*/

#include <app/reg_symm.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>

#include <interpol.h>
#include <powell.h>
#include "conv.h"
#include "segment.h"

/* additonal number of data points at each end taken for spline calcluation */
#define SPLINE_INC 2

typedef struct {
  AppRegionDataP regDataP;
  float *origDataA;
  float minPosA[APP_MAX_DIM], maxPosA[APP_MAX_DIM];
  float splitDiff;
  float noiseMagn;
  AppDataRange rangeA[APP_MAX_DIM];
  float *symmDataA;
  BOOL *symmValidA;
} SymmData;

typedef struct {
  int dim;
  BOOL *fullValidA, *validA;
  int fullSizeA[APP_MAX_DIM], sizeA[APP_MAX_DIM], offsA[APP_MAX_DIM];
  AppDataRange *subRangeA;
} FillData;

static void
symmetrize(AppRegionDataP regDataP, float posA[], BOOL dirSelA[],
    float *resDataA, BOOL *resValidA)
{
  float *dataA, *tmp1, *tmp2;
  BOOL *tmpValidA;
  int fullSizeA[APP_MAX_DIM], sizeA[APP_MAX_DIM], resSizeA[APP_MAX_DIM];
  int leftA[APP_MAX_DIM], rightA[APP_MAX_DIM], coord[APP_MAX_DIM];
  float *xA, *yA, *y2A;
  int totSize, inc, dom, fullInd, ind, resInd, i;
  float ps, vs;

  dataA = ConvGetRegionFloat(regDataP);

  totSize = 1;
  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    fullSizeA[dom] = regDataP->rangeA[dom][1] - regDataP->rangeA[dom][0] + 1;
    leftA[dom] = regDataP->subRangeA[dom][0] - SPLINE_INC;
    if (leftA[dom] < 0)
      leftA[dom] = 0;
    rightA[dom] = regDataP->subRangeA[dom][1] + SPLINE_INC;
    if (rightA[dom] >= fullSizeA[dom])
      rightA[dom] = fullSizeA[dom] - 1;
    sizeA[dom] = rightA[dom] - leftA[dom] + 1;
    totSize *= sizeA[dom];
  }

  tmp1 = malloc(totSize * sizeof(*tmp1));
  tmp2 = malloc(totSize * sizeof(*tmp2));
  tmpValidA = malloc(totSize * sizeof(*tmpValidA));

  for (i = 0; i < regDataP->dimensionNo; i++)
    coord[i] = 0;

  ind = 0;
  for (;;) {
    fullInd = 0;
    for (dom = regDataP->dimensionNo - 1; dom >= 0; dom--)
      fullInd = fullInd * fullSizeA[dom] + coord[dom] + leftA[dom];
    
    tmp1[ind] = dataA[fullInd];
    tmpValidA[ind] = regDataP->validA[fullInd];

    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      if (coord[dom] == sizeA[dom] - 1) {
        coord[dom] = 0;
      } else {
        coord[dom]++;
        break;
      }
    }

    if (dom == regDataP->dimensionNo)
      break;

    ind++;
  }

  inc = 1;

  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    if (dirSelA != NULL && ! dirSelA[dom]) {
      inc *= sizeA[dom];
      continue;
    }

    xA = malloc(sizeA[dom] * sizeof(*xA));
    for (i = 0; i < sizeA[dom]; i++)
      xA[i] = (float) (leftA[dom] + i);

    yA = malloc(sizeA[dom] * sizeof(*yA));
    y2A = malloc(sizeA[dom] * sizeof(*y2A));

    for (i = 0; i < regDataP->dimensionNo; i++)
      coord[i] = 0;
    
    for (;;) {
      ind = 0;
      for (i = regDataP->dimensionNo - 1; i >= 0; i--)
	ind = ind * sizeA[i] + coord[i];

      for (i = 0; i < sizeA[dom]; i++)
	yA[i] = tmp1[ind + i * inc];

      (void) InterpolSplineCalc(xA, yA, sizeA[dom], y2A);

      for (i = 0; i < sizeA[dom]; i++) {
	ps = 2.0 * posA[dom] - (leftA[dom] + i);
	if (ps < leftA[dom] || ps > rightA[dom])
	  vs = 0.0;
	else
	  (void) InterpolSplineVal(xA, yA, y2A, sizeA[dom], ps, &vs);

	if (vs < tmp1[ind + i * inc]) {
	  tmp2[ind + i * inc] = vs;
	  if (vs < regDataP->baseLevel)
	    tmpValidA[ind + i * inc] = FALSE;
	} else {
	  tmp2[ind + i * inc] = tmp1[ind + i * inc];
	}
      }

      for (i = 0; i < regDataP->dimensionNo; i++) {
	if (i == dom)
	  continue;
	if (coord[i] == sizeA[i] - 1) {
	  coord[i] = 0;
	} else {
	  coord[i]++;
	  break;
	}
      }

      if (i == regDataP->dimensionNo)
	break;
    }

    for (i = 0; i < totSize; i++)
      tmp1[i] = tmp2[i];

    inc *= sizeA[dom];

    free(xA);
    free(yA);
    free(y2A);
  }

  for (i = 0; i < regDataP->dimensionNo; i++) {
    resSizeA[i] = regDataP->subRangeA[i][1] - regDataP->subRangeA[i][0] + 1;
    coord[i] = 0;
  }

  resInd = 0;
  for (;;) {
    ind = 0;
    for (i = regDataP->dimensionNo - 1; i >= 0; i--)
      ind = ind * sizeA[i] + coord[i] + regDataP->subRangeA[i][0] - leftA[i];

    resDataA[resInd] = tmp1[ind];
    resValidA[resInd] = tmpValidA[ind];

    for (i = 0; i < regDataP->dimensionNo; i++)
      if (coord[i] == resSizeA[i] - 1) {
	coord[i] = 0;
      } else {
	coord[i]++;
	break;
      }

    if (i == regDataP->dimensionNo)
      break;

    resInd++;
  }

  free(tmp1);
  free(tmp2);
  free(tmpValidA);
}

static float
getSymm(float cA[], int dim, void *clientData)
{
  SymmData *dataP = clientData;
  AppRegionDataP regDataP = dataP->regDataP;
  int fullSizeA[APP_MAX_DIM], fullIncA[APP_MAX_DIM];
  int sizeA[APP_MAX_DIM], incA[APP_MAX_DIM], coord[APP_MAX_DIM];
  float target, origMaxVal, symmMaxVal, derSum;
  float diff, nDiff, relNoise, t;
  int fullInd, ind, nInd, dom;

  symmetrize(regDataP, cA, NULL, dataP->symmDataA, dataP->symmValidA);

  for (dom = 0; dom < dim; dom++) {
    fullSizeA[dom] = regDataP->rangeA[dom][1] - regDataP->rangeA[dom][0] + 1;
    sizeA[dom] = regDataP->subRangeA[dom][1] - regDataP->subRangeA[dom][0] + 1;
    coord[dom] = 0;
  }

  fullIncA[0] = 1;
  incA[0] = 1;
  for (dom = 1; dom < dim; dom++) {
    fullIncA[dom] = fullIncA[dom - 1] * fullSizeA[dom - 1];
    incA[dom] = incA[dom - 1] * sizeA[dom - 1];
  }

  origMaxVal = 0.0;
  symmMaxVal = 0.0;
  derSum = 0.0;

  ind = 0;
  for (;;) {
    fullInd = 0;
    for (dom = regDataP->dimensionNo - 1; dom >= 0; dom--)
      fullInd = fullInd * fullSizeA[dom] + coord[dom] +
	  regDataP->subRangeA[dom][0];

    if (regDataP->validA[fullInd]) {
      if (dataP->origDataA[fullInd] > origMaxVal)
	origMaxVal = dataP->origDataA[fullInd];

      if (dataP->symmDataA[ind] > symmMaxVal)
	symmMaxVal = dataP->symmDataA[ind];

      diff = dataP->origDataA[fullInd] - dataP->symmDataA[ind];

      for (dom = 0; dom < dim; dom++) {
	nInd = fullInd - fullIncA[dom];
	if (regDataP->validA[nInd]) {
	  nDiff = dataP->origDataA[nInd] - dataP->symmDataA[ind - incA[dom]];
	  if (diff > nDiff)
	    derSum += diff - nDiff;
	  else
	    derSum -= diff - nDiff;
	} else {
	  if (diff > 0.0)
	    derSum += diff;
	  else
	    derSum -= diff;
	}

	nInd = fullInd + fullIncA[dom];
	if (! regDataP->validA[nInd]) {
	  if (diff > 0.0)
	    derSum += diff;
	  else
	    derSum -= diff;
	}
      }
    }

    for (dom = 0; dom < dim; dom++)
      if (coord[dom] == sizeA[dom] - 1) {
	coord[dom] = 0;
      } else {
	coord[dom]++;
	break;
      }

    if (dom == dim)
      break;
    
    ind++;
  }

  derSum /= origMaxVal;
  relNoise = dataP->noiseMagn / origMaxVal;
  diff = 1.5 * (origMaxVal - symmMaxVal) /
      (origMaxVal * (dataP->splitDiff + relNoise));
  if (diff < 0.0)
    diff = 0.0;

  target = derSum + diff * diff * diff * diff;

  for (dom = 0; dom < dim; dom++) {
    if (cA[dom] < dataP->minPosA[dom]) {
      t = 10.0 * (dataP->minPosA[dom] - cA[dom]);
      target += t * t * t * t;
    } else if (cA[dom] > dataP->maxPosA[dom]) {
      t = 10.0 * (cA[dom] - dataP->maxPosA[dom]);
      target += t * t * t * t;
    }
  }

  return target;
}

void
AppBestSymmetry(AppRegionDataP regDataP, int maxPosA[],
    float maxSplitA[], float splitDiff, float noiseMagn, float posA[])
{
  float *dataA;
  int maxCoordA[APP_MAX_DIM];
  int totSize;
  SymmData data;
  float bestSymm, symm;
  int bestI, startI, endI;
  int dom, i;

  dataA = ConvGetRegionFloat(regDataP);

  totSize = 1;
  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    maxCoordA[dom] = maxPosA[dom] - regDataP->rangeA[dom][0];
    totSize *= regDataP->subRangeA[dom][1] - regDataP->subRangeA[dom][0] + 1;
  }

  data.regDataP = regDataP;

  data.origDataA = dataA;
  data.symmDataA = malloc(totSize * sizeof(*data.symmDataA));
  data.symmValidA = malloc(totSize * sizeof(*data.symmValidA));

  data.splitDiff = splitDiff;
  data.noiseMagn = noiseMagn;

  if (maxSplitA == NULL) {
    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      data.minPosA[dom] = (float) regDataP->subRangeA[dom][0];
      data.maxPosA[dom] = (float) regDataP->subRangeA[dom][1];
      posA[dom] = (float) maxCoordA[dom];
    }
  } else {
    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      data.minPosA[dom] = maxCoordA[dom] - 0.5 * maxSplitA[dom];
      if (data.minPosA[dom] < regDataP->subRangeA[dom][0])
	data.minPosA[dom] = (float) regDataP->subRangeA[dom][0];

      data.maxPosA[dom] = maxCoordA[dom] + 0.5 * maxSplitA[dom];
      if (data.maxPosA[dom] > regDataP->subRangeA[dom][1])
	data.maxPosA[dom] = (float) regDataP->subRangeA[dom][1];

      posA[dom] = (float) maxCoordA[dom];
    }

    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      bestSymm = MAXFLOAT;
      startI = (int) (data.minPosA[dom] + 0.99);
      endI = (int) data.maxPosA[dom];
      for (i = startI; i <= endI; i++) {
	posA[dom] = (float) i;
	symm = getSymm(posA, regDataP->dimensionNo, &data);
	if (symm < bestSymm) {
	  bestI = i;
	  bestSymm = symm;
	}
      }
      posA[dom] = (float) bestI;
    }
  }

  (void) PowellMin(getSymm, posA, regDataP->dimensionNo, &data, 0.01);

  free(data.symmDataA);
  free(data.symmValidA);

  for (dom = 0; dom < regDataP->dimensionNo; dom++)
    posA[dom] += regDataP->rangeA[dom][0];
}

void
AppCalcSymmetry(AppRegionDataP regDataP,
    float posA[], float errA[])
{
  float *dataA;
  int fullSizeA[APP_MAX_DIM], sizeA[APP_MAX_DIM], coord[APP_MAX_DIM];
  float cA[APP_MAX_DIM], diff;
  float *symmDataA;
  BOOL *symmValidA;
  BOOL dirSelA[APP_MAX_DIM];
  int totSize, dom, fullInd, ind, i;

  dataA = ConvGetRegionFloat(regDataP);

  totSize = 1;
  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    fullSizeA[dom] = regDataP->rangeA[dom][1] - regDataP->rangeA[dom][0] + 1;
    sizeA[dom] = regDataP->subRangeA[dom][1] - regDataP->subRangeA[dom][0] + 1;
    totSize *= sizeA[dom];
    cA[dom] = posA[dom] - regDataP->rangeA[dom][0];
  }

  symmDataA = malloc(totSize * sizeof(*symmDataA));
  symmValidA = malloc(totSize * sizeof(*symmValidA));

  for (dom = 0; dom < regDataP->dimensionNo; dom++)
    dirSelA[dom] = FALSE;

  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    dirSelA[dom] = TRUE;

    symmetrize(regDataP, cA, dirSelA, symmDataA, symmValidA);

    errA[dom] = 0.0;

    for (i = 0; i < regDataP->dimensionNo; i++)
      coord[i] = 0;

    ind = 0;
    for (;;) {
      fullInd = 0;
      for (i = regDataP->dimensionNo - 1; i >= 0; i--)
	fullInd = fullInd * fullSizeA[i] + coord[i] +
	    regDataP->subRangeA[i][0];

      if (regDataP->validA[fullInd]) {
	diff = dataA[fullInd] - symmDataA[ind];
	if (diff > errA[dom])
	  errA[dom] = diff;
      }

      for (i = 0; i < regDataP->dimensionNo; i++) {
	if (coord[i] == sizeA[i] - 1) {
	  coord[i] = 0;
	} else {
	  coord[i]++;
	  break;
	}
      }

      if (i == regDataP->dimensionNo)
	break;

      ind++;
    }

    dirSelA[dom] = FALSE;
  }

  free(symmDataA);
  free(symmValidA);
}

static BOOL
fillCheck(int *coord, BOOL sameRow, void *clientData)
{
  FillData *dataP = clientData;
  int ind, dom;

  ind = 0;
  for (dom = dataP->dim - 1; dom >= 0; dom--) {
    if (coord[dom] < dataP->offsA[dom] ||
	coord[dom] >= dataP->offsA[dom] + dataP->sizeA[dom])
      return FALSE;

    ind = ind * dataP->sizeA[dom] + coord[dom] - dataP->offsA[dom];
  }

  if (! dataP->validA[ind])
    return FALSE;

  ind = 0;
  for (dom = dataP->dim - 1; dom >= 0; dom--)
    ind = ind * dataP->fullSizeA[dom] + coord[dom];
  
  return ! dataP->fullValidA[ind];
}

static void
fillAddSpan(int *coord, int left, int right, void *clientData)
{
  FillData *dataP = clientData;
  int ind, dom, i;

  ind = 0;
  for (dom = dataP->dim - 1; dom >= 1; dom--)
    ind = ind * dataP->fullSizeA[dom] + coord[dom];
  ind *= dataP->fullSizeA[0];

  for (i = left; i <= right; i++)
    dataP->fullValidA[ind + i] = TRUE;

  for (dom = 1; dom < dataP->dim; dom++) {
    if (coord[dom] < dataP->subRangeA[dom][0])
      dataP->subRangeA[dom][0] = coord[dom];
    if (coord[dom] > dataP->subRangeA[dom][1])
      dataP->subRangeA[dom][1] = coord[dom];
  }

  if (left < dataP->subRangeA[0][0])
    dataP->subRangeA[0][0] = left;
  if (right > dataP->subRangeA[0][1])
    dataP->subRangeA[0][1] = right;
}

void
AppSymmetrize(AppRegionDataP regDataP, float posA[], BOOL dirSelA[])
{
  int fullSizeA[APP_MAX_DIM], sizeA[APP_MAX_DIM], coord[APP_MAX_DIM];
  float cA[APP_MAX_DIM];
  int seedA[APP_MAX_DIM];
  float *resDataA;
  BOOL *resValidA;
  FillData data;
  int totSize, dom, fullInd, ind;

  totSize = 1;
  for (dom = 0; dom < regDataP->dimensionNo; dom++) {
    fullSizeA[dom] = regDataP->rangeA[dom][1] - regDataP->rangeA[dom][0] + 1;
    sizeA[dom] = regDataP->subRangeA[dom][1] - regDataP->subRangeA[dom][0] + 1;
    totSize *= sizeA[dom];
    cA[dom] = posA[dom] - regDataP->rangeA[dom][0];
  }

  resDataA = malloc(totSize * sizeof(*resDataA));
  resValidA = malloc(totSize * sizeof(*resValidA));

  symmetrize(regDataP, cA, dirSelA, resDataA, resValidA);

  for (dom = 0; dom < regDataP->dimensionNo; dom++)
    coord[dom] = 0;

  ind = 0;
  for (;;) {
    fullInd = 0;
    for (dom = regDataP->dimensionNo - 1; dom >= 0; dom--)
      fullInd = fullInd * fullSizeA[dom] + coord[dom] +
	  regDataP->subRangeA[dom][0];

    if (regDataP->validA[fullInd]) {
      if (regDataP->dataType == ADT_FLT32)
	((float *) regDataP->dataA)[fullInd] = resDataA[ind];
      else
	((int *) regDataP->dataA)[fullInd] = (int) resDataA[ind];
    }

    regDataP->validA[fullInd] = FALSE;

    for (dom = 0; dom < regDataP->dimensionNo; dom++) {
      if (coord[dom] == sizeA[dom] - 1) {
        coord[dom] = 0;
      } else {
        coord[dom]++;
        break;
      }
    }

    if (dom == regDataP->dimensionNo)
      break;

    ind++;
  }

  /* make sure that symmetrized region is connected by doing a
     flood fill */
  data.dim = regDataP->dimensionNo;
  data.fullValidA = regDataP->validA;
  data.validA = resValidA;
  data.subRangeA = regDataP->subRangeA;

  for (dom = 0; dom < data.dim; dom++) {
    data.fullSizeA[dom] = fullSizeA[dom];
    data.sizeA[dom] = sizeA[dom];
    data.offsA[dom] = regDataP->subRangeA[dom][0];
    seedA[dom] = (int) (cA[dom] + 0.5);
    data.subRangeA[dom][0] = seedA[dom] + 1;
    data.subRangeA[dom][1] = seedA[dom] - 1;
  }

  SegFill(data.dim, seedA, fillCheck, fillAddSpan, &data);

  free(resDataA);
  free(resValidA);
}
