/*
************************************************************************
*
*   Elim.c - eliminate peaks in region
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/08
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.Elim.c
*   SCCS identification       : 1.7
*
************************************************************************
*/

#include <app/elim.h>

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>

#include <app/reg_symm.h>
#include <app/reg_shape.h>
#include <app/reg.h>
#include <app/spec_segm.h>
#include <app/peak.h>
#include <app/shape_list.h>
#include "conv.h"
#include "shape.h"
#include "reg_struc.h"

#define MAX_LEVEL 0.5
#define ERR_FACT 1.2
#define LEVEL_FACT 0.8

typedef struct {
  int dim;
  AppMaxP maxP;
  float amp;
  float posA[APP_MAX_DIM];
  BOOL separated, valid;
  float unifErr, symmErrA[APP_MAX_DIM];
} MaxData;

static BOOL
compVal(MaxData *max1P, MaxData *max2P)
{
  float errSum1, errSum2, errViol1, errViol2;
  int dom;

  errSum1 = max1P->unifErr;
  errViol1 = max1P->unifErr - 1.0;
  if (errViol1 < 0.0)
    errViol1 = 0.0;
  for (dom = 0; dom < max1P->dim; dom++) {
    errSum1 += max1P->symmErrA[dom];
    if (max1P->symmErrA[dom] > 1.0)
      errViol1 += max1P->symmErrA[dom] - 1.0;
  }

  errSum2 = max2P->unifErr;
  errViol2 = max2P->unifErr - 1.0;
  if (errViol2 < 0.0)
    errViol2 = 0.0;
  for (dom = 0; dom < max2P->dim; dom++) {
    errSum2 += max2P->symmErrA[dom];
    if (max2P->symmErrA[dom] > 1.0)
      errViol2 += max2P->symmErrA[dom] - 1.0;
  }

  if (errViol1 > 0.0 && errViol2 > 0.0)
    return errViol1 / max1P->amp < errViol2 / max2P->amp;

  if (errViol1 > 0.0)
    return FALSE;

  if (errViol2 > 0.0)
    return TRUE;
  
  return errSum1 < errSum2;
}

static BOOL
compMax(MaxData *max1P, MaxData *max2P)
{
  if (! max1P->valid)
    return FALSE;

  if (! max2P->valid)
    return TRUE;

  if (! max1P->separated && ! max2P->separated)
    return max1P->maxP->amplitude / max1P->maxP->noiseMagnitude >
	max2P->maxP->amplitude / max2P->maxP->noiseMagnitude;

  if (! max1P->separated)
    return FALSE;

  if (! max2P->separated)
    return TRUE;

  return compVal(max1P, max2P);
}

static void
sortMax(MaxData *maxA, int leftI, int rightI)
{
  int i, j;
  MaxData x, w;

  /* quicksort (Wirth) */
  i = leftI;
  j = rightI;
  x = maxA[(leftI + rightI) / 2];
  for (;;) {
    while (compMax(maxA + i, &x))
      i++;
    while (compMax(&x, maxA + j))
      j--;

    if (i <= j) {
      w = maxA[i];
      maxA[i] = maxA[j];
      maxA[j] = w;
      i++;
      j--;
    }

    if (i > j)
      break;
  }

  if (leftI < j)
    sortMax(maxA, leftI, j);
  if (i < rightI)
    sortMax(maxA, i, rightI);
}

void
AppElimAll(AppRegionP regP, float minLevel, int minSizeA[],
    float maxSplitA[], float splitDiff,
    float maxErrFact)
{
  float baseLevel, errLevel;
  int dim, dom;
  float *noiseA;
  int maxNo, maxI;
  MaxData *maxA;
  struct AppShapeS shapeA[APP_MAX_DIM];
  BOOL accA[APP_MAX_DIM];
  AppShapeP accShapePA[APP_MAX_DIM], shapePA[APP_MAX_DIM];
  float posA[APP_MAX_DIM], unifErr, symmErrA[APP_MAX_DIM];
  float minAmp, amp;
  float noiseMagn, level;
  float symmErrInc, unifErrInc;
  float errSum, err;
  int errNo, subtrNo;
  int firstInNo, invalidInNo, inNo;
  int phase;
  BOOL tooSmall;
  struct AppPeakS peakS;
  AppPeakP peakP;
  AppShapeP *shapePP;
  int i;

  minLevel *= regP->segLevel;
  if (minLevel < 1.0 / MAX_LEVEL)
    minLevel = 1.0 / MAX_LEVEL;
  
  baseLevel = regP->d.baseLevel;
  errLevel = 0.0;
  dim = regP->d.dimensionNo;

  for (dom = 0; dom < dim; dom++)
    if (minSizeA[dom] < 1)
      minSizeA[dom] = 1;

  noiseA = RegionGetNoise(regP);

  for (;;) {
    maxNo = ListSize(regP->maxL);
    if (maxNo == 0)
      break;

    maxA = malloc(maxNo * sizeof(*maxA));

    maxA[0].maxP = ListFirst(regP->maxL);
    for (maxI = 1; maxI < maxNo; maxI++)
      maxA[maxI].maxP = ListNext(regP->maxL, maxA[maxI - 1].maxP);

    for (maxI = 0; maxI < maxNo; maxI++) {
      maxA[maxI].dim = dim;
      maxA[maxI].valid = (maxA[maxI].maxP->amplitude >
	  minLevel * maxA[maxI].maxP->noiseMagnitude &&
	  maxA[maxI].maxP->amplitude > errLevel);
      maxA[maxI].separated = TRUE;
    }

    for (maxI = 0; maxI < maxNo; maxI++) {
      if (! maxA[maxI].valid)
	continue;

      if (! maxA[maxI].separated)
	continue;

      noiseMagn = maxA[maxI].maxP->noiseMagnitude;

      level = MAX_LEVEL * maxA[maxI].maxP->amplitude;
      for (;;) {
	AppRegionPushState(regP, APP_PUSH_VALID);
	AppRegionSubsegment(regP, level, maxA[maxI].maxP->coordinates);

	tooSmall = FALSE;
	for (dom = 0; dom < dim; dom++)
	  if (regP->d.subRangeA[dom][1] - regP->d.subRangeA[dom][0] + 1 <
	      minSizeA[dom])
	    tooSmall = TRUE;
	if (! tooSmall)
	  break;

	AppRegionPopState(regP);

	if (level < baseLevel)
	  break;

	level *= LEVEL_FACT;
      }

      if (tooSmall) {
	maxA[maxI].valid = FALSE;
	continue;
      }

      for (i = 0; i < maxI; i++)
	if (AppRegionIsInside(&regP->d, maxA[i].maxP->coordinates)) {
	  maxA[maxI].separated = FALSE;
	  break;
	}

      if (! maxA[maxI].separated) {
	AppRegionPopState(regP);
	continue;
      }

      AppCalcShapes(&regP->d, noiseA,
	  shapeA, &maxA[maxI].amp, &maxA[maxI].unifErr);
      maxA[maxI].unifErr = maxA[maxI].unifErr / noiseMagn;
      for (dom = 0; dom < dim; dom++)
	AppShapeDestroy(shapeA + dom);

      AppBestSymmetry(&regP->d, maxA[maxI].maxP->coordinates,
	  maxSplitA, splitDiff, noiseMagn, maxA[maxI].posA);
      AppCalcSymmetry(&regP->d, maxA[maxI].posA, maxA[maxI].symmErrA);
      for (dom = 0; dom < dim; dom++)
	maxA[maxI].symmErrA[dom] =
	    0.5 * maxA[maxI].symmErrA[dom] / noiseMagn;

      for (i = maxI + 1; i < maxNo; i++)
	if (AppRegionIsInside(&regP->d, maxA[i].maxP->coordinates))
	  maxA[i].separated = FALSE;

      AppRegionPopState(regP);
    }

    symmErrInc = maxErrFact * regP->d.maxAmplitude;
    unifErrInc = 0.0;

    sortMax(maxA, 0, maxNo - 1);

    subtrNo = 0;
    for (maxI = 0; maxI < maxNo; maxI++) {
      if (! maxA[maxI].valid)
	break;

      noiseMagn = maxA[maxI].maxP->noiseMagnitude;

      minAmp = 0.75 * maxA[maxI].maxP->amplitude;
      if (minLevel * noiseMagn > minAmp)
	minAmp = minLevel * noiseMagn;
      if (errLevel > minAmp)
	minAmp = errLevel;

      for (dom = 0; dom < dim; dom++)
	accShapePA[dom] = NULL;

      maxA[maxI].amp = maxA[maxI].maxP->amplitude;
      level = MAX_LEVEL * maxA[maxI].maxP->amplitude;
      phase = 0;

      for (;;) {
	AppRegionPushState(regP, APP_PUSH_VALID);
	AppRegionSubsegment(regP, level, maxA[maxI].maxP->coordinates);

	inNo = 0;
	invalidInNo = 0;
	for (i = 0; i < maxNo; i++) {
	  if (i == maxI)
	    continue;

	  if (AppRegionIsInside(&regP->d, maxA[i].maxP->coordinates)) {
	    inNo++;
	    if (i < maxI && ! maxA[i].valid)
	      invalidInNo++;
	  }
	}

	if (phase == 0) {
	  if (! maxA[maxI].separated && invalidInNo > 0) {
	    AppRegionPopState(regP);
	    break;
	  }
	  firstInNo = inNo;
	} else if (inNo > firstInNo) {
	  AppRegionPopState(regP);
	  break;
	}

	AppCalcShapes(&regP->d, noiseA,
	    shapeA, &amp, &unifErr);
	unifErr = unifErr / noiseMagn;

	tooSmall = FALSE;
	for (dom = 0; dom < dim; dom++) {
	  if (shapeA[dom].valueNo < minSizeA[dom])
	    tooSmall = TRUE;
	  AppShapeDestroy(shapeA + dom);
	}

	if (amp < minAmp) {
	  maxA[maxI].amp = amp;
	  AppRegionPopState(regP);
	  break;
	}

	if (phase == 0 && ! tooSmall)
	  phase = 1;

	if (phase == 1)
	  AppBestSymmetry(&regP->d, maxA[maxI].maxP->coordinates,
	      maxSplitA, splitDiff, noiseMagn, maxA[maxI].posA);

	if (phase > 0) {
	  AppCalcSymmetry(&regP->d, maxA[maxI].posA, symmErrA);
	  for (dom = 0; dom < dim; dom++)
	    symmErrA[dom] = 0.5 * symmErrA[dom] / noiseMagn;
	}

	if (phase == 1 || phase == 2) {
	  maxA[maxI].unifErr = unifErr;
	  for (dom = 0; dom < dim; dom++) {
	    maxA[maxI].symmErrA[dom] = symmErrA[dom];
	    accA[dom] = TRUE;
	  }
	} else if (phase == 3) {
	  if (unifErr > 1.0 && unifErr > ERR_FACT * maxA[maxI].unifErr) {
	    AppRegionPopState(regP);
	    break;
	  }

	  errNo = 0;
	  for (dom = 0; dom < dim; dom++) {
	    accA[dom] = ! (symmErrA[dom] > 1.0 &&
		symmErrA[dom] > ERR_FACT * maxA[maxI].symmErrA[dom]);
	    if (! accA[dom])
	      errNo++;
	  }

	  if (errNo == dim) {
	    AppRegionPopState(regP);
	    break;
	  }
	}

	if (phase == 1)
	  phase = 2;

	if (phase > 1) {
	  AppRegionPushState(regP, APP_PUSH_DATA | APP_PUSH_VALID);
	  AppSymmetrize(&regP->d, maxA[maxI].posA, accA);
	  AppCalcShapes(&regP->d, noiseA,
	      shapeA, &maxA[maxI].amp, &unifErr);
	  AppRegionPopState(regP);

	  if (maxA[maxI].amp < minAmp) {
	    for (dom = 0; dom < dim; dom++)
	      AppShapeDestroy(shapeA + dom);
	    AppRegionPopState(regP);
	    break;
	  }

	  tooSmall = FALSE;
	  for (dom = 0; dom < dim; dom++)
	    if (shapeA[dom].valueNo < minSizeA[dom])
	      tooSmall = TRUE;

	  if (tooSmall) {
	    phase = 2;
	  } else {
	    if (unifErr - noiseMagn > unifErrInc)
	      unifErrInc = unifErr - noiseMagn;

	    for (dom = 0; dom < dim; dom++)
	      if (accA[dom]) {
		if (accShapePA[dom] != NULL) {
		  AppShapeDestroy(accShapePA[dom]);
		  free(accShapePA[dom]);
		}
		accShapePA[dom] = AppShapeCopy(shapeA + dom);
	      }

	    phase = 3;
	  }

	  for (dom = 0; dom < dim; dom++)
	    AppShapeDestroy(shapeA + dom);
	}

	AppRegionPopState(regP);

	if (level < baseLevel)
	  break;

	level *= LEVEL_FACT;
      }

      if (phase < 3 || maxA[maxI].amp < minAmp) {
	for (dom = 0; dom < dim; dom++)
	  if (accShapePA[dom] != NULL) {
	    AppShapeDestroy(accShapePA[dom]);
	    free(accShapePA[dom]);
	  }

	if (maxA[maxI].amp >= minAmp)
	  maxA[maxI].valid = FALSE;

	continue;
      }

      for (dom = 0; dom < dim; dom++) {
	shapePA[dom] = AppShapeCopy(accShapePA[dom]);
	shapeA[dom] = *shapePA[dom];
      }

      for (dom = 0; dom < dim; dom++) {
	AppShapeCalcPar(accShapePA[dom]);
	AppExtendShape(&regP->d, noiseA, shapeA, maxA[maxI].amp,
	    accShapePA[dom], dom);
	for (i = 0; i < accShapePA[dom]->valueNo; i++) {
	  err = accShapePA[dom]->errorA[i];
	  errSum = err * err;

	  if (errLevel > 0.0) {
	    err = (errLevel - noiseMagn) / maxA[maxI].amp;
	    errSum += err * err;
	  }

	  err = maxA[maxI].symmErrA[dom] * noiseMagn / maxA[maxI].amp;
	  errSum += err * err;

	  accShapePA[dom]->errorA[i] = sqrt(errSum);
	}
	AppShapeCalcPar(accShapePA[dom]);
      }

      for (dom = 0; dom < dim; dom++) {
	AppShapeDestroy(shapePA[dom]);
	free(shapePA[dom]);
      }

      for (dom = 0; dom < dim; dom++) {
	AppShapeComplete(accShapePA[dom],
	    regP->d.rangeA[dom][0], regP->d.rangeA[dom][1],
	    0.1 * noiseMagn / maxA[maxI].amp);
	shapeA[dom] = *accShapePA[dom];
      }

      AppSubtractShapes(&regP->d, shapeA, maxA[maxI].amp);
      subtrNo++;

      for (dom = 0; dom < dim; dom++)
	AppShapeOriginal(accShapePA[dom]);

      peakP = AppAddPeak(regP->specP, &peakS);

      for (dom = 0; dom < dim; dom++) {
	peakP->positionA[dom] = maxA[maxI].posA[dom];
	peakP->symmetryErrorA[dom] = maxA[maxI].symmErrA[dom];

	peakP->shapePA[dom] = AppAddShape(regP->specP, accShapePA[dom]);
	peakP->shapePA[dom]->pointerList = ListOpen(sizeof(AppShapeP *));
	shapePP = peakP->shapePA + dom;
	(void) ListInsertLast(peakP->shapePA[dom]->pointerList, &shapePP);
	AppShapeDestroy(accShapePA[dom]);
	free(accShapePA[dom]);
      }
      
      peakP->amplitude = maxA[maxI].amp;
      peakP->uniformError = maxA[maxI].unifErr;
      peakP->integrationLevel = 0.5 * noiseMagn / peakP->amplitude;

      errSum = 0.0;

      err = errLevel / noiseMagn - 1.0;
      if (err > 0.0)
	errSum = err * err;

      err = peakP->uniformError;
      errSum = err * err;

      for (dom = 0; dom < dim; dom++) {
	err = peakP->symmetryErrorA[dom];
	errSum = err * err;
      }

      peakP->quality = 1.0 - sqrt(errSum) * noiseMagn / peakP->amplitude;

      /* less quality for smaller peaks, factor 0.5 for minimum */
      peakP->quality *= 1.0 - 0.5 * minLevel * noiseMagn / peakP->amplitude;

      if (peakP->quality < 0.1)
	peakP->quality = 0.1;
    }

    errLevel = sqrt(errLevel * errLevel +
	symmErrInc * symmErrInc + unifErrInc * unifErrInc +
	4.0 * regP->d.noiseMagnitudeMax * regP->d.noiseMagnitudeMax);

    AppMaxUpdate(regP);

    free(maxA);

    if (subtrNo == 0)
      break;
  }

  free(noiseA);
}
