/*
************************************************************************
*
*   Deconv.c - deconvolute region
*
*   Copyright (c) 1996
*
*   SPECTROSPIN AG
*   Industriestr. 26
*   CH-8117 Faellanden
*
*   All Rights Reserved
*
*   Date of last modification : 96/10/15
*   Pathname of SCCS file     : /sgiext/autopsy/app/src/app/SCCS/s.Deconv.c
*   SCCS identification       : 1.7
*
************************************************************************
*/

#include <app/deconv.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#include <math.h>

#include <rand_num.h>
#include <linlist.h>
#include <queue.h>
#include <conj_grad.h>
#include <app/spec_segm.h>
#include <app/peak.h>
#include "conv.h"
#include "reg_struc.h"

#define MAX_NO 50

#define PS_OUTSIDE    -1
#define PS_UNASSIGNED -2
#define PS_QUEUED     -3

typedef struct {
  int segI;
  AppDataRange subRangeA[APP_MAX_DIM];
  int insideNo;
} MaxData;

typedef struct {
  int ind;
  float val;
  int segI;
} QueueEntry;

typedef struct {
  float val;
  int seg1I, seg2I;
} NeighListEntry;

typedef struct {
  AppRegionP regP;
  int dim;
  LINLIST *shapeLA;
} ShapeLists;

typedef enum {
  CS_ALREADY,
  CS_MAYBE,
  CS_NEW
} CombState;

typedef struct {
  AppRegionP regP;
  int dim;
  LINLIST *shapeLA;
  BOOL *insideA;
  CombState *combA;
} PeakData;

typedef struct {
  AppShapeP shapePA[APP_MAX_DIM];
  float *valA;
  float *errA;
  float integr;
  float noiseMagn;
  float outVal;
  BOOL *dependA;
} ShapeData;

typedef struct {
  int combNo;
  int validNo;
  ShapeData *shapeA;
  float *dataA;
  float *noiseA;
  float minMatch;
} CombData;

typedef struct {
  CombData *combDataP;
  float *dataA;
  float *errA;
  float target;
  BOOL *switchA;
  BOOL *variableA;
} AmpCalcData;

void
AppDeconvolHint(AppShapeP shapePA[], float prob)
{
}

static void
gatherShapes(AppShapeP shapeP, void *clientData)
{
  ShapeLists *listsP = clientData;
  int dom;

  dom = shapeP->domain;
  if (shapeP->centerPosition <
      listsP->regP->d.rangeA[dom][0] + listsP->regP->d.subRangeA[dom][0] ||
      shapeP->centerPosition >
      listsP->regP->d.rangeA[dom][0] + listsP->regP->d.subRangeA[dom][1])
    return;
  
  (void) ListInsertLast(listsP->shapeLA[dom], &shapeP);
}

static int
getShapeLists(AppRegionP regP, LINLIST *shapeLA)
{
  int dim, combNo;
  ShapeLists lists;
  int i;

  dim = regP->d.dimensionNo;

  for (i = 0; i < dim; i++)
    shapeLA[i] = ListOpen(sizeof(AppShapeP));

  lists.regP = regP;
  lists.dim = dim;
  lists.shapeLA = shapeLA;

  AppShapeApply(regP->specP, gatherShapes, &lists);

  combNo = 1;
  for (i = 0; i < dim; i++)
    combNo *= ListSize(shapeLA[i]);

  return combNo;
}

static void
freeShapeLists(AppRegionP regP, LINLIST *shapeLA)
{
  int dim;
  ShapeLists lists;
  int i;

  dim = regP->d.dimensionNo;

  for (i = 0; i < dim; i++)
    ListClose(shapeLA[i]);
}

static void
getFirstShape(LINLIST *shapeLA, int dim, AppShapeP **shapePPA, int *indP)
{
  int dom;

  for (dom = 0; dom < dim; dom++)
    shapePPA[dom] = ListFirst(shapeLA[dom]);

  *indP = 0;
}

static BOOL
getNextShape(LINLIST *shapeLA, int dim, AppShapeP **shapePPA, int *indP)
{
  int dom;

  for (dom = 0; dom < dim; dom++) {
    shapePPA[dom] = ListNext(shapeLA[dom], shapePPA[dom]);
    if (shapePPA[dom] == NULL)
      shapePPA[dom] = ListFirst(shapeLA[dom]);
    else
      break;
  }

  if (dom == dim)
    return TRUE;

  (*indP)++;

  return FALSE;
}

static void
getFirstCoord(int *sizeA, AppDataRange *subRangeA, int dim,
    int *coord, int *indP)
{
  int dom, ind;

  for (dom = 0; dom < dim; dom++)
    coord[dom] = subRangeA[dom][0];

  ind = 0;
  for (dom = dim - 1; dom >= 0; dom--)
    ind = ind * sizeA[dom] + coord[dom];

  *indP = ind;
}

static BOOL
getNextCoord(int *sizeA, AppDataRange *subRangeA, int dim,
    int *coord, int *indP)
{
  int dom, ind;

  for (dom = 0; dom < dim; dom++)
    if (coord[dom] == subRangeA[dom][1]) {
      coord[dom] = subRangeA[dom][0];
    } else {
      coord[dom]++;
      break;
    }

  if (dom == dim)
    return TRUE;

  ind = 0;
  for (dom = dim - 1; dom >= 0; dom--)
    ind = ind * sizeA[dom] + coord[dom];

  *indP = ind;

  return FALSE;
}

static int
compQueueEntry(void *p1, void *p2)
{
  QueueEntry *entry1P = p1;
  QueueEntry *entry2P = p2;

  if (entry1P->val > entry2P->val)
    return -1;

  if (entry1P->val < entry2P->val)
    return 1;
  
  return 0;
}

static void
extendRange(AppDataRange *rangeA, int dim, int *coord)
{
  int dom;

  for (dom = 0; dom < dim; dom++) {
    if (coord[dom] < rangeA[dom][0])
      rangeA[dom][0] = coord[dom];
    if (coord[dom] > rangeA[dom][1])
      rangeA[dom][1] = coord[dom];
  }
}

static void
prepareComb(AppRegionP regP, float *noiseA,
    LINLIST *shapeLA, float minShapeVal, BOOL *insideA,
    CombData *combDataP)
{
  int dim;
  int sizeA[APP_MAX_DIM], offsA[APP_MAX_DIM], coord[APP_MAX_DIM];
  AppDataRange (*rangeAA)[APP_MAX_DIM];
  int pointI;
  int allCombNo, combNo, allCombI, combI, combI1, combI2;
  int validNo, validI;
  AppShapeP *shapePPA[APP_MAX_DIM], *shapePP;
  float maxVal, val, err, errSum, prod, minSum, sum;
  float *dataA;
  int i;

  dim = regP->d.dimensionNo;
  dataA = ConvGetRegionFloat((AppRegionDataP) regP);

  allCombNo = 1;
  for (i = 0; i < dim; i++) {
    allCombNo *= ListSize(shapeLA[i]);
    sizeA[i] = regP->d.rangeA[i][1] - regP->d.rangeA[i][0] + 1;
  }

  combNo = 0;
  for (allCombI = 0; allCombI < allCombNo; allCombI++)
    if (insideA[allCombI])
      combNo++;

  validNo = 0;
  maxVal = 0.0;
  getFirstCoord(sizeA, regP->d.subRangeA, dim, coord, &pointI);
  for (;;) {
    if (regP->d.validA[pointI]) {
      validNo++;
      if (dataA[pointI] > maxVal)
	maxVal = dataA[pointI];
    }

    if (getNextCoord(sizeA, regP->d.subRangeA, dim, coord, &pointI))
      break;
  }

  rangeAA = malloc(combNo * sizeof(*rangeAA));

  combDataP->shapeA = malloc(combNo * sizeof(*combDataP->shapeA));
  for (combI = 0; combI < combNo; combI++) {
    combDataP->shapeA[combI].valA = malloc(validNo * sizeof(float));
    combDataP->shapeA[combI].errA = malloc(validNo * sizeof(float));
    combDataP->shapeA[combI].dependA = malloc(combNo * sizeof(BOOL));
  }

  getFirstShape(shapeLA, dim, shapePPA, &allCombI);
  combI = 0;
  for (;;) {   /* loop over all shape combinations */
    if (! insideA[allCombI]) {
      if (getNextShape(shapeLA, dim, shapePPA, &allCombI))
	break;
      continue;
    }

    for (i = 0; i < dim; i++)
      combDataP->shapeA[combI].shapePA[i] = *shapePPA[i];

    for (i = 0; i < dim; i++) {
      rangeAA[combI][i][0] = (*shapePPA[i])->startIndex;
      rangeAA[combI][i][1] =
	  (*shapePPA[i])->startIndex + (*shapePPA[i])->valueNo - 1;
      offsA[i] = (*shapePPA[i])->startIndex - regP->d.rangeA[i][0];
      coord[i] = 0;
    }

    combDataP->shapeA[combI].integr = 0.0;

    getFirstCoord(sizeA, regP->d.subRangeA, dim, coord, &pointI);
    validI = 0;
    for (;;) {   /* loop over all points */
      if (regP->d.validA[pointI]) {
	prod = 1.0;
	for (i = 0; i < dim; i++) {
	  if (coord[i] < offsA[i] ||
	      coord[i] >= (*shapePPA[i])->valueNo + offsA[i])
	    break;
	  prod *= (*shapePPA[i])->valueA[coord[i] - offsA[i]];
	}
	if (i < dim || prod < minShapeVal) {
	  prod = 0.0;
	  errSum = 0.0;
	} else if (prod == 0.0) {
	  errSum = 0.0;
	} else {
	  errSum = 0.0;
	  for (i = 0; i < dim; i++) {
	    val = (*shapePPA[i])->valueA[coord[i] - offsA[i]];
	    err = (prod / val) * (*shapePPA[i])->errorA[coord[i] - offsA[i]];
	    errSum += err * err;
	  }
	  errSum = sqrt(errSum);
	}

	if (errSum > 0.9 * prod)
	  /* can happen if error calculation produces underflow */
	  errSum = 0.9 * prod;

	combDataP->shapeA[combI].valA[validI] = prod;
	combDataP->shapeA[combI].errA[validI] = errSum;
	combDataP->shapeA[combI].integr += prod;

	validI++;
      }

      if (getNextCoord(sizeA, regP->d.subRangeA, dim, coord, &pointI))
	break;
    }

    if (getNextShape(shapeLA, dim, shapePPA, &allCombI))
      break;
    
    combI++;
  }
       
  for (combI1 = 0; combI1 < combNo; combI1++) {
    combDataP->shapeA[combI1].dependA[combI1] = TRUE;

    for (combI2 = combI1 + 1; combI2 < combNo; combI2++) {
      for (i = 0; i < dim; i++)
	if (rangeAA[combI1][i][0] > rangeAA[combI2][i][1] ||
	    rangeAA[combI1][i][1] < rangeAA[combI2][i][0])
	  break;

      if (i < dim) {
	combDataP->shapeA[combI1].dependA[combI2] = FALSE;
      } else {
	minSum = combDataP->shapeA[combI1].integr;
	if (combDataP->shapeA[combI2].integr < minSum)
	  combDataP->shapeA[combI2].integr;
	minSum *= 0.0001;

	sum = 0.0;
	for (validI = 0; validI < validNo; validI++) {
	  if (combDataP->shapeA[combI1].valA[validI] <
	      combDataP->shapeA[combI2].valA[validI])
	    sum += combDataP->shapeA[combI1].valA[validI];
	  else
	    sum += combDataP->shapeA[combI2].valA[validI];
	  if (sum > minSum)
	    break;
	}

	combDataP->shapeA[combI1].dependA[combI2] = (validI < validNo);
      }

      combDataP->shapeA[combI2].dependA[combI1] = 
	  combDataP->shapeA[combI1].dependA[combI2];
    }
  }

  free(rangeAA);

  combDataP->dataA = malloc(validNo * sizeof(float));
  combDataP->noiseA = malloc(validNo * sizeof(float));

  validI = 0;
  getFirstCoord(sizeA, regP->d.subRangeA, dim, coord, &pointI);
  for (;;) {
    if (regP->d.validA[pointI]) {
      combDataP->dataA[validI] = dataA[pointI];
      combDataP->noiseA[validI] = noiseA[pointI];
      validI++;
    }

    if (getNextCoord(sizeA, regP->d.subRangeA, dim, coord, &pointI))
      break;
  }

  combDataP->combNo = combNo;
  combDataP->validNo = validNo;
}

static void
freeComb(CombData *combDataP)
{
  int combI;

  for (combI = 0; combI < combDataP->combNo; combI++) {
    free(combDataP->shapeA[combI].valA);
    free(combDataP->shapeA[combI].errA);
    free(combDataP->shapeA[combI].dependA);
  }
  free(combDataP->shapeA);

  free(combDataP->dataA);
  free(combDataP->noiseA);
}

static void
getPeak(AppPeakP peakP, void *clientData)
{
  PeakData *dataP = clientData;
  int coord[APP_MAX_DIM];
  int allCombI, combSize, combI;
  int shapeI;
  AppShapeP *shapePP, **sPPP;
  int dom, i;

  for (dom = 0; dom < dataP->dim; dom++)
    coord[dom] = (int) (peakP->positionA[dom] + 0.5);
  
  if (! AppRegionIsInside((AppRegionDataP) dataP->regP, coord))
    return;

  allCombI = 0;
  combSize = 1;

  for (dom = 0; dom < dataP->dim; dom++) {
    shapePP = ListFirst(dataP->shapeLA[dom]);
    shapeI = 0;

    while (shapePP != NULL) {
      sPPP = ListFirst((*shapePP)->pointerList);
      while (sPPP != NULL) {
	if (*sPPP == peakP->shapePA + dom)
	  break;
	sPPP = ListNext((*shapePP)->pointerList, sPPP);
      }

      if (sPPP != NULL)
	break;

      shapePP = ListNext(dataP->shapeLA[dom], shapePP);
      shapeI++;
    }

    if (shapePP == NULL)
      break;

    allCombI += shapeI * combSize;
    combSize *= ListSize(dataP->shapeLA[dom]);
  }

  if (dom < dataP->dim || ! dataP->insideA[allCombI])
    return;

  combI = 0;
  for (i = 0; i < allCombI; i++)
    if (dataP->insideA[i])
      combI++;

  dataP->combA[combI] = CS_ALREADY;
}

static void
getAlready(AppRegionP regP, LINLIST *shapeLA, BOOL *insideA, CombState *combA)
{
  PeakData data;

  data.regP = regP;
  data.dim = regP->d.dimensionNo;
  data.shapeLA = shapeLA;
  data.insideA = insideA;
  data.combA = combA;

  AppPeakApply(regP->specP, getPeak, &data);
}

static BOOL
fitTarget(float solA[], int switchNo, void *clientData,
    float *fP, float df[])
{
  AmpCalcData *calcDataP = clientData;
  CombData *combDataP = calcDataP->combDataP;
  int validNo, validI;
  int combNo, combI;
  int switchI;
  float f, amp, res, err, fact, d, e;

  validNo = combDataP->validNo;
  combNo = combDataP->combNo;

  f = 0.0;
  switchI = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! calcDataP->switchA[combI])
      continue;

    df[switchI] = 0.0;

    if (solA[switchI] >= 0.0) {
      switchI++;
      continue;
    }

    /* penalty for negative amplitude */
    fact = 10.0 / combDataP->shapeA[combI].noiseMagn;
    fact = fact * fact * fact * fact;
    f += fact *
	solA[switchI] * solA[switchI] * solA[switchI] * solA[switchI];
    df[switchI] += fact * 4.0 * solA[switchI] * solA[switchI] * solA[switchI];

    switchI++;
  }

  for (validI = 0; validI < validNo; validI++) {
    res = 0.0;
    err = combDataP->noiseA[validI] * combDataP->noiseA[validI];
    if (calcDataP->errA != NULL)
      err += calcDataP->errA[validI];
    switchI = 0;
    for (combI = 0; combI < combNo; combI++) {
      if (! calcDataP->switchA[combI])
	continue;

      if (solA[switchI] <= 0.0) {
	switchI++;
	continue;
      }

      if (combDataP->shapeA[combI].valA[validI] > 0.0) {
	res += solA[switchI] * combDataP->shapeA[combI].valA[validI];
	d = solA[switchI] * combDataP->shapeA[combI].errA[validI];
	err += d * d;
      }

      switchI++;
    }
    res -= calcDataP->dataA[validI];
    err = sqrt(err);
    res /= err;

    if (res > 20.0)  /* avoid overflow */
      res = 20.0;

    e = exp(res);
    f += e - res - 1.0;
    switchI = 0;
    for (combI = 0; combI < combNo; combI++) {
      if (! calcDataP->switchA[combI])
	continue;

      if (combDataP->shapeA[combI].valA[validI] <= 0.0) {
	switchI++;
	continue;
      }

      if (solA[switchI] <= 0.0)
	df[switchI] += (e - 1.0) *
	    (combDataP->shapeA[combI].valA[validI] / err);
      else
	df[switchI] += (e - 1.0) *
	    (combDataP->shapeA[combI].valA[validI] / err -
	     res * combDataP->shapeA[combI].errA[validI] *
	     combDataP->shapeA[combI].errA[validI] * solA[switchI] /
	     (err * err));

      switchI++;
    }
  }

  *fP = f;

  return *fP < calcDataP->target;
}

static float
calcFull(CombData *combDataP, BOOL *switchA, float *ampA)
{
  int validNo, validI;
  int combNo, combI;
  int switchNo, switchI;
  float integrSum, dataSum, ampFact;
  float *solA;
  AmpCalcData calcData;
  float a, f, *df;

  validNo = combDataP->validNo;
  combNo = combDataP->combNo;

  integrSum = 0.0;
  switchNo = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! switchA[combI]) {
      ampA[combI] = 0.0;
      continue;
    }

    ampA[combI] = MAXFLOAT;

    for (validI = 0; validI < validNo; validI++) {
      if (combDataP->dataA[validI] < 0.0)
	continue;
      
      a = (combDataP->dataA[validI] + combDataP->noiseA[validI]) /
	  combDataP->shapeA[combI].valA[validI];
      if (a < ampA[combI])
	ampA[combI] = a;
    }

    integrSum += ampA[combI] * combDataP->shapeA[combI].integr;

    switchNo++;
  }

  dataSum = 0.0;
  for (validI = 0; validI < validNo; validI++)
    dataSum += combDataP->dataA[validI];

  /* scale estimated amplitudes so that sum of integrals is equal
     to sum of data points */
  ampFact = dataSum / integrSum;

  solA = malloc(switchNo * sizeof(*solA));
  switchI = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! switchA[combI])
      continue;

    solA[switchI] = ampFact * ampA[combI];
    switchI++;
  }

  calcData.combDataP = combDataP;
  calcData.dataA = combDataP->dataA;
  calcData.errA = NULL;
  calcData.target = 0.0;
  calcData.switchA = switchA;

  (void) ConjGradMin(fitTarget, solA, switchNo, &calcData, 0.0001);

  switchI = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! switchA[combI])
      continue;

    if (solA[switchI] < 0.0) {
      solA[switchI] = 0.0;
      ampA[combI] = 0.0;
    } else {
      ampA[combI] = solA[switchI];
    }

    switchI++;
  }
      
  df = malloc(switchNo * sizeof(*df));
  fitTarget(solA, switchNo, &calcData, &f, df);
  free(df);

  free(solA);

  return f;
}

static float
calcAmp(CombData *combDataP, float target, BOOL *switchA, BOOL *variableA,
    float *ampA)
{
  int validNo, validI;
  int combNo, combI;
  int switchNo, switchI;
  BOOL *varSwitchA;
  float *varDataA, *varErrA;
  float *solA;
  AmpCalcData calcData;
  BOOL ok;
  float d, f, *df;

  validNo = combDataP->validNo;
  combNo = combDataP->combNo;

  varSwitchA = malloc(combNo * sizeof(*varSwitchA));
  varDataA = malloc(validNo * sizeof(*varDataA));
  varErrA = malloc(validNo * sizeof(*varErrA));

  for (validI = 0; validI < validNo; validI++) {
    varDataA[validI] = combDataP->dataA[validI];
    varErrA[validI] = 0.0;
  }

  switchNo = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! switchA[combI]) {
      ampA[combI] = 0.0;
      varSwitchA[combI] = FALSE;
      continue;
    }

    if (! variableA[combI]) {
      for (validI = 0; validI < validNo; validI++) {
	varDataA[validI] -=
	    ampA[combI] * combDataP->shapeA[combI].valA[validI];
	d = ampA[combI] * combDataP->shapeA[combI].errA[validI];
	varErrA[validI] += d * d;
      }
      varSwitchA[combI] = FALSE;
      continue;
    }

    varSwitchA[combI] = TRUE;
    switchNo++;
  }

  solA = malloc(switchNo * sizeof(*solA));
  switchI = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! varSwitchA[combI])
      continue;

    solA[switchI] = ampA[combI];
    switchI++;
  }

  calcData.combDataP = combDataP;
  calcData.dataA = varDataA;
  calcData.errA = varErrA;
  calcData.target = target;
  calcData.switchA = varSwitchA;

  (void) ConjGradMin(fitTarget, solA, switchNo, &calcData, 0.0001);
  
  switchI = 0;
  for (combI = 0; combI < combNo; combI++) {
    if (! varSwitchA[combI])
      continue;

    if (solA[switchI] < 0.0) {
      solA[switchI] = 0.0;
      ampA[combI] = 0.0;
    } else {
      ampA[combI] = solA[switchI];
    }

    switchI++;
  }

  df = malloc(switchNo * sizeof(*df));
  fitTarget(solA, switchNo, &calcData, &f, df);
  free(df);

  free(varSwitchA);
  free(varDataA);
  free(varErrA);

  free(solA);

  return f;
}

static void
deconvolSubreg(AppRegionP regP, float minLevel,
    CombData *combDataP, CombState *combA)
{
  int dim, maybeNo;
  int combNo, combI, switchNo;
  int validNo, validI;
  float *ampA, err;
  BOOL *switchA;
  float *qualA;
  float *restA, *errA, bestMatch, match;
  float v1Sum, v2Sum, v12Sum, v1, v2;
  float v1Err, v2Err, v12Err, den, denErr, e1, e2;
  int bestCombI;
  struct AppPeakS peakS;
  AppPeakP peakP;
  AppShapeP *shapePP;
  int i;

  dim = regP->d.dimensionNo;

  combNo = combDataP->combNo;
  validNo = combDataP->validNo;

  ampA = malloc(combNo * sizeof(*ampA));
  switchA = malloc(combNo * sizeof(*switchA));

  maybeNo = 0;
  for (combI = 0; combI < combNo; combI++)
    if (combA[combI] == CS_MAYBE)
      maybeNo++;

  /* calculate amplitudes for the already found peaks */
  switchNo = 0;
  for (combI = 0; combI < combNo; combI++) {
    switchA[combI] = (combA[combI] == CS_ALREADY);
    if (switchA[combI])
      switchNo++;
    else
      ampA[combI] = 0.0;
  }
  if (switchNo > 0)
    (void) calcFull(combDataP, switchA, ampA);

  qualA = malloc(combNo * sizeof(*qualA));
  restA = malloc(validNo * sizeof(*restA));
  errA = malloc(validNo * sizeof(*errA));

  for (;;) {
    if (maybeNo == 0)
      break;

    for (validI = 0; validI < validNo; validI++) {
      restA[validI] = combDataP->dataA[validI];
      errA[validI] = combDataP->noiseA[validI] * combDataP->noiseA[validI];

      for (combI = 0; combI < combNo; combI++) {
	if (combA[combI] != CS_ALREADY && combA[combI] != CS_NEW)
	  continue;
	
	restA[validI] -= ampA[combI] * combDataP->shapeA[combI].valA[validI];
	err = ampA[combI] * combDataP->shapeA[combI].errA[validI];
	errA[validI] += err * err;
      }

      errA[validI] = sqrt(errA[validI]);
    }

    bestMatch = -1.0;
    for (combI = 0; combI < combNo; combI++) {
      if (combA[combI] != CS_MAYBE)
	continue;
	
      v1Sum = 0.0;
      v1Err = 0.0;
      v2Sum = combDataP->shapeA[combI].outVal;
      v2Err = 0.0;
      v12Sum = 0.0;
      v12Err = 0.0;

      for (validI = 0; validI < validNo; validI++) {
	v2 = combDataP->shapeA[combI].valA[validI];
	if (v2 == 0.0)
	  continue;

	v1 = restA[validI];
	e1 = errA[validI];
	e2 = combDataP->shapeA[combI].errA[validI];

	v1Sum += v1 * v1;
	v2Sum += v2 * v2;
	v12Sum += v1 * v2;

	if (v1 < 0.0)
	  v1 = - v1;

	v1Err += (2.0 * v1 + e1) * e1;
	v2Err += (2.0 * v2 + e2) * e2;
	v12Err += v1 * e2 + v2 * e1 + e1 * e2;
      }

      if (v12Sum < 0.0)
	continue;

      den = sqrt(v1Sum * v2Sum);
      match = v12Sum / den;

      v1Err = sqrt(v1Err);
      v2Err = sqrt(v2Err);
      v12Err = sqrt(v12Err);

      denErr = sqrt(v1Sum * v2Err + v2Sum * v1Err + v1Err * v2Err);
      denErr /= 2.0 * sqrt(den);
      denErr /= (den * den);
      err = sqrt(v12Sum * denErr + v12Err / den + v12Err * denErr);

      match -= err;

      if (match > bestMatch) {
	bestMatch = match;
	bestCombI = combI;
      }
    }

    if (bestMatch < combDataP->minMatch)
      break;

    combA[bestCombI] = CS_NEW;
    qualA[bestCombI] = (bestMatch - combDataP->minMatch) /
	(1.0 - combDataP->minMatch);

    for (combI = 0; combI < combNo; combI++)
      switchA[combI] =
	  (combA[combI] == CS_ALREADY || combA[combI] == CS_NEW);
    (void) calcAmp(combDataP, 0.0,
	switchA, combDataP->shapeA[bestCombI].dependA,
	ampA);

    maybeNo--;
  }

  for (combI = 0; combI < combNo; combI++) {
    if (! (combA[combI] == CS_NEW &&
	ampA[combI] > minLevel * combDataP->shapeA[combI].noiseMagn))
      continue;

    for (i = 0; i < dim; i++) {
      peakS.positionA[i] = combDataP->shapeA[combI].shapePA[i]->centerPosition;
      peakS.symmetryErrorA[i] = 0.0;
    }
    peakS.amplitude = ampA[combI];
    peakS.uniformError = 0.0;
    peakS.integrationLevel =
	0.5 * combDataP->shapeA[combI].noiseMagn / peakS.amplitude;
    peakS.quality = qualA[combI];

    peakP = AppAddPeak(regP->specP, &peakS);
    for (i = 0; i < dim; i++) {
      peakP->shapePA[i] = combDataP->shapeA[combI].shapePA[i];
      shapePP = peakP->shapePA + i;
      (void) ListInsertLast(peakP->shapePA[i]->pointerList, &shapePP);
    }
  }

  free(qualA);
  free(restA);
  free(errA);
  free(ampA);
  free(switchA);
}

void
AppDeconvol(AppRegionP regP, float minLevel, float minMatch)
{
  int dim;
  LINLIST shapeLA[APP_MAX_DIM];
  int combNo, combI;
  int insideNo, insideI;
  BOOL *insideA;
  int maxNo, maxI;
  int sizeA[APP_MAX_DIM], totSize, incA[APP_MAX_DIM], coord[APP_MAX_DIM];
  int *pointA;
  MaxData *maxA;
  AppMaxP maxP;
  float *dataA, *noiseA;
  QUEUE queue;
  QueueEntry qEntry;
  LINLIST neighList;
  NeighListEntry nEntry, *nEntryP;
  int *shapeSegA;
  int seg1I, seg2I, segI;
  AppShapeP *shapePPA[APP_MAX_DIM], *shapePP;
  float posA[APP_MAX_DIM];
  int offsA[APP_MAX_DIM];
  float *noiseMagnA, *outValA;
  float maxVal, minShapeVal, prod;
  CombData combData;
  CombState *combA;
  int dom, ind, nInd, i;

  dim = regP->d.dimensionNo;

  maxNo = ListSize(regP->maxL);
  if (maxNo == 0)
    return;

  combNo = getShapeLists(regP, shapeLA);
  if (combNo == 0) {
    freeShapeLists(regP, shapeLA);
    return;
  }

  totSize = 1;
  for (dom = 0; dom < dim; dom++) {
    sizeA[dom] = regP->d.rangeA[dom][1] - regP->d.rangeA[dom][0] + 1;
    totSize *= sizeA[dom];
  }

  pointA = malloc(totSize * sizeof(*pointA));
  for (ind = 0; ind < totSize; ind++)
    if (regP->d.validA[ind])
      pointA[ind] = PS_UNASSIGNED;
    else
      pointA[ind] = PS_OUTSIDE;

  dataA = ConvGetRegionFloat((AppRegionDataP) regP);
  noiseA = RegionGetNoise(regP);

  maxA = malloc(maxNo * sizeof(*maxA));

  queue = QueueOpen(sizeof(QueueEntry), compQueueEntry);

  maxP = ListFirst(regP->maxL);
  for (maxI = 0; maxI < maxNo; maxI++) {
    maxA[maxI].segI = maxI;
    maxA[maxI].insideNo = 0;

    for (dom = 0; dom < dim; dom++) {
      maxA[maxI].subRangeA[dom][0] = sizeA[dom];
      maxA[maxI].subRangeA[dom][1] = -1;
    }

    ind = 0;
    for (dom = dim - 1; dom >= 0; dom--)
      ind = ind * sizeA[dom] + maxP->coordinates[dom] - regP->d.rangeA[dom][0];

    qEntry.ind = ind;
    qEntry.val = dataA[ind];
    qEntry.segI = maxI;
    QueuePut(queue, &qEntry);

    pointA[ind] = PS_QUEUED;

    maxP = ListNext(regP->maxL, maxP);
  }

  incA[0] = 1;
  for (dom = 1; dom < dim; dom++)
    incA[dom] = incA[dom - 1] * sizeA[dom - 1];

  neighList = ListOpen(sizeof(NeighListEntry));

  while (QueueSize(queue) > 0) {
    QueueGet(queue, &qEntry);
    ind = qEntry.ind;
    segI = qEntry.segI;

    pointA[ind] = segI;

    for (dom = 0; dom < dim; dom++) {
      nInd = ind - incA[dom];
      if (pointA[nInd] == PS_UNASSIGNED) {
	qEntry.ind = nInd;
	qEntry.val = dataA[nInd];
	QueuePut(queue, &qEntry);

	pointA[nInd] = PS_QUEUED;
      } else if (pointA[nInd] >= 0 && pointA[nInd] != segI) {
	nEntry.seg1I = segI;
	nEntry.seg2I = pointA[nInd];
	(void) ListInsertLast(neighList, &nEntry);
      }

      nInd = ind + incA[dom];
      if (pointA[nInd] == PS_UNASSIGNED) {
	qEntry.ind = nInd;
	qEntry.val = dataA[nInd];
	QueuePut(queue, &qEntry);

	pointA[nInd] = PS_QUEUED;
      } else if (pointA[nInd] >= 0 && pointA[nInd] != segI) {
	nEntry.seg1I = segI;
	nEntry.seg2I = pointA[nInd];
	(void) ListInsertLast(neighList, &nEntry);
      }
    }
  }

  QueueClose(queue);

  shapeSegA = malloc(combNo * sizeof(*shapeSegA));
  insideA = malloc(combNo * sizeof(*insideA));

  getFirstShape(shapeLA, dim, shapePPA, &combI);
  for (;;) {  /* loop over all shape combinations */
    ind = 0;
    for (dom = dim - 1; dom >= 0; dom--)
      ind = ind * sizeA[dom] +
	  (int) ((*shapePPA[dom])->centerPosition + 0.5) -
	  regP->d.rangeA[dom][0];

    if (pointA[ind] < 0) {
      shapeSegA[combI] = -1;
    } else {
      shapeSegA[combI] = pointA[ind];
      maxA[pointA[ind]].insideNo++;
    }

    if (getNextShape(shapeLA, dim, shapePPA, &combI))
      break;
  }

  nEntryP = ListFirst(neighList);
  while (nEntryP != NULL) {
    seg1I = nEntryP->seg1I;
    while (maxA[seg1I].segI < seg1I)
      seg1I = maxA[seg1I].segI;

    seg2I = nEntryP->seg2I;
    while (maxA[seg2I].segI < seg2I)
      seg2I = maxA[seg2I].segI;

    if (seg1I != seg2I &&
	maxA[seg1I].insideNo + maxA[seg2I].insideNo <= MAX_NO) {
      if (seg1I < seg2I) {
	maxA[seg2I].segI = seg1I;
	maxA[seg1I].insideNo += maxA[seg2I].insideNo;
      } else {
	maxA[seg1I].segI = seg2I;
	maxA[seg2I].insideNo += maxA[seg1I].insideNo;
      }
    }

    nEntryP = ListNext(neighList, nEntryP);
  }

  ListClose(neighList);

  getFirstCoord(sizeA, regP->d.subRangeA, dim, coord, &ind);
  for (;;) {
    if (pointA[ind] >= 0) {
      segI = pointA[ind];
      while (maxA[segI].segI < segI)
	segI = maxA[segI].segI;

      pointA[ind] = segI;
      extendRange(maxA[segI].subRangeA, dim, coord);
    }

    if (getNextCoord(sizeA, regP->d.subRangeA, dim, coord, &ind))
      break;
  }

  minLevel *= regP->segLevel;

  for (maxI = 0; maxI < maxNo; maxI++) {
    segI = maxA[maxI].segI;
    if (maxI > segI)
      continue;

    maxVal = 0.0;
    getFirstCoord(sizeA, maxA[segI].subRangeA, dim, coord, &ind);
    for (;;) {
      if (pointA[ind] == segI && dataA[ind] > maxVal)
	  maxVal = dataA[ind];

      if (getNextCoord(sizeA, maxA[segI].subRangeA, dim, coord, &ind))
	break;
    }

    minShapeVal = 0.2 * regP->d.baseLevel / maxVal;
    for (dom = 0; dom < dim; dom++) {
      shapePP = ListFirst(shapeLA[dom]);
      while (shapePP != NULL) {
	AppShapeComplete(*shapePP,
	    regP->d.rangeA[dom][0], regP->d.rangeA[dom][1], minShapeVal);
	shapePP = ListNext(shapeLA[dom], shapePP);
      }
    }

    for (combI = 0; combI < combNo; combI++) {
      if (shapeSegA[combI] < 0) {
	insideA[combI] = FALSE;
	continue;
      }

      i = shapeSegA[combI];
      while (maxA[i].segI < i)
	i = maxA[i].segI;

      insideA[combI] = (i == segI);
    }

    insideNo = maxA[segI].insideNo;

    noiseMagnA = malloc(combNo * sizeof(*noiseMagnA));

    /* detect and disable shape combinations where there is no intensity */
    getFirstShape(shapeLA, dim, shapePPA, &combI);
    for (;;) {  /* loop over all shape combinations */
      if (! insideA[combI]) {
	if (getNextShape(shapeLA, dim, shapePPA, &combI))
	  break;
	continue;
      }

      for (dom = 0; dom < dim; dom++) {
	posA[dom] = (*shapePPA[dom])->centerPosition;
	offsA[dom] = (*shapePPA[dom])->startIndex - regP->d.rangeA[dom][0];
	coord[dom] = offsA[dom];
      }

      noiseMagnA[combI] = AppNoiseGetMagnitudeFloat(regP->noiseP, posA);

      for (;;) {  /* loop over all points of one shape combination */
	ind = 0;
	prod = 1.0;
	for (dom = dim - 1; dom >= 0; dom--) {
	  if (coord[dom] < 1 || coord[dom] > sizeA[dom] - 2)
	    break;
	  ind = ind * sizeA[dom] + coord[dom];
	  prod *= (*shapePPA[dom])->valueA[coord[dom] - offsA[dom]];
	}

	if (dom < 0 && pointA[ind] != PS_OUTSIDE && prod > minShapeVal &&
	    minLevel * noiseMagnA[combI] * prod > dataA[ind] + noiseA[ind]) {
	  insideA[combI] = FALSE;
	  insideNo--;
	  break;
	}

	for (dom = 0; dom < dim; dom++)
	  if (coord[dom] == offsA[dom] + (*shapePPA[dom])->valueNo - 1) {
	    coord[dom] = offsA[dom];
	  } else {
	    coord[dom]++;
	    break;
	  }

	if (dom == dim)
	  break;
      }

      if (getNextShape(shapeLA, dim, shapePPA, &combI))
	break;
    }

    outValA = malloc(combNo * sizeof(*outValA));

    /* set validA of region so that only points that are members of
       a shape are used */
    AppRegionPushState(regP, APP_PUSH_VALID);
    (void) memset(regP->d.validA, 0, totSize * sizeof(*regP->d.validA));

    for (dom = 0; dom < dim; dom++) {
      regP->d.subRangeA[dom][0] = maxA[segI].subRangeA[dom][0];
      regP->d.subRangeA[dom][1] = maxA[segI].subRangeA[dom][1];
    }

    getFirstShape(shapeLA, dim, shapePPA, &combI);
    for (;;) {  /* loop over all shape combinations */
      if (! insideA[combI]) {
	if (getNextShape(shapeLA, dim, shapePPA, &combI))
	  break;
	continue;
      }

      for (dom = 0; dom < dim; dom++) {
	offsA[dom] = (*shapePPA[dom])->startIndex - regP->d.rangeA[dom][0];
	coord[dom] = offsA[dom];
      }

      outValA[combI] = 0.0;
      
      for (;;) {  /* loop over all points of one shape combination */
	prod = 1.0;
	for (dom = 0; dom < dim; dom++)
	  prod *= (*shapePPA[dom])->valueA[coord[dom] - offsA[dom]];

	ind = 0;
	for (dom = dim - 1; dom >= 0; dom--) {
	  if (coord[dom] < 1 || coord[dom] > sizeA[dom] - 2)
	    break;
	  ind = ind * sizeA[dom] + coord[dom];
	}

	if (dom < 0 && pointA[ind] != PS_OUTSIDE && prod > minShapeVal) {
	  regP->d.validA[ind] = TRUE;
	  extendRange(regP->d.subRangeA, dim, coord);
	} else {
	  outValA[combI] += prod * prod;
	}

	for (dom = 0; dom < dim; dom++)
	  if (coord[dom] == offsA[dom] + (*shapePPA[dom])->valueNo - 1) {
	    coord[dom] = offsA[dom];
	  } else {
	    coord[dom]++;
	    break;
	  }

	if (dom == dim)
	  break;
      }

      if (getNextShape(shapeLA, dim, shapePPA, &combI))
	break;
    }

    if (insideNo > 0) {
      combData.minMatch = minMatch;
      prepareComb(regP, noiseA, shapeLA, minShapeVal, insideA, &combData);

      insideI = 0;
      for (combI = 0; combI < combNo; combI++) {
	if (! insideA[combI])
	  continue;

	combData.shapeA[insideI].noiseMagn = noiseMagnA[combI];
	combData.shapeA[insideI].outVal = outValA[combI];
	insideI++;
      }

      combA = malloc(insideNo * sizeof(*combA));
      for (combI = 0; combI < insideNo; combI++)
	combA[combI] = CS_MAYBE;
      getAlready(regP, shapeLA, insideA, combA);

      deconvolSubreg(regP, minLevel, &combData, combA);

      free(combA);
      freeComb(&combData);
    }

    free(noiseMagnA);
    free(outValA);

    for (dom = 0; dom < dim; dom++) {
      shapePP = ListFirst(shapeLA[dom]);
      while (shapePP != NULL) {
	AppShapeOriginal(*shapePP);
	shapePP = ListNext(shapeLA[dom], shapePP);
      }
    }

    AppRegionPopState(regP);
  }

  free(shapeSegA);
  free(insideA);
  free(maxA);
  free(pointA);
  free(noiseA);

  freeShapeLists(regP, shapeLA);
}
