(* POMA - Product Operator Formalism in Mathematica ------------------------------------------------------------------*)

(* Version 2.0 including raising/lowering operators and gradients
   Copyright (c) 2002 by Peter Guntert. All rights reserved.
   
   References:
   Guntert et al., J. Magn. Reson. A101, 103-105 and A105, 328 (1993)
   Sorensen et al., Progress in NMR Spectroscopy 16, 163-192 (1983) (equation numbers refer to this paper)

   WWW: guentert.gsc.riken.jp
   Email: guentert@gsc.riken.jp
*)


BeginPackage["Poma`"]


(* Title -------------------------------------------------------------------------------------------------------------*)

Print[""]
Print["----------------------------------------------------"]
Print["POMA 2.0 - Product Operator Formalism in Mathematica"]
Print[""]
Print["Copyright (c) 2002 Peter Guntert"]
Print["All rights reserved."]
Print["----------------------------------------------------"]
Print[""]


(* Usage of commands -------------------------------------------------------------------------------------------------*)

commands::usage=
"\<
POMA 2.0 commands:

spin[n,k]                Spin operator (k = x, y, z, plus, minus)
j[n,m]                   Scalar coupling constant between spin n and m
g[n]                     Gyromagnetic ratio of spin n
pulse[angle,phase,spins] RF pulse
delay[time,couplings]    Delay with active scalar couplings
receiver[phase]          Receiver phase
gradient[strength]       Pulsed field gradient
dephase                  Discard magnetization dephased by gradients
raiselower[spins]        Transform to raising/lowering spin operators
cartesian[spins]         Transform to Cartesian spin operators
nucleus[n]               Set symbol used for spin n in the output
show[label]              Show current magnetization
store[var]               Store current magnetization in a variable
\>"

spin::usage=
"spin[n,x], spin[n,y], spin[n,z] are the Cartesian spin operators Ix, Iy, Iz 
for a spin n. spin[n,plus], spin[n,minus] are the raising/lowering operators 
I+ and I- for a spin n."

pulse::usage=
"pulse[angle,phase] is a pulse with the given angle and phase that acts on all 
spins. pulse[angle,phase,spins] is a pulse that acts only on the given list
of spins. If angle or phase are given as numbers, then they are interpreted
in degrees, otherwise in radians. phase can also be given as x, y, or z.
pulse[angle,{phase1,phase2,...},spins] implements phase cycling."

delay::usage=
"delay[t,couplings] is a delay of duration t during which scalar couplings
and chemical shift evolution are active. couplings is a list of pairs of 
scalar-coupled spins: couplings = {{m1,n1},{m2,n2},...}. delay[t] is a delay 
of duration t during which only chemical shift evolution is active. 
delay[t,couplings,spins] is a delay of duration t during which scalar couplings
are active and chemical shift evolution is active only for the given list
of spins."

receiver::usage="receiver[phase] sets the recveiver phase. A phase values
given as a number is interpreted in degrees, otherwise in radians.
receiver[{phase1,phase2,...}] implements receiver phase cycling.
receiver[phase,spins] sets the receiver phase only for the given spins."

gradient::usage="gradient[G] applies a pulsed-field gradient of strength G
along the z-axis."

dephase::usage="dephase discards all magnetization that has been dephased
(and never refocussed) by gradients."

observable::usage="observable retains only observable magnetization. 
observable[spins] retains observable magnetization only for the given spins."

raiselower::usage="raiselower transforms from Cartesian to raising/lowering basis
spin operators. raiselower[spins] transforms from Cartesian to raising/lowering basis
spin operators only for the given spins."

cartesian::usage="cartesian transforms from raising/lowering to Cartesian basis
spin operators. cartesian[spins] transforms from raising/lowering to Cartesian basis
spin operators only for the given spins."

nucleus::usage="nucleus[n]=\"X\" sets the symbol to represent spin n in the output
to X. By default, all spin operators are denoted by I in the output."

show::usage="show[label] prints the given label and the current magnetization.
The show command can be inserted at any point into a pulse sequence."

store::usage="store[var] stores the current magnetization in the variable var.
The store command can be inserted at any point into a pulse sequence to capture
intermediate results."

x::usage="x denotes the Cartesian x-dimension"
y::usage="y denotes the Cartesian y-dimension"
z::usage="z denotes the Cartesian z-dimension"
plus::usage="plus denotes a raising operator"
minus::usage="minus denotes a lowering operator"
allspins::usage="allspins denotes the set of all spins present in the pulse sequence"
j::usage="j[n,m] denotes the scalar coupling constant between spin n and m"
g::usage="g[n] denotes the gyromagnetic ratio of spin n"

license::usage="license prints the software license for POMA 2.0"


Begin["`Private`"]

(* Transform between Cartesian and raising/lowering operators --------------------------------------------------------*)

MemberQ[allspins,_] ^= True
raiselower[sigma_] := raiselower[allspins][sigma] /; !FreeQ[sigma,spin]
raiselower[spins_][sigma_] := sigma /. { 
	spin[n_,x] :> (spin[n,plus] + spin[n,minus])/2           /; MemberQ[spins,n],                       (* Eq. 73 *)
	spin[n_,y] :> (spin[n,plus] - spin[n,minus])/(2I)        /; MemberQ[spins,n],                       (* Eq. 74 *)
	Cos[a_. w_[n_]] :> (Exp[I a w[n]] + Exp[-I a w[n]])/2    /; MemberQ[spins,n] && MemberQ[{w,g},w],
	Sin[a_. w_[n_]] :> (Exp[I a w[n]] - Exp[-I a w[n]])/(2I) /; MemberQ[spins,n] && MemberQ[{w,g},w]} // 
        simplifysigma
cartesian[sigma_] := cartesian[allspins][sigma] /; !FreeQ[sigma,spin]
cartesian[spins_][sigma_] := sigma /. {
	spin[n_,plus]  :> spin[n,x] + I spin[n,y]       /; MemberQ[spins,n],
	spin[n_,minus] :> spin[n,x] - I spin[n,y]       /; MemberQ[spins,n]} //.
        Exp[Complex[0,i_] a_. w_[n_] + b_.] :> (Cos[i a w[n]] + I Sin[i a w[n]]) Exp[b] /; 
		MemberQ[spins,n] && MemberQ[{w,g},w] // 
        simplifysigma
thespins[sigma_] := Block[{s,sapp},s={}; 
	sapp[spin[n_,k_]] := (s=Append[s,n]);
	Scan[sapp,1+sigma,Infinity]; Union[s]]
raiselowerspins[sigma_] := Block[{s,sapp},s={}; 
	sapp[spin[n_,k_]] := (s=Append[s,n]) /; MemberQ[{plus,minus},k];
	Scan[sapp,1+sigma,Infinity]; Union[s]]
cartesianspins[sigma_] := Block[{s,sapp},s={}; 
	sapp[spin[n_,k_]] := (s=Append[s,n]) /; MemberQ[{x,y},k];
	Scan[sapp,1+sigma,Infinity]; Union[s]]


(* Delays ------------------------------------------------------------------------------------------------------------*)

delay[t_,couples_:{},spins_:allspins][sigma_] := Block[{},
	WriteString["stdout","delay[",InputForm[t],",",couples,",",spins,"]"]; 
	time[sigma // shift[t,spins] // coupling[t,couples] // simplifysigma]]                              (* Eq. 13 *)
                  

(* Chemical shift evolution ------------------------------------------------------------------------------------------*)
 
shift[t_,spins_:allspins][sigma_] := sigma /. { 
	spin[n_,x] :> spin[n,x] Cos[w[n]t] + spin[n,y] Sin[w[n]t] /; MemberQ[spins,n],                      (* Eq. 14 *)
	spin[n_,y] :> spin[n,y] Cos[w[n]t] - spin[n,x] Sin[w[n]t] /; MemberQ[spins,n],                      (* Eq. 15 *)
        spin[n_,plus]  :> spin[n,plus] Exp[-I w[n]t]              /; MemberQ[spins,n],
        spin[n_,minus] :> spin[n,minus] Exp[I w[n]t]              /; MemberQ[spins,n]}


(* Spin-spin couplings ---------------------------------------------------------------------------------------------- *)

Attributes[j] = {Orderless}
coupling[t_,couples_:{}][sigma_] := sigma /; couples=={}
coupling[t_,{{m1_,m2_}}][sigma_] := normalize[sigma] /. {
	spin[n1_,k_] spin[n2_,l_] :> spin[n1,k] spin[n2,l] /; 
		MemberQ[{x,y,plus,minus},k] && MemberQ[{x,y,plus,minus},l] && {m1,m2}=={n1,n2},
	spin[n1_,x] spin[n2_,z] :> spin[n1,x] spin[n2,z] Cos[Pi j[m1,m2]t] + 1/2 spin[n1,y] Sin[Pi j[m1,m2]t] /;
		{m1,m2}=={n1,n2} || {m1,m2}=={n2,n1},                                                       (* Eq. 19 *)
	spin[n1_,y] spin[n2_,z] :> spin[n1,y] spin[n2,z] Cos[Pi j[m1,m2]t] - 1/2 spin[n1,x] Sin[Pi j[m1,m2]t] /;
		{m1,m2}=={n1,n2} || {m1,m2}=={n2,n1},                                                       (* Eq. 20 *)
	spin[n1_,plus] spin[n2_,z]  :> spin[n1,plus] spin[n2,z] Cos[Pi j[m1,m2]t] - I/2 spin[n1,plus] Sin[Pi j[m1,m2]t] /;
		{m1,m2}=={n1,n2} || {m1,m2}=={n2,n1},
	spin[n1_,minus] spin[n2_,z] :> spin[n1,minus]spin[n2,z] Cos[Pi j[m1,m2]t] + I/2 spin[n1,minus]Sin[Pi j[m1,m2]t] /;
		{m1,m2}=={n1,n2} || {m1,m2}=={n2,n1},
	spin[n_,x] :> spin[n,x] Cos[Pi j[m1,m2]t] + 2 spin[n,y] spin[m2,z] Sin[Pi j[m1,m2]t] /; m1==n,      (* Eq. 17 *)
	spin[n_,x] :> spin[n,x] Cos[Pi j[m1,m2]t] + 2 spin[n,y] spin[m1,z] Sin[Pi j[m1,m2]t] /; m2==n,
	spin[n_,y] :> spin[n,y] Cos[Pi j[m1,m2]t] - 2 spin[n,x] spin[m2,z] Sin[Pi j[m1,m2]t] /; m1==n,      (* Eq. 18 *)
	spin[n_,y] :> spin[n,y] Cos[Pi j[m1,m2]t] - 2 spin[n,x] spin[m1,z] Sin[Pi j[m1,m2]t] /; m2==n,
	spin[n_,plus]  :> spin[n,plus]  Cos[Pi j[m1,m2]t] - 2I spin[n,plus]  spin[m2,z] Sin[Pi j[m1,m2]t] /; m1==n,
	spin[n_,plus]  :> spin[n,plus]  Cos[Pi j[m1,m2]t] - 2I spin[n,plus]  spin[m1,z] Sin[Pi j[m1,m2]t] /; m2==n,
	spin[n_,minus] :> spin[n,minus] Cos[Pi j[m1,m2]t] + 2I spin[n,minus] spin[m2,z] Sin[Pi j[m1,m2]t] /; m1==n,
	spin[n_,minus] :> spin[n,minus] Cos[Pi j[m1,m2]t] + 2I spin[n,minus] spin[m1,z] Sin[Pi j[m1,m2]t] /; m2==n }
coupling[t_,couples_][sigma_] := sigma // coupling[t,{First[couples]}] // coupling[t,Rest[couples]]


(* RF Pulses -------------------------------------------------------------------------------------------------------- *)

pulse[b_,p_,spins_:allspins][sigma_] := Block[{s},
	WriteString["stdout","pulse[",InputForm[b],",",InputForm[p],",",spins,"]"]; 
	time[s=raiselowerspins[sigma]; res=sigma // cartesian[s] // pulses[b,p,spins] // raiselower[s] // simplifysigma]] 
pulses[b_,p_List,spins_][sigma_List] := Block[{i},Table[pulses[b,p[[i]],spins][sigma[[i]]],{i,1,Length[p]}] ]
pulses[b_,p_List,spins_][sigma_]     := Block[{i},Table[pulses[b,p[[i]],spins][sigma],{i,1,Length[p]}]]
pulses[b_, p_,spins_] := pulses[b Pi/180,p,spins] /; NumberQ[b] && b!=0
pulses[b_, p_,spins_] := pulses[b,p Pi/180,spins] /; NumberQ[p] && p!=0
pulses[b_,-p_,spins_] := pulses[-b,p,spins]
pulses[b_, p_,spins_][sigma_]:= sigma /. {
	spin[n_,k_]:>  spin[n,k] /; p==k,
	spin[n_,z] :> spin[n,z] Cos[b] - spin[n,y] Sin[b] /; p==x && MemberQ[spins,n],                      (* Eq. 24 *)
	spin[n_,y] :> spin[n,y] Cos[b] + spin[n,z] Sin[b] /; p==x && MemberQ[spins,n],                      (* Eq. 25 *)
	spin[n_,z] :> spin[n,z] Cos[b] + spin[n,x] Sin[b] /; p==y && MemberQ[spins,n],                      (* Eq. 26 *)
	spin[n_,x] :> spin[n,x] Cos[b] - spin[n,z] Sin[b] /; p==y && MemberQ[spins,n],                      (* Eq. 27 *)
	spin[n_,x] :> spin[n,x] Cos[b] + spin[n,y] Sin[b] /; p==z && MemberQ[spins,n],                      (* Eq. 28 *)
	spin[n_,y] :> spin[n,y] Cos[b] - spin[n,x] Sin[b] /; p==z && MemberQ[spins,n],                      (* Eq. 29 *)
	spin[n_,z] :>  spin[n,z] Cos[b] + spin[n,x] Sin[b] Sin[p] - spin[n,y] Sin[b] Cos[p] /; 
		MemberQ[spins,n],                                                                           (* Eq. 35 *)
	spin[n_,x] :> -spin[n,z] Sin[b] Sin[p] + spin[n,x] (Cos[b] Sin[p]^2 + Cos[p]^2) + spin[n,y] Sin[b/2]^2 Sin[2p] /;
		MemberQ[spins,n],                                                                           (* Eq. 36 *)
	spin[n_,y] :>  spin[n,z] Sin[b] Cos[p] + spin[n,x] Sin[b/2]^2 Sin[2p] + spin[n,y] (Cos[b] Cos[p]^2 + Sin[p]^2) /;
		MemberQ[spins,n]                                                                            (* Eq. 37 *)
	} /. {  
	Cos[n_?Negative u_.] :> Cos[-n u], Sin[n_?Negative u_.] :> -Sin[-n u],
        Cos[n_?Negative u_ + v_] :>  Cos[-n u - v] /; Order[u,v] == 1 && NumberQ[n],
        Sin[n_?Negative u_ + v_] :> -Sin[-n u - v] /; Order[u,v] == 1 && NumberQ[n]}
  

(* Gradients -------------------------------------------------------------------------------------------------------- *)

gradient[G_][sigma_] := Block[{}, WriteString["stdout","gradient[",InputForm[G],"]"]; 
	time[s=raiselowerspins[sigma]; sigma // cartesian[s] // gradients[G,thespins[sigma]] // raiselower[s] //
             simplifysigma]]
gradients[G_,{k_}][sigma_] := sigma // pulses[g[k] G z,z,{k}]
gradients[G_,spins_][sigma_] := sigma // gradients[G,{First[spins]}] // gradients[G,Rest[spins]]

dephase[sigma_] := Block[{}, WriteString["stdout","dephase"]; time[sigma // dephases // simplifysigma]]
dephases[sigma_] := (sigma /. {
        Cos[a_. z] :> (Exp[I a z] + Exp[-I a z])/2,
	Sin[a_. z] :> (Exp[I a z] - Exp[-I a z])/(2I),
        Cos[a_. z + b_] :> (Exp[I a z] + Exp[-I a z])/2 Cos[b] - (Exp[I a z] - Exp[-I a z])/(2I) Sin[b],
        Sin[a_. z + b_] :> (Exp[I a z] + Exp[-I a z])/2 Sin[b] + (Exp[I a z] - Exp[-I a z])/(2I) Cos[b]} //
	simplifysigma) /. {
        Exp[u_] :> Exp[Expand[Simplify[u]]] /; !FreeQ[u,z]} /. { Exp[a_. z + b_.] -> 0 }


(* Receiver --------------------------------------------------------------------------------------------------------- *)

receiver[p_,spins_:allspins][sigma_] := Block[{s}, WriteString["stdout","receiver[",InputForm[p],",",spins,"]"]; 
	time[s=raiselowerspins[sigma]; sigma // cartesian[s] // receivers[p,spins] // raiselower[s]]]
receivers[ p_List,spins_][sigma_List] := Block[{i},Sum[receivers[p[[i]],spins][sigma[[i]]],{i,1,Length[p]}]/Length[p]]
receivers[ p_List,spins_][sigma_]     := Block[{i},Sum[receivers[p[[i]],spins][sigma],{i,1,Length[p]}]/Length[p]]
receivers[ p_,spins_][sigma_List]     := Block[{i},Sum[receivers[p,spins][sigma[[i]]],{i,1,Length[sigma]}]/Length[sigma]]
receivers[ x,spins_][sigma_] := sigma
receivers[-x,spins_][sigma_] := sigma // pulses[180,z,spins]
receivers[ y,spins_][sigma_] := sigma // pulses[-90,z,spins]
receivers[-y,spins_][sigma_] := sigma // pulses[ 90,z,spins]
receivers[p_,spins_][sigma_] := sigma // pulses[ -p,z,spins]
  

(* Observables ------------------------------------------------------------------------------------------------------ *)

transverse[k_] := MemberQ[{x,y,plus,minus},k]
tfree[u_] := !MatchQ[u,a_. spin[_,_?transverse]]
observable[sigma_] := observable[allspins][sigma] /; !FreeQ[sigma,spin[_,_]] || sigma==0
observable[spins_][sigma_] := Block[{},WriteString["stdout","observable[",spins,"]"]; 
	time[normalize[observables[spins][sigma]]]] 
observables[spins_][0] := 0
observables[spins_][sigma_List] := Block[{i},Table[Apply[Plus,observables[spins][sigma[[i]]]],{i,1,Length[sigma]}]]
observables[spins_][sigma_] := Block[{detected,qloc},detected[n_] := MemberQ[spins,n];
	Apply[Plus,Cases[Apply[List,1+normalize[qloc sigma]],a_?tfree spin[n_?detected,k_?transverse]] /. qloc->1 // 
                   trigsimplify]]


(* Normalization and simplification --------------------------------------------------------------------------------- *)

spinlist[sigma_] := sigma // If[Head[#]===Plus,Apply[List,#], {#}]& // 
	Map[If[Head[#]===Times,Apply[List,#],{#}]&,#]&  //
	Map[Cases[#,spin[_,_]]&,#]& // Map[(# /. List->Times)&,#]& // Union
normalize[0] := 0
normalize[sigma_List] := Map[normalize,sigma] 
normalize[sigma_] := Block[{e,p}, e=Expand[sigma]; p=spinlist[e]; WriteString["stdout","."];
        (Map[Coefficient[e,#]&,p] /. spin[_,_] -> 0).p]

trigsimplify[u_] := Expand[u] //. { 
        Cos[n_?Negative a_.] :> Cos[-n a], Sin[n_?Negative a_.] :> -Sin[-n a],
        Cos[n_?Negative a_ + b_] :>  Cos[-n a - b] /; Order[a, b] == 1 && NumberQ[n],
        Sin[n_?Negative a_ + b_] :> -Sin[-n a - b] /; Order[a, b] == 1 && NumberQ[n],     
	c_. Sin[a_]^2 + c_. Cos[a_]^2 :> c, 
	c_. Cos[a_]^2 + d_. Sin[a_]^2 :> c Cos[2a] /; c+d==0,
	Sin[a_] Cos[a_] :> 1/2 Sin[2 a],
	c_. Sin[a_] Cos[b_] + c_. Cos[a_] Sin[b_] :>  c Sin[a+b],
	c_. Sin[a_] Cos[b_] + d_. Cos[a_] Sin[b_] :>  c Sin[a-b] /; c+d == 0,
	c_. Cos[a_] Cos[b_] + d_. Sin[a_] Sin[b_] :>  c Cos[a+b] /; c+d == 0,
	c_. Cos[a_] Cos[b_] + c_. Sin[a_] Sin[b_] :>  c Cos[a-b]}

simplifysigma[sigma_List] := Map[simplifysigma,sigma]
simplifysigma[sigma_] := Apply[Plus,trigsimplify[Apply[List,1+normalize[sigma]]]]-1 // normalize


(* Output formats --------------------------------------------------------------------------------------------------- *)

nucleus[_] = "I"
spinfree[u_] := FreeQ[u,spin[_,_]]
spinop[u_] := !FreeQ[u,spin[_,_]]
times[u_] := FreeQ[u,j[_,_]] && FreeQ[u,w[_]] && FreeQ[u,_Integer] && FreeQ[u,_Real] && FreeQ[u,Pi]
sortspins[sigma_] := (Print[TableForm[Complement[Apply[List,1 + Apply[Plus,sigma /. w[n_]->Aw[n]] //. { 
	Sin[a_?times b_.] :> AA[a,b,Sin], Cos[a_?times b_.] :> AA[a,b,Cos],
	a_Times + b_Times :> Select[a,spinfree] (Select[a,spinop] + Select[b,spinop]) /; 
		Select[a,spinfree]==Select[b,spinfree]}],{1}]]]; 
	sigma;)
(*show[sigma_] := show[""][sigma]*)
show[label_][sigma_] := Block[{i,s}, s=StringInsert["","=",Table[1,{i,1,36-StringLength[label]/2}]];
        Print[""]; Print[s," ",label," ",s]; Print[sigma]; Print[""]; sigma]
store[var_][sigma_] := (var=sigma)

Format[AA[a_,b_Plus,c_]] := SequenceForm[c,"[(",b,") ",a,"]"]
Format[AA[a_,1,c_]]      := SequenceForm[c,"[",a,"]"]
Format[AA[a_,b_,c_]]     := SequenceForm[c,"[",b," ",a,"]"]
Format[Aw[n_]]           := SequenceForm["w",n]
Format[w[n_]]            := SequenceForm["w",n]
Format[g[n_]]            := SequenceForm["g",n]
Format[j[n1_,n2_]]       := SequenceForm["J",n1,n2]
Format[spin[n_,plus]]    := SequenceForm[nucleus[n],n,"+"]
Format[spin[n_,minus]]   := SequenceForm[nucleus[n],n,"-"]
Format[spin[n_,k_]]      := SequenceForm[nucleus[n],n,k]

terms[sigma_List] := Sum[terms[sigma[[i]]],{i,1,Length[sigma]}]
terms[sigma_]     := Max[0,Length[1+sigma]-1]
time[expr_] := Block[{r,n,c},
	r = Timing[expr]; c = r[[1]] /. Second->1;  n = terms[r[[2]]];
        Print[" "]; Print["  (",n," term",If[n==1,"","s"],", ",N[c,3]," s CPU time)"]; r[[2]]]
SetAttributes[time,HoldAll]


(* POMA software license -------------------------------------------------------------------------------------------- *)

license:=Print["\<
POMA 2.0 Software License


The POMA 2.0 Software License is a legal agreement, governed by the
laws of Switzerland, between an end user (the \"Licensee\"), either an
individual or an entity, and Dr. Peter Guntert (the \"Licensor\"). The
program package POMA 2.0 (copyright (c) 2002 by Peter Guntert) for
simulation of NMR pulse sequences, comprising all computer programs,
source code, license keys, documentation, example data and other files
delivered to the Licensee, as well as any copies, modifications or
derivative works made by the Licensee, are hereinafter referred to
collectively as the \"Software\". A derivative work is any software that
contains one or several parts of the Software in original or modified
form. If the Licensor provides the Licensee with updates of the
Software, these will become part of the Software and will be
controlled by this license. 

 1. The Licensor grants to the Licensee a non-exclusive,
    non-transferable, permanent license to install and use the
    Software on computer systems located at the site of the Licensee's
    organization. However, a violation of any of the clauses of this
    license agreement by the Licensee shall terminate automatically
    and with immediate effect the Licensee's right to install, use or
    store the Software in any form. Use of the Software is restricted
    to the Licensee and to direct collaborators who are members of the
    organization of the Licensee at the site of the Licensee and who
    accept the terms of this license. The Licensee shall neither use
    the Software to produce other software that duplicates
    functionality of the Software nor translate source code of the
    Software into another programming language.

 2. The Licensor retains at all times ownership of the Software
    delivered to the Licensee. Any modifications or derivative works
    based on the Software are considered part of the Software, and
    ownership thereof is retained by the Licensor. All parts of the
    Software must carry the copyright notice, will be controlled by
    this license, and will be promptly destroyed by the Licensee upon
    termination of this license.

 3. The Licensee shall not disclose in any form the Software or any
    modifications or derivative works based on the Software to third
    parties without prior written authorization from the Licensor.

 4. The Licensee agrees that the Software has been developed in
    connection with academic research projects and is provided \"as
    is\". The Licensor disclaims all warranties with regard to the
    Software or any of its results, including any implied warranties
    of merchantability or fitness for a particular purpose. In no
    event shall the Licensor be liable for any damages, however
    caused, including, without limitation, any damages arising out of
    the use of the Software, loss of use of the Software, or damage of
    any sort to the Licensee.

 5. The Licensee agrees that any reports or publications of results
    obtained with the Software will acknowledge its use by the
    appropriate literature citation: Guntert, P., Schaefer, N.,
    Otting, G. and Wuthrich, K. (1993). POMA, a complete Mathematica
    implementation of the NMR product operator formalism. J. Magn.
    Reson. A 101, 103-105.
\>"]

End[ ] 
EndPackage[ ]
